"use client";

import React, { useCallback, useState } from "react";
import {
  Button,
  Card,
  Col,
  DatePicker,
  Form,
  Input,
  Modal,
  Row,
  Space,
  Switch,
  Tooltip,
  Tag,
  Breadcrumb,
  Pagination,
  Dropdown,
  Select,
} from "antd";
import {
  PlusOutlined,
  EditOutlined,
  DeleteOutlined,
  EyeOutlined,
  HomeOutlined,
  TableOutlined,
  AppstoreOutlined,
  SearchOutlined,
  DownloadOutlined,
  CheckCircleTwoTone,
  WarningOutlined,
  StopOutlined,
  CheckCircleOutlined,
  CloseCircleOutlined,
} from "@ant-design/icons";
import type { ColumnsType } from "antd/es/table";
import dayjs from "dayjs";
import DataTable from "../components/common/DataTable";
import WidgetCard from "../components/common/WidgetCard";
import ColumnVisibilityToggle from "../components/common/ManageColumn";
import { useIsMobile } from "@/data/useIsMobile";
import { SampleTest } from "../types/SupplierTest";
import { toast } from "sonner";
import { motion } from "framer-motion";

const { RangePicker } = DatePicker;
const inspectionResultOptions = ["Pass", "Fail", "Pending"] as const;

export default function SampleTestPage() {
  const [form] = Form.useForm();
  const [tests, setTests] = useState<SampleTest[]>([]);
  const [isModalOpen, setIsModalOpen] = useState(false);
  const [editingTest, setEditingTest] = useState<SampleTest | null>(null);
  const [viewingTest, setViewingTest] = useState<SampleTest | null>(null);
  const [searchTerm, setSearchTerm] = useState("");
  const [isGridView, setIsGridView] = useState(false);
  const [selectedRows, setSelectedRows] = useState<string[]>([]);
  const [advancedFilters, setAdvancedFilters] = useState({
    inspectionResult: undefined,
  });
  const [dateRange, setDateRange] = useState<[dayjs.Dayjs, dayjs.Dayjs] | null>(null);
  const [currentPage, setCurrentPage] = useState(1);
  const [pageSize, setPageSize] = useState(10);
  const isMobile = useIsMobile();

  const openAddModal = () => {
    form.resetFields();
    setEditingTest(null);
    setIsModalOpen(true);
  };

  const openEditModal = (test: SampleTest) => {
    form.setFieldsValue({
      ...test,
      poDate: test.poDate ? dayjs(test.poDate) : undefined,
      deliveryDate: test.deliveryDate ? dayjs(test.deliveryDate) : undefined,
      receivingDate: test.receivingDate ? dayjs(test.receivingDate) : undefined,
    });
    setEditingTest(test);
    setIsModalOpen(true);
  };

  const handleDelete = (id: string) => {
    Modal.confirm({
      title: "Are you sure you want to delete this sample test?",
      onOk: () => {
        setTests((prev) => prev.filter((t) => t.id !== id));
        toast.success("Sample test deleted successfully");
      },
      onCancel: () => { },
    });
  };

  const handleBulkDelete = () => {
    Modal.confirm({
      title: `Are you sure you want to delete ${selectedRows.length} sample tests?`,
      onOk: () => {
        setTests((prev) => prev.filter((t) => !selectedRows.includes(t.id.toString())));
        setSelectedRows([]);
        toast.success("Sample tests deleted successfully");
      },
    });
  };

  const handleFormSubmit = () => {
    form.validateFields().then((values) => {
      const data: SampleTest = {
        ...values,
        id: editingTest ? editingTest.id : Date.now().toString(),
        poDate: dayjs(values.poDate).format("YYYY-MM-DD"),
        deliveryDate: dayjs(values.deliveryDate).format("YYYY-MM-DD"),
        receivingDate: dayjs(values.receivingDate).format("YYYY-MM-DD"),
        created_at: editingTest ? editingTest.created_at : dayjs().toISOString(),
      };
      if (editingTest) {
        setTests((prev) => prev.map((t) => (t.id === editingTest.id ? data : t)));
        toast.success("Sample test updated successfully");
      } else {
        setTests((prev) => [...prev, data]);
        toast.success("Sample test added successfully");
      }
      setIsModalOpen(false);
      form.resetFields();
    });
  };

  const renderTag = (value: string) => {
    const colorMap: Record<string, string> = {
      Pass: "green",
      Fail: "red",
      Pending: "orange",
    };

    const iconMap: Record<string, React.ReactNode> = {
      Pass: <CheckCircleOutlined />,
      Fail: <CloseCircleOutlined />,
      Pending: <WarningOutlined />,
    };

    return (
      <Tag icon={iconMap[value]} color={colorMap[value] || "default"}>
        {value}
      </Tag>
    );
  };

  const filteredTests = tests.filter((test) => {
    const matchesSearch =
      test.supplier.toLowerCase().includes(searchTerm.toLowerCase()) ||
      test.itemSupplied.toLowerCase().includes(searchTerm.toLowerCase());
    const matchesDate =
      !dateRange ||
      (test.created_at &&
        (!dateRange[0] ||
          dayjs(test.created_at).isSame(dateRange[0]) ||
          dayjs(test.created_at).isAfter(dateRange[0])) &&
        (!dateRange[1] ||
          dayjs(test.created_at).isSame(dateRange[1]) ||
          dayjs(test.created_at).isBefore(dateRange[1])));
    const matchesAdvanced =
      !advancedFilters.inspectionResult || test.inspectionResult === advancedFilters.inspectionResult;
    return matchesSearch && matchesDate && matchesAdvanced;
  });

  const paginatedTests = filteredTests.slice(
    (currentPage - 1) * pageSize,
    currentPage * pageSize
  );

  const columns: ColumnsType<SampleTest> = [
    {
      title: (
        <input
          type="checkbox"
          onChange={(e) => {
            const checked = e.target.checked;
            setSelectedRows(checked ? filteredTests.map((t) => t.id.toString()) : []);
          }}
          className="w-4 h-4"
        />
      ),
      key: "selection",
      render: (_, record) => (
        <input
          type="checkbox"
          checked={selectedRows.includes(record.id.toString())}
          onChange={(e) => {
            setSelectedRows((prev) =>
              e.target.checked
                ? [...prev, record.id.toString()]
                : prev.filter((id) => id !== record.id.toString())
            );
          }}
          className="w-4 h-4"
        />
      ),
      width: 44,
    },
    {
      title: "Supplier",
      dataIndex: "supplier",
      key: "supplier",
      sorter: (a, b) => a.supplier.localeCompare(b.supplier),
      render: (text) => <span className="text-green-600 font-medium">{text}</span>,
      filterSearch: true,
      onFilter: (value, record) =>
        record.supplier.toLowerCase().includes((value as string).toLowerCase()),
      filters: Array.from(new Set(tests.map((t) => t.supplier))).map((value) => ({
        text: value,
        value: value,
      })),
    },
    {
      title: "Item Supplied",
      dataIndex: "itemSupplied",
      key: "itemSupplied",
      sorter: (a, b) => a.itemSupplied.localeCompare(b.itemSupplied),
      filterSearch: true,
      onFilter: (value, record) =>
        record.itemSupplied.toLowerCase().includes((value as string).toLowerCase()),
      filters: Array.from(new Set(tests.map((t) => t.itemSupplied))).map((value) => ({
        text: value,
        value: value,
      })),
    },
    {
      title: "PO Date",
      dataIndex: "poDate",
      key: "poDate",
      sorter: (a, b) => new Date(a.poDate ?? 0).getTime() - new Date(b.poDate ?? 0).getTime(),
      render: (text) => dayjs(text).format("YYYY-MM-DD"),
    },
    {
      title: "Delivery Date",
      dataIndex: "deliveryDate",
      key: "deliveryDate",
      sorter: (a, b) => new Date(a.deliveryDate ?? 0).getTime() - new Date(b.deliveryDate ?? 0).getTime(),
      render: (text) => dayjs(text).format("YYYY-MM-DD"),
    },
    {
      title: "Receiving Date",
      dataIndex: "receivingDate",
      key: "receivingDate",
      sorter: (a, b) => new Date(a.receivingDate ?? 0).getTime() - new Date(b.receivingDate ?? 0).getTime(),
      render: (text) => dayjs(text).format("YYYY-MM-DD"),
    },
    {
      title: "Inspection Result",
      dataIndex: "inspectionResult",
      key: "inspectionResult",
      sorter: (a, b) => a.inspectionResult.localeCompare(b.inspectionResult),
      filterSearch: true,
      onFilter: (value, record) => record.inspectionResult === value,
      filters: inspectionResultOptions.map((res) => ({ text: res, value: res })),
      render: renderTag,
    },
    {
      title: "Rejection Reason",
      dataIndex: "rejectionReason",
      key: "rejectionReason",
      sorter: (a, b) => (a.rejectionReason || "").localeCompare(b.rejectionReason || ""),
      render: (text) => (
        <Tooltip title={text}>
          <span className="block truncate max-w-[100px] sm:max-w-[150px] md:max-w-[180px]">{text || "-"}</span>
        </Tooltip>
      ),
    },
    {
      title: "Actions",
      key: "actions",
      fixed: "right",
      render: (_, record) => (
        <Space size="small">
          <Tooltip title="View">
            <Button
              icon={<EyeOutlined />}
              onClick={() => setViewingTest(record)}
              size="small"
              className="text-blue-600"
            />
          </Tooltip>
          <Tooltip title="Edit">
            <Button
              icon={<EditOutlined />}
              onClick={() => openEditModal(record)}
              size="small"
              type="primary"
              className="bg-blue-600 hover:bg-blue-700"
            />
          </Tooltip>
          <Tooltip title="Delete">
            <Button
              icon={<DeleteOutlined />}
              onClick={() => handleDelete(record.id)}
              danger
              size="small"
              className="text-red-600"
            />
          </Tooltip>
        </Space>
      ),
    },
  ];

  const allColumns = [
    { key: "selection", label: "Select" },
    { key: "supplier", label: "Supplier" },
    { key: "itemSupplied", label: "Item Supplied" },
    { key: "poDate", label: "PO Date" },
    { key: "deliveryDate", label: "Delivery Date" },
    { key: "receivingDate", label: "Receiving Date" },
    { key: "inspectionResult", label: "Inspection Result" },
    { key: "rejectionReason", label: "Rejection Reason" },
    { key: "actions", label: "Actions" },
  ];

  const columnKeys = allColumns.map((c) => c.key);
  const labelsMap = Object.fromEntries(allColumns.map((c) => [c.key, c.label]));
  const [visibleColumnKeys, setVisibleColumnKeys] = useState<string[]>(columnKeys);

  const handleVisibleColumnChange = useCallback((cols: string[]) => {
    setVisibleColumnKeys(cols);
  }, []);

  const filteredColumns = columns.filter(
    (col) => typeof col.key === "string" && visibleColumnKeys.includes(col.key)
  );

  const SampleCard = ({ test }: { test: SampleTest }) => (
    <div className="border rounded-xl p-4 shadow-sm hover:shadow-md bg-white dark:bg-gray-900 transition-all w-full">
      <div className="flex flex-col sm:flex-row justify-between items-start sm:items-center mb-2 gap-2">
        <h3 className="text-base sm:text-lg font-semibold text-gray-800 dark:text-white truncate">{test.supplier}</h3>
        <Tag color="blue">{test.inspectionResult}</Tag>
      </div>
      <p className="text-sm text-gray-600 dark:text-gray-300 mb-1 truncate"><strong>Item:</strong> {test.itemSupplied}</p>
      <p className="text-sm text-gray-600 dark:text-gray-300 mb-1"><strong>PO Date:</strong> {dayjs(test.poDate).format("YYYY-MM-DD")}</p>
      <p className="text-sm text-gray-600 dark:text-gray-300 mb-1"><strong>Result:</strong> {renderTag(test.inspectionResult)}</p>
      <p className="text-sm text-gray-500 dark:text-gray-400 text-xs"><strong>Received:</strong> {dayjs(test.receivingDate).format("YYYY-MM-DD")}</p>
      <div className="mt-2 flex gap-2">
        <Tooltip title="View">
          <Button icon={<EyeOutlined />} onClick={() => setViewingTest(test)} size="small" className="text-blue-600" />
        </Tooltip>
        <Tooltip title="Edit">
          <Button icon={<EditOutlined />} onClick={() => openEditModal(test)} size="small" type="primary" className="bg-blue-600 hover:bg-blue-700" />
        </Tooltip>
        <Tooltip title="Delete">
          <Button icon={<DeleteOutlined />} onClick={() => handleDelete(test.id)} size="small" danger className="text-red-600" />
        </Tooltip>
      </div>
    </div>
  );

  const total = tests.length;
  const passed = tests.filter((t) => t.inspectionResult === "Pass").length;
  const failed = tests.filter((t) => t.inspectionResult === "Fail").length;
  const pending = tests.filter((t) => t.inspectionResult === "Pending").length;

  // Sample trend data for widgets
  const trendData = [
    { value: 10 },
    { value: 15 },
    { value: 12 },
    { value: 18 },
    { value: 20 },
  ];

  return (
    <>
      <div className="px-2 sm:px-4 md:px-6 lg:px-8 py-4">
        <h2 className="text-xl sm:text-2xl md:text-3xl font-bold text-gray-800 dark:text-white mb-2">
          Sample Tests <span className="text-gray-500 text-base sm:text-lg font-medium">({tests.length})</span>
        </h2>
        <p className="text-gray-600 dark:text-gray-300 text-sm sm:text-base mb-4">
          Track sample inspection records for deliveries.
        </p>
        <Breadcrumb
          className="mb-4 sm:mb-6 mt-2 text-sm text-gray-500 dark:text-gray-400"
          separator=">"
          items={[
            {
              title: (
                <Tooltip title="Dashboard">
                  <HomeOutlined />
                </Tooltip>
              ),
              href: "/dashboard",
            },
            {
              title: "Master Data",
            },
            {
              title: "Sample Tests",
            },
          ]}
        />

        <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-4 mt-3 mb-6">
          <WidgetCard
            title="Total Tests"
            value={total}
            icon={<AppstoreOutlined />}
            color="green"
            percentage={12}
            description="All sample tests conducted"
            trendData={trendData}
          />
          <WidgetCard
            title="Passed Tests"
            value={passed}
            icon={<CheckCircleTwoTone twoToneColor="#52c41a" />}
            color="blue"
            percentage={8}
            description="Tests with pass result"
            trendData={trendData}
          />
          <WidgetCard
            title="Failed Tests"
            value={failed}
            icon={<StopOutlined />}
            color="red"
            percentage={-3}
            description="Tests with fail result"
            trendData={trendData}
          />
          <WidgetCard
            title="Pending Tests"
            value={pending}
            icon={<WarningOutlined />}
            color="orange"
            percentage={5}
            description="Tests awaiting results"
            trendData={trendData}
          />
        </div>

        <div className="mt-4">
          <Card title="Sample Test List" className="p-0 mb-6 shadow-sm">
            <div className="p-4 flex flex-col gap-4">
              <div className="flex flex-col sm:flex-row justify-between items-stretch sm:items-center gap-4">
                <Input
                  prefix={<SearchOutlined />}
                  allowClear
                  placeholder="Search Supplier or Item"
                  value={searchTerm}
                  onChange={(e) => setSearchTerm(e.target.value)}
                  className="w-full sm:w-64 border-gray-300 dark:border-gray-600 rounded-md shadow-sm"
                />
                <Dropdown
                  className="w-full sm:w-auto border-gray-300 dark:border-gray-600"
                  menu={{
                    items: [
                      { label: "Export CSV", key: "csv" },
                      { label: "Export PDF", key: "pdf" },
                    ],
                  }}
                >
                  <Button icon={<DownloadOutlined />}>Export</Button>
                </Dropdown>
                <Button
                  type="primary"
                  icon={<PlusOutlined />}
                  onClick={openAddModal}
                  className="bg-blue-600 hover:bg-blue-700 w-full sm:w-auto"
                >
                  Add Sample Test
                </Button>
              </div>
              <div className="flex flex-col gap-4 lg:flex-row lg:justify-between lg:items-center">
                <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-4 w-full lg:w-auto">
                  <Select
                    placeholder="Inspection Result"
                    className="w-full max-w-[160px]"
                    onChange={(value) =>
                      setAdvancedFilters((prev) => ({ ...prev, inspectionResult: value }))
                    }
                    allowClear
                    options={inspectionResultOptions.map((v) => ({ label: v, value: v }))}
                  />
                </div>
                <div className="flex flex-col sm:flex-row flex-wrap gap-4 justify-start lg:justify-end w-full lg:w-auto">
                  <ColumnVisibilityToggle
                    columns={columnKeys}
                    labels={labelsMap}
                    storageKey="sample_test_table_visibility"
                    onChange={handleVisibleColumnChange}
                  />
                  <RangePicker
                    value={dateRange}
                    onChange={(dates) =>
                      setDateRange(dates as [dayjs.Dayjs, dayjs.Dayjs] | null)
                    }
                    className="w-full sm:w-auto border-gray-300 dark:border-gray-600"
                  />
                  <Button
                    onClick={handleBulkDelete}
                    disabled={selectedRows.length === 0}
                    danger
                    className="w-full sm:w-auto"
                  >
                    Bulk Delete
                  </Button>
                  <div className="hidden sm:block">
                    <Switch
                      checkedChildren={<AppstoreOutlined />}
                      unCheckedChildren={<TableOutlined />}
                      checked={isGridView}
                      onChange={setIsGridView}
                      className="w-full sm:w-auto"
                    />
                  </div>
                </div>
              </div>

              {advancedFilters.inspectionResult && (
                <div className="flex flex-wrap gap-2 mt-2">
                  <Tag
                    closable
                    onClose={() =>
                      setAdvancedFilters((prev) => ({
                        ...prev,
                        inspectionResult: undefined,
                      }))
                    }
                  >
                    Inspection Result: {advancedFilters.inspectionResult}
                  </Tag>
                </div>
              )}
            </div>

            {isGridView || isMobile ? (
              <div className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-4 p-4">
                {paginatedTests.map((test) => (
                  <SampleCard key={test.id} test={test} />
                ))}
              </div>
            ) : (
              <div className="overflow-auto">
                <DataTable<SampleTest>
                  columns={filteredColumns}
                  dataSource={paginatedTests}
                  rowKey="id"
                  scrollX="max-content"
                  className="mt-4"
                />
              </div>
            )}
            <div className="flex flex-col sm:flex-row justify-between items-center px-4 py-2 text-sm text-gray-600 dark:text-gray-300">
              <div className="mb-2 sm:mb-0">
                Showing <strong>{(currentPage - 1) * pageSize + 1}</strong> to{" "}
                <strong>{Math.min(currentPage * pageSize, filteredTests.length)}</strong> of{" "}
                <strong>{filteredTests.length}</strong> tests
              </div>
              <Pagination
                current={currentPage}
                pageSize={pageSize}
                total={filteredTests.length}
                onChange={(page, size) => {
                  setCurrentPage(page);
                  setPageSize(size);
                }}
                showSizeChanger
                pageSizeOptions={["10", "30", "50", "100"]}
                size="small"
              />
            </div>
          </Card>
        </div>

        <motion.div
          initial={{ opacity: 0, scale: 0.95 }}
          animate={{ opacity: 1, scale: 1 }}
          exit={{ opacity: 0, scale: 0.95 }}
        >
          <Modal
            title={editingTest ? "Edit Sample Test" : "Add Sample Test"}
            open={isModalOpen}
            onOk={handleFormSubmit}
            onCancel={() => {
              setIsModalOpen(false);
              form.resetFields();
            }}
            destroyOnHidden
            className="w-full max-w-[90vw] sm:max-w-[600px] md:max-w-[800px] lg:max-w-[1000px] top-4"
          >
            <Form form={form} layout="vertical" className="px-2 sm:px-4">
              <Row gutter={[16, 16]}>
                <Col xs={24} sm={12}>
                  <Form.Item name="supplier" label="Supplier" rules={[{ required: true, message: "Supplier is required" }]}>
                    <Input />
                  </Form.Item>
                </Col>
                <Col xs={24} sm={12}>
                  <Form.Item name="itemSupplied" label="Item Supplied" rules={[{ required: true, message: "Item Supplied is required" }]}>
                    <Input />
                  </Form.Item>
                </Col>
                <Col xs={24} sm={12}>
                  <Form.Item name="poDate" label="PO Date" rules={[{ required: true, message: "PO Date is required" }]}>
                    <DatePicker className="w-full" />
                  </Form.Item>
                </Col>
                <Col xs={24} sm={12}>
                  <Form.Item name="deliveryDate" label="Delivery Date" rules={[{ required: true, message: "Delivery Date is required" }]}>
                    <DatePicker className="w-full" />
                  </Form.Item>
                </Col>
                <Col xs={24} sm={12}>
                  <Form.Item name="receivingDate" label="Date of Receiving" rules={[{ required: true, message: "Receiving Date is required" }]}>
                    <DatePicker className="w-full" />
                  </Form.Item>
                </Col>
                <Col xs={24} sm={12}>
                  <Form.Item name="inspectionResult" label="Inspection Result" rules={[{ required: true, message: "Inspection Result is required" }]}>
                    <Select options={inspectionResultOptions.map((v) => ({ label: v, value: v }))} />
                  </Form.Item>
                </Col>
                <Col xs={24}>
                  <Form.Item name="rejectionReason" label="Reason of Rejection (if any)">
                    <Input.TextArea rows={2} />
                  </Form.Item>
                </Col>
              </Row>
            </Form>
          </Modal>
        </motion.div>

        <Modal
          title="Sample Test Details"
          open={!!viewingTest}
          onCancel={() => setViewingTest(null)}
          footer={null}
          className="w-full max-w-[90vw] sm:max-w-[500px] top-4"
        >
          {viewingTest && (
            <div className="space-y-2 text-sm sm:text-base">
              <p><strong>Supplier:</strong> <span className="text-green-600">{viewingTest.supplier}</span></p>
              <p><strong>Item Supplied:</strong> {viewingTest.itemSupplied}</p>
              <p><strong>PO Date:</strong> {dayjs(viewingTest.poDate).format("YYYY-MM-DD")}</p>
              <p><strong>Delivery Date:</strong> {dayjs(viewingTest.deliveryDate).format("YYYY-MM-DD")}</p>
              <p><strong>Date of Receiving:</strong> {dayjs(viewingTest.receivingDate).format("YYYY-MM-DD")}</p>
              <p><strong>Inspection Result:</strong> {viewingTest.inspectionResult}</p>
              <p><strong>Reason of Rejection:</strong> {viewingTest.rejectionReason || "-"}</p>
              <p><strong>Created:</strong> {dayjs(viewingTest.created_at).format("YYYY-MM-DD")}</p>
            </div>
          )}
        </Modal>
      </div>
    </>
  );
}