"use client";

import React, { useState, useEffect } from "react";
import { Button, Form, Input, Modal, Select, Space, Row, Col, Tooltip, Breadcrumb, Card, DatePicker, Switch, Table, Tag } from "antd";
import { PlusOutlined, EditOutlined, DeleteOutlined, EyeOutlined, HomeOutlined, IdcardOutlined, TableOutlined, AppstoreOutlined } from "@ant-design/icons";
import type { ColumnsType } from "antd/es/table";
import { toast } from "sonner";
import { motion } from "framer-motion";
import { useSearchParams, useRouter } from "next/navigation";
import dayjs from "dayjs";

const { TextArea } = Input;
const certificationTypeOptions = [
  "ISO 9001",
  "ISO 13485",
  "ISO 14001",
  "ISO 45001",
  "CE Marking",
  "FDA Registration",
  "Other",
] as const;
const statusOptions = ["Valid", "Expired", "Pending"] as const;

interface SupplierCertification {
  id: string;
  supplierId: string;
  supplierName: string;
  certificationType: string;
  certificationNumber: string;
  issuingBody: string;
  issueDate: string;
  expiryDate: string;
  scope: string;
  status: typeof statusOptions[number];
  created_at: string;
}

export default function SupplierCertificationPage() {
  const [form] = Form.useForm();
  const [certifications, setCertifications] = useState<SupplierCertification[]>([]);
  const [isModalOpen, setIsModalOpen] = useState(false);
  const [editingCertification, setEditingCertification] = useState<SupplierCertification | null>(null);
  const [viewingCertification, setViewingCertification] = useState<SupplierCertification | null>(null);
  const [otherCertification, setOtherCertification] = useState("");
  const [isTableView, setIsTableView] = useState(false);
  const searchParams = useSearchParams();
  const router = useRouter();
  const supplierId = searchParams.get("id");
  const supplierName = searchParams.get("name") || "Unknown Supplier";

  useEffect(() => {
    if (!supplierId) router.push("/suppliers");
  }, [supplierId, router]);

  const openAddModal = () => {
    form.resetFields();
    setOtherCertification("");
    setEditingCertification(null);
    setIsModalOpen(true);
  };

  const openEditModal = (certification: SupplierCertification) => {
    form.setFieldsValue({
      ...certification,
      issueDate: certification.issueDate ? dayjs(certification.issueDate) : undefined,
      expiryDate: certification.expiryDate ? dayjs(certification.expiryDate) : undefined,
    });
    setOtherCertification(certification.certificationType === "Other" ? certification.certificationType : "");
    setEditingCertification(certification);
    setIsModalOpen(true);
  };

  const handleDelete = (id: string) => {
    Modal.confirm({
      title: "Are you sure you want to delete this certification?",
      onOk: () => {
        setCertifications((prev) => prev.filter((c) => c.id !== id));
        toast.success("Certification deleted successfully");
      },
      onCancel: () => {},
    });
  };

  const handleFormSubmit = () => {
    form.validateFields().then((values: Omit<SupplierCertification, "id" | "created_at">) => {
      const finalCertificationType = values.certificationType === "Other" && otherCertification.trim()
        ? otherCertification.trim()
        : values.certificationType;
      if (values.certificationType === "Other" && !otherCertification.trim()) {
        toast.error("Please specify the Other Certification Type");
        return;
      }
      const newCertification: SupplierCertification = {
        ...values,
        id: editingCertification?.id ?? Date.now().toString(),
        supplierId: supplierId!,
        supplierName,
        certificationType: finalCertificationType,
        issueDate: dayjs(values.issueDate).format("YYYY-MM-DD"),
        expiryDate: dayjs(values.expiryDate).format("YYYY-MM-DD"),
        created_at: editingCertification?.created_at ?? dayjs().toISOString(),
      };
      if (editingCertification) {
        setCertifications((prev) => prev.map((c) => (c.id === editingCertification.id ? newCertification : c)));
        toast.success("Certification updated successfully");
      } else {
        setCertifications((prev) => [...prev, newCertification]);
        toast.success("Certification added successfully");
      }
      setIsModalOpen(false);
      form.resetFields();
      setOtherCertification("");
    });
  };

  const columns: ColumnsType<SupplierCertification> = [
    {
      title: "Certification Type",
      dataIndex: "certificationType",
      filters: certificationTypeOptions.map((t) => ({ text: t, value: t })),
      onFilter: (value, record) => record.certificationType.includes(value as string),
    },
    {
      title: "Certification Number",
      dataIndex: "certificationNumber",
      sorter: (a, b) => a.certificationNumber.localeCompare(b.certificationNumber),
    },
    {
      title: "Issuing Body",
      dataIndex: "issuingBody",
      sorter: (a, b) => a.issuingBody.localeCompare(b.issuingBody),
    },
    {
      title: "Issue Date",
      dataIndex: "issueDate",
      sorter: (a, b) => dayjs(a.issueDate).unix() - dayjs(b.issueDate).unix(),
      render: (text) => dayjs(text).format("YYYY-MM-DD"),
    },
    {
      title: "Expiry Date",
      dataIndex: "expiryDate",
      sorter: (a, b) => dayjs(a.expiryDate).unix() - dayjs(b.expiryDate).unix(),
      render: (text) => dayjs(text).format("YYYY-MM-DD"),
    },
    {
      title: "Status",
      dataIndex: "status",
      filters: statusOptions.map((s) => ({ text: s, value: s })),
      onFilter: (value, record) => record.status === value,
      render: (text) => (
        <Tag color={text === "Valid" ? "green" : text === "Expired" ? "red" : "orange"}>{text}</Tag>
      ),
    },
    {
      title: "Actions",
      render: (_, record) => (
        <Space size="small">
          <Tooltip title="View">
            <Button
              icon={<EyeOutlined />}
              onClick={() => setViewingCertification(record)}
              size="small"
              className="text-blue-600"
            />
          </Tooltip>
          <Tooltip title="Edit">
            <Button
              icon={<EditOutlined />}
              onClick={() => openEditModal(record)}
              size="small"
              type="primary"
              className="bg-blue-600 hover:bg-blue-700"
            />
          </Tooltip>
          <Tooltip title="Delete">
            <Button
              icon={<DeleteOutlined />}
              onClick={() => handleDelete(record.id)}
              danger
              size="small"
              className="text-red-600"
            />
          </Tooltip>
        </Space>
      ),
    },
  ];

  const CertificationCard = ({ certification }: { certification: SupplierCertification }) => (
    <div className="border rounded-xl p-4 shadow-sm hover:shadow-md bg-white dark:bg-gray-900 transition-all w-full">
      <div className="flex flex-col sm:flex-row justify-between items-start sm:items-center mb-2 gap-2">
        <h3 className="text-base sm:text-lg font-semibold text-gray-800 dark:text-white truncate">
          {certification.certificationType}
        </h3>
        <Tag color="teal">{dayjs(certification.issueDate).format("YYYY-MM-DD")}</Tag>
      </div>
      <p className="text-sm text-gray-600 dark:text-gray-300 mb-1"><strong>Number:</strong> {certification.certificationNumber}</p>
      <p className="text-sm text-gray-600 dark:text-gray-300 mb-1"><strong>Status:</strong> {certification.status}</p>
      <div className="mt-2 flex gap-2">
        <Tooltip title="View">
          <Button icon={<EyeOutlined />} onClick={() => setViewingCertification(certification)} size="small" className="text-blue-600" />
        </Tooltip>
        <Tooltip title="Edit">
          <Button icon={<EditOutlined />} onClick={() => openEditModal(certification)} size="small" type="primary" className="bg-blue-600 hover:bg-blue-700" />
        </Tooltip>
        <Tooltip title="Delete">
          <Button icon={<DeleteOutlined />} onClick={() => handleDelete(certification.id)} size="small" danger className="text-red-600" />
        </Tooltip>
      </div>
    </div>
  );

  return (
    <div className="px-2 sm:px-4 md:px-6 lg:px-8 py-4">
      <h2 className="text-xl sm:text-2xl md:text-3xl font-bold text-gray-800 dark:text-white mb-2">
        Supplier Certification - {decodeURIComponent(supplierName)}
      </h2>
      <p className="text-gray-600 dark:text-gray-300 text-sm sm:text-base mb-4">
        Manage certifications for the selected supplier.
      </p>
      <Breadcrumb
        className="mb-4 sm:mb-6 mt-2 text-sm text-gray-500 dark:text-gray-400"
        separator=">"
        items={[
          {
            title: <Tooltip title="Dashboard"><HomeOutlined /></Tooltip>,
            href: "/dashboard",
          },
          {
            title: "Suppliers",
            href: "/suppliers",
          },
          {
            title: "Certification",
          },
        ]}
      />
      <Button
     icon={<HomeOutlined />}
     onClick={() => router.push("/suppliers")}
     className="mt-3 mb-2 bg-green-600 hover:bg-green-700 text-white"
   >
     Back to Suppliers
   </Button>

      <div className="mt-4">
        <Card title="Certification List" className="p-0 mb-6 shadow-sm">
          <div className="p-4 flex flex-col gap-4">
            <div className="flex justify-end gap-4">
              <Button
                type="primary"
                icon={<PlusOutlined />}
                onClick={openAddModal}
                className="bg-blue-600 hover:bg-blue-700"
              >
                Add Certification
              </Button>
              <Switch
                checkedChildren={<TableOutlined />}
                unCheckedChildren={<AppstoreOutlined />}
                checked={isTableView}
                onChange={setIsTableView}
              />
            </div>
          </div>

          {!isTableView ? (
  <div className="overflow-auto p-4">
    <Table
      columns={columns}
      dataSource={certifications}
      rowKey="id"
      scroll={{ x: "max-content" }}
      pagination={{ pageSize: 10 }}
    />
  </div>
) : (
  <div className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-4 p-4">
    {certifications.map((certification) => (
      <CertificationCard key={certification.id} certification={certification} />
    ))}
  </div>
)}
          <motion.div
            initial={{ opacity: 0, scale: 0.95 }}
            animate={{ opacity: 1, scale: 1 }}
            exit={{ opacity: 0, scale: 0.95 }}
          >
            <Modal
              title={editingCertification ? "Edit Certification" : "Add Certification"}
              open={isModalOpen}
              onOk={handleFormSubmit}
              onCancel={() => {
                setIsModalOpen(false);
                form.resetFields();
                setOtherCertification("");
              }}
              destroyOnHidden
              className="w-full max-w-[90vw] sm:max-w-[600px] md:max-w-[800px] lg:max-w-[1000px] top-4"
            >
              <Form form={form} layout="vertical" className="px-2 sm:px-4">
                <Row gutter={[16, 16]}>
                  <Col xs={24} sm={12}>
                    <Form.Item
                      name="certificationType"
                      label="Certification Type"
                      rules={[{ required: true, message: "Certification Type is required" }]}
                    >
                      <Select
                        options={certificationTypeOptions.map((v) => ({ label: v, value: v }))}
                        onChange={(value) => {
                          if (value !== "Other") setOtherCertification("");
                          form.setFieldsValue({ certificationType: value });
                        }}
                      />
                    </Form.Item>
                  </Col>
                  {form.getFieldValue("certificationType") === "Other" && (
                    <Col xs={24} sm={12}>
                      <Form.Item
                        name="otherCertification"
                        label="Other Certification Type"
                        rules={[{ required: true, message: "Other Certification Type is required" }]}
                      >
                        <Input value={otherCertification} onChange={(e) => setOtherCertification(e.target.value)} />
                      </Form.Item>
                    </Col>
                  )}
                  <Col xs={24} sm={12}>
                    <Form.Item
                      name="certificationNumber"
                      label="Certification Number"
                      rules={[{ required: true, message: "Certification Number is required" }]}
                    >
                      <Input />
                    </Form.Item>
                  </Col>
                  <Col xs={24} sm={12}>
                    <Form.Item
                      name="issuingBody"
                      label="Issuing Body"
                      rules={[{ required: true, message: "Issuing Body is required" }]}
                    >
                      <Input />
                    </Form.Item>
                  </Col>
                  <Col xs={24} sm={12}>
                    <Form.Item
                      name="issueDate"
                      label="Issue Date"
                      rules={[{ required: true, message: "Issue Date is required" }]}
                    >
                      <DatePicker className="w-full" />
                    </Form.Item>
                  </Col>
                  <Col xs={24} sm={12}>
                    <Form.Item
                      name="expiryDate"
                      label="Expiry Date"
                      rules={[{ required: true, message: "Expiry Date is required" }]}
                    >
                      <DatePicker className="w-full" />
                    </Form.Item>
                  </Col>
                  <Col xs={24}>
                    <Form.Item name="scope" label="Scope">
                      <TextArea rows={2} />
                    </Form.Item>
                  </Col>
                  <Col xs={24} sm={12}>
                    <Form.Item
                      name="status"
                      label="Status"
                      rules={[{ required: true, message: "Status is required" }]}
                    >
                      <Select options={statusOptions.map((v) => ({ label: v, value: v }))} />
                    </Form.Item>
                  </Col>
                </Row>
              </Form>
            </Modal>
          </motion.div>

          <Modal
            title="Certification Details"
            open={!!viewingCertification}
            onCancel={() => setViewingCertification(null)}
            footer={null}
            className="w-full max-w-[90vw] sm:max-w-[500px] top-4"
          >
            {viewingCertification && (
              <div className="space-y-2 text-sm sm:text-base">
                <p><strong>Type:</strong> {viewingCertification.certificationType}</p>
                <p><strong>Number:</strong> {viewingCertification.certificationNumber}</p>
                <p><strong>Issuing Body:</strong> {viewingCertification.issuingBody}</p>
                <p><strong>Issue Date:</strong> {dayjs(viewingCertification.issueDate).format("YYYY-MM-DD")}</p>
                <p><strong>Expiry Date:</strong> {dayjs(viewingCertification.expiryDate).format("YYYY-MM-DD")}</p>
                <p><strong>Scope:</strong> {viewingCertification.scope || "-"}</p>
                <p><strong>Status:</strong> {viewingCertification.status}</p>
                <p><strong>Created:</strong> {dayjs(viewingCertification.created_at).format("YYYY-MM-DD")}</p>
              </div>
            )}
          </Modal>
        </Card>
      </div>
    </div>
  );
}