"use client";

import React, { useState, useEffect } from "react";
import { Button, Form, Input, Modal, Select, Space, Row, Col, Tooltip, Breadcrumb, Card, DatePicker, Switch, Table, Tag } from "antd";
import { PlusOutlined, EditOutlined, DeleteOutlined, EyeOutlined, HomeOutlined, FileTextOutlined, TableOutlined, AppstoreOutlined } from "@ant-design/icons";
import type { ColumnsType } from "antd/es/table";
import { toast } from "sonner";
import { motion } from "framer-motion";
import { useSearchParams, useRouter } from "next/navigation";
import dayjs from "dayjs";

const { TextArea } = Input;
const assessmentTypeOptions = ["Initial", "Routine", "Follow Up"] as const;
const statusOptions = ["Pass", "Fail", "Conditional"] as const;

interface SupplierAssessment {
  id: string;
  supplierId: string;
  supplierName: string;
  assessmentDate: string;
  assessmentType: typeof assessmentTypeOptions[number];
  assessor: string;
  score: number;
  findings: string;
  recommendations: string;
  status: typeof statusOptions[number];
  created_at: string;
}

export default function SupplierAssessmentPage() {
  const [form] = Form.useForm();
  const [assessments, setAssessments] = useState<SupplierAssessment[]>([]);
  const [isModalOpen, setIsModalOpen] = useState(false);
  const [editingAssessment, setEditingAssessment] = useState<SupplierAssessment | null>(null);
  const [viewingAssessment, setViewingAssessment] = useState<SupplierAssessment | null>(null);
  const [isTableView, setIsTableView] = useState(false);
  const searchParams = useSearchParams();
  const router = useRouter();
  const supplierId = searchParams.get("id");
  const supplierName = searchParams.get("name") || "Unknown Supplier";

  useEffect(() => {
    if (!supplierId) router.push("/suppliers");
  }, [supplierId, router]);

  const openAddModal = () => {
    form.resetFields();
    setEditingAssessment(null);
    setIsModalOpen(true);
  };

  const openEditModal = (assessment: SupplierAssessment) => {
    form.setFieldsValue({
      ...assessment,
      assessmentDate: assessment.assessmentDate ? dayjs(assessment.assessmentDate) : undefined,
    });
    setEditingAssessment(assessment);
    setIsModalOpen(true);
  };

  const handleDelete = (id: string) => {
    Modal.confirm({
      title: "Are you sure you want to delete this assessment?",
      onOk: () => {
        setAssessments((prev) => prev.filter((a) => a.id !== id));
        toast.success("Assessment deleted successfully");
      },
      onCancel: () => {},
    });
  };

  const handleFormSubmit = () => {
    form.validateFields().then((values: Omit<SupplierAssessment, "id" | "created_at">) => {
      const newAssessment: SupplierAssessment = {
        ...values,
        id: editingAssessment?.id ?? Date.now().toString(),
        supplierId: supplierId!,
        supplierName,
        assessmentDate: dayjs(values.assessmentDate).format("YYYY-MM-DD"),
        created_at: editingAssessment?.created_at ?? dayjs().toISOString(),
      };
      if (editingAssessment) {
        setAssessments((prev) => prev.map((a) => (a.id === editingAssessment.id ? newAssessment : a)));
        toast.success("Assessment updated successfully");
      } else {
        setAssessments((prev) => [...prev, newAssessment]);
        toast.success("Assessment added successfully");
      }
      setIsModalOpen(false);
      form.resetFields();
    });
  };

  const columns: ColumnsType<SupplierAssessment> = [
    {
      title: "Assessment Date",
      dataIndex: "assessmentDate",
      sorter: (a, b) => dayjs(a.assessmentDate).unix() - dayjs(b.assessmentDate).unix(),
      render: (text) => dayjs(text).format("YYYY-MM-DD"),
    },
    {
      title: "Type",
      dataIndex: "assessmentType",
      filters: assessmentTypeOptions.map((t) => ({ text: t, value: t })),
      onFilter: (value, record) => record.assessmentType === value,
    },
    {
      title: "Assessor",
      dataIndex: "assessor",
      sorter: (a, b) => a.assessor.localeCompare(b.assessor),
    },
    {
      title: "Score (%)",
      dataIndex: "score",
      sorter: (a, b) => (a.score ?? 0) - (b.score ?? 0),
      render: (text) => `${text || 0}%`,
    },
    {
      title: "Status",
      dataIndex: "status",
      filters: statusOptions.map((s) => ({ text: s, value: s })),
      onFilter: (value, record) => record.status === value,
      render: (text) => (
        <Tag color={text === "Pass" ? "green" : text === "Fail" ? "red" : "orange"}>{text}</Tag>
      ),
    },
    {
      title: "Actions",
      render: (_, record) => (
        <Space size="small">
          <Tooltip title="View">
            <Button
              icon={<EyeOutlined />}
              onClick={() => setViewingAssessment(record)}
              size="small"
              className="text-blue-600"
            />
          </Tooltip>
          <Tooltip title="Edit">
            <Button
              icon={<EditOutlined />}
              onClick={() => openEditModal(record)}
              size="small"
              type="primary"
              className="bg-blue-600 hover:bg-blue-700"
            />
          </Tooltip>
          <Tooltip title="Delete">
            <Button
              icon={<DeleteOutlined />}
              onClick={() => handleDelete(record.id)}
              danger
              size="small"
              className="text-red-600"
            />
          </Tooltip>
        </Space>
      ),
    },
  ];

  const AssessmentCard = ({ assessment }: { assessment: SupplierAssessment }) => (
    <div className="border rounded-xl p-4 shadow-sm hover:shadow-md bg-white dark:bg-gray-900 transition-all w-full">
      <div className="flex flex-col sm:flex-row justify-between items-start sm:items-center mb-2 gap-2">
        <h3 className="text-base sm:text-lg font-semibold text-gray-800 dark:text-white truncate">
          {assessment.assessmentType} Assessment
        </h3>
        <Tag color="purple">{dayjs(assessment.assessmentDate).format("YYYY-MM-DD")}</Tag>
      </div>
      <p className="text-sm text-gray-600 dark:text-gray-300 mb-1"><strong>Assessor:</strong> {assessment.assessor}</p>
      <p className="text-sm text-gray-600 dark:text-gray-300 mb-1"><strong>Score:</strong> {assessment.score}%</p>
      <p className="text-sm text-gray-600 dark:text-gray-300 mb-1"><strong>Status:</strong> {assessment.status}</p>
      <div className="mt-2 flex gap-2">
        <Tooltip title="View">
          <Button icon={<EyeOutlined />} onClick={() => setViewingAssessment(assessment)} size="small" className="text-blue-600" />
        </Tooltip>
        <Tooltip title="Edit">
          <Button icon={<EditOutlined />} onClick={() => openEditModal(assessment)} size="small" type="primary" className="bg-blue-600 hover:bg-blue-700" />
        </Tooltip>
        <Tooltip title="Delete">
          <Button icon={<DeleteOutlined />} onClick={() => handleDelete(assessment.id)} size="small" danger className="text-red-600" />
        </Tooltip>
      </div>
    </div>
  );

  return (
    <div className="px-2 sm:px-4 md:px-6 lg:px-8 py-4">
      <h2 className="text-xl sm:text-2xl md:text-3xl font-bold text-gray-800 dark:text-white mb-2">
        Supplier Assessment - {decodeURIComponent(supplierName)}
      </h2>
      <p className="text-gray-600 dark:text-gray-300 text-sm sm:text-base mb-4">
        Manage assessments for the selected supplier.
      </p>
      <Breadcrumb
        className="mb-4 sm:mb-6 mt-2 text-sm text-gray-500 dark:text-gray-400"
        separator=">"
        items={[
          {
            title: <Tooltip title="Dashboard"><HomeOutlined /></Tooltip>,
            href: "/dashboard",
          },
          {
            title: "Suppliers",
            href: "/suppliers",
          },
          {
            title: "Assessment",
          },
        ]}
      />
      <Button
  icon={<HomeOutlined />}
  onClick={() => router.push("/suppliers")}
  className="mt-3 mb-2 bg-green-600 hover:bg-green-700 text-white"
>
  Back to Suppliers
</Button>

      <div className="mt-4">
        <Card title="Assessment List" className="p-0 mb-6 shadow-sm">
          <div className="p-4 flex flex-col gap-4">
            <div className="flex justify-end gap-4">
              <Button
                type="primary"
                icon={<PlusOutlined />}
                onClick={openAddModal}
                className="bg-blue-600 hover:bg-blue-700"
              >
                Add Assessment
              </Button>
              <Switch
                checkedChildren={<TableOutlined />}
                unCheckedChildren={<AppstoreOutlined />}
                checked={isTableView}
                onChange={setIsTableView}
              />
            </div>
          </div>

          {!isTableView ? (
  <div className="overflow-auto p-4">
    <Table
      columns={columns}
      dataSource={assessments}
      rowKey="id"
      scroll={{ x: "max-content" }}
      pagination={{ pageSize: 10 }}
    />
  </div>
) : (
  <div className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-4 p-4">
    {assessments.map((assessment) => (
      <AssessmentCard key={assessment.id} assessment={assessment} />
    ))}
  </div>
)}

          <motion.div
            initial={{ opacity: 0, scale: 0.95 }}
            animate={{ opacity: 1, scale: 1 }}
            exit={{ opacity: 0, scale: 0.95 }}
          >
            <Modal
              title={editingAssessment ? "Edit Assessment" : "Add Assessment"}
              open={isModalOpen}
              onOk={handleFormSubmit}
              onCancel={() => {
                setIsModalOpen(false);
                form.resetFields();
              }}
              destroyOnHidden
              className="w-full max-w-[90vw] sm:max-w-[600px] md:max-w-[800px] lg:max-w-[1000px] top-4"
            >
              <Form form={form} layout="vertical" className="px-2 sm:px-4">
                <Row gutter={[16, 16]}>
                  <Col xs={24} sm={12}>
                    <Form.Item
                      name="assessmentDate"
                      label="Assessment Date"
                      rules={[{ required: true, message: "Assessment Date is required" }]}
                    >
                      <DatePicker className="w-full" />
                    </Form.Item>
                  </Col>
                  <Col xs={24} sm={12}>
                    <Form.Item
                      name="assessmentType"
                      label="Assessment Type"
                      rules={[{ required: true, message: "Assessment Type is required" }]}
                    >
                      <Select options={assessmentTypeOptions.map((v) => ({ label: v, value: v }))} />
                    </Form.Item>
                  </Col>
                  <Col xs={24} sm={12}>
                    <Form.Item
                      name="assessor"
                      label="Assessor"
                      rules={[{ required: true, message: "Assessor is required" }]}
                    >
                      <Input />
                    </Form.Item>
                  </Col>
                  <Col xs={24} sm={12}>
                    <Form.Item
                      name="score"
                      label="Score (%)"
                      rules={[{ required: true, message: "Score is required" }]}
                    >
                      <Input type="number" min={0} max={100} />
                    </Form.Item>
                  </Col>
                  <Col xs={24}>
                    <Form.Item name="findings" label="Findings">
                      <TextArea rows={2} />
                    </Form.Item>
                  </Col>
                  <Col xs={24}>
                    <Form.Item name="recommendations" label="Recommendations">
                      <TextArea rows={2} />
                    </Form.Item>
                  </Col>
                  <Col xs={24} sm={12}>
                    <Form.Item
                      name="status"
                      label="Status"
                      rules={[{ required: true, message: "Status is required" }]}
                    >
                      <Select options={statusOptions.map((v) => ({ label: v, value: v }))} />
                    </Form.Item>
                  </Col>
                </Row>
              </Form>
            </Modal>
          </motion.div>

          <Modal
            title="Assessment Details"
            open={!!viewingAssessment}
            onCancel={() => setViewingAssessment(null)}
            footer={null}
            className="w-full max-w-[90vw] sm:max-w-[500px] top-4"
          >
            {viewingAssessment && (
              <div className="space-y-2 text-sm sm:text-base">
                <p><strong>Date:</strong> {dayjs(viewingAssessment.assessmentDate).format("YYYY-MM-DD")}</p>
                <p><strong>Type:</strong> {viewingAssessment.assessmentType}</p>
                <p><strong>Assessor:</strong> {viewingAssessment.assessor}</p>
                <p><strong>Score:</strong> {viewingAssessment.score}%</p>
                <p><strong>Findings:</strong> {viewingAssessment.findings || "-"}</p>
                <p><strong>Recommendations:</strong> {viewingAssessment.recommendations || "-"}</p>
                <p><strong>Status:</strong> {viewingAssessment.status}</p>
                <p><strong>Created:</strong> {dayjs(viewingAssessment.created_at).format("YYYY-MM-DD")}</p>
              </div>
            )}
          </Modal>
        </Card>
      </div>
    </div>
  );
}