"use client";

import React, { useCallback, useState } from "react";
import {
  Breadcrumb,
  Card,
  Input,
  Space,
  Tooltip,
  Tag,
  Button,
  Switch,
  Modal,
  Pagination,
  Dropdown,
  Select,
} from "antd";
import {
  HomeOutlined,
  TableOutlined,
  AppstoreOutlined,
  EyeOutlined,
  SearchOutlined,
  DownloadOutlined,
  CheckCircleTwoTone,
  WarningOutlined,
  StopOutlined,
  CheckCircleOutlined,
  PauseCircleOutlined,
  FireOutlined,
  CloseCircleOutlined,
} from "@ant-design/icons";
import type { ColumnsType } from "antd/es/table";
import DataTable from "../components/common/DataTable";
import WidgetCard from "../components/common/WidgetCard";
import ColumnVisibilityToggle from "../components/common/ManageColumn";
import { useIsMobile } from "@/data/useIsMobile";
import type { SupplierApproval } from "../types/SupplierApproval";
import { toast } from "sonner";
import { motion } from "framer-motion";

const validationOptions = ["Approved", "Rejected", "Pending"] as const;
const riskOptions = ["Low", "Medium", "High"] as const;

export default function SupplierApprovalPage() {
  const [searchTerm, setSearchTerm] = useState("");
  const [isGridView, setIsGridView] = useState(false);
  const [viewingSupplier, setViewingSupplier] = useState<SupplierApproval | null>(null);
  const [data, setData] = useState<SupplierApproval[]>([]);
  const [selectedRows, setSelectedRows] = useState<string[]>([]);
  const [advancedFilters, setAdvancedFilters] = useState({
    riskClassification: undefined,
    validation: undefined,
  });
  const [currentPage, setCurrentPage] = useState(1);
  const [pageSize, setPageSize] = useState(10);
  const isMobile = useIsMobile();

  const renderTag = (value: string) => {
    const colorMap: Record<string, string> = {
      Low: "green",
      Medium: "orange",
      High: "red",
      Approved: "green",
      Rejected: "red",
      Pending: "gold",
    };

    const iconMap: Record<string, React.ReactNode> = {
      Low: <CheckCircleOutlined />,
      Medium: <PauseCircleOutlined />,
      High: <FireOutlined />,
      Approved: <CheckCircleOutlined />,
      Rejected: <CloseCircleOutlined />,
      Pending: <WarningOutlined />,
    };

    return (
      <Tag icon={iconMap[value]} color={colorMap[value] || "default"}>
        {value}
      </Tag>
    );
  };

  const filteredData = data.filter((supplier) => {
    const matchesSearch =
      supplier.companyName.toLowerCase().includes(searchTerm.toLowerCase()) ||
      supplier.supplierCode.toLowerCase().includes(searchTerm.toLowerCase());
    const matchesAdvanced =
      (!advancedFilters.riskClassification || supplier.riskClassification === advancedFilters.riskClassification) &&
      (!advancedFilters.validation || supplier.validation === advancedFilters.validation);
    return matchesSearch && matchesAdvanced;
  });

  const paginatedData = filteredData.slice(
    (currentPage - 1) * pageSize,
    currentPage * pageSize
  );

  const handleBulkDelete = () => {
    Modal.confirm({
      title: `Are you sure you want to delete ${selectedRows.length} supplier approvals?`,
      onOk: () => {
        setData((prev) => prev.filter((s) => !selectedRows.includes(s.id.toString())));
        setSelectedRows([]);
        toast.success("Supplier approvals deleted successfully");
      },
    });
  };

  const columns: ColumnsType<SupplierApproval> = [
    {
      title: (
        <input
          type="checkbox"
          onChange={(e) => {
            const checked = e.target.checked;
            setSelectedRows(checked ? filteredData.map((s) => s.id.toString()) : []);
          }}
          className="w-4 h-4"
        />
      ),
      key: "selection",
      render: (_, record) => (
        <input
          type="checkbox"
          checked={selectedRows.includes(record.id.toString())}
          onChange={(e) => {
            setSelectedRows((prev) =>
              e.target.checked
                ? [...prev, record.id.toString()]
                : prev.filter((id) => id !== record.id.toString())
            );
          }}
          className="w-4 h-4"
        />
      ),
      width: 44,
    },
    {
      title: "Supplier Code",
      dataIndex: "supplierCode",
      key: "supplierCode",
      sorter: (a, b) => a.supplierCode.localeCompare(b.supplierCode),
      filterSearch: true,
      onFilter: (value, record) =>
        record.supplierCode.toLowerCase().includes((value as string).toLowerCase()),
      filters: Array.from(new Set(data.map((s) => s.supplierCode))).map((value) => ({
        text: value,
        value: value,
      })),
    },
    {
      title: "Company Name",
      dataIndex: "companyName",
      key: "companyName",
      sorter: (a, b) => a.companyName.localeCompare(b.companyName),
      render: (text) => <span className="text-green-600 font-medium">{text}</span>,
      filterSearch: true,
      onFilter: (value, record) =>
        record.companyName.toLowerCase().includes((value as string).toLowerCase()),
      filters: Array.from(new Set(data.map((s) => s.companyName))).map((value) => ({
        text: value,
        value: value,
      })),
    },
    {
      title: "Contact Person",
      dataIndex: "contactPerson",
      key: "contactPerson",
      sorter: (a, b) => (a.contactPerson || "").localeCompare(b.contactPerson || ""),
      filterSearch: true,
      onFilter: (value, record) =>
        (record.contactPerson || "").toLowerCase().includes((value as string).toLowerCase()),
      filters: Array.from(new Set(data.map((s) => s.contactPerson).filter((v): v is string => !!v))).map((value) => ({
        text: value,
        value: value,
      })),
    },
    {
      title: "Email",
      dataIndex: "email",
      key: "email",
      sorter: (a, b) => (a.email || "").localeCompare(b.email || ""),
    },
    {
      title: "Phone",
      dataIndex: "phone",
      key: "phone",
      sorter: (a, b) => (a.phone || "").localeCompare(b.phone || ""),
    },
    {
      title: "Risk Classification",
      dataIndex: "riskClassification",
      key: "riskClassification",
      filters: riskOptions.map((r) => ({ text: r, value: r })),
      onFilter: (value, record) => record.riskClassification === value,
      render: renderTag,
      sorter: (a, b) => a.riskClassification.localeCompare(b.riskClassification),
    },
    {
      title: "Validation",
      dataIndex: "validation",
      key: "validation",
      filters: validationOptions.map((v) => ({ text: v, value: v })),
      onFilter: (value, record) => record.validation === value,
      render: renderTag,
      sorter: (a, b) => a.validation.localeCompare(b.validation),
    },
    {
      title: "Actions",
      key: "actions",
      fixed: "right",
      render: (_, record) => (
        <Space size="small">
          <Tooltip title="View">
            <Button
              icon={<EyeOutlined />}
              onClick={() => setViewingSupplier(record)}
              size="small"
              className="text-blue-600"
            />
          </Tooltip>
        </Space>
      ),
    },
  ];

  const allColumns = [
    { key: "selection", label: "Select" },
    { key: "supplierCode", label: "Supplier Code" },
    { key: "companyName", label: "Company Name" },
    { key: "contactPerson", label: "Contact Person" },
    { key: "email", label: "Email" },
    { key: "phone", label: "Phone" },
    { key: "riskClassification", label: "Risk Classification" },
    { key: "validation", label: "Validation" },
    { key: "actions", label: "Actions" },
  ];

  const columnKeys = allColumns.map((c) => c.key);
  const labelsMap = Object.fromEntries(allColumns.map((c) => [c.key, c.label]));
  const [visibleColumnKeys, setVisibleColumnKeys] = useState<string[]>(columnKeys);

  const handleVisibleColumnChange = useCallback((cols: string[]) => {
    setVisibleColumnKeys(cols);
  }, []);

  const filteredColumns = columns.filter(
    (col) => typeof col.key === "string" && visibleColumnKeys.includes(col.key)
  );

  const SupplierCard = ({ supplier }: { supplier: SupplierApproval }) => (
    <div className="border rounded-xl p-4 shadow-sm hover:shadow-md bg-white dark:bg-gray-900 transition-all w-full">
      <div className="flex flex-col sm:flex-row justify-between items-start sm:items-center mb-2 gap-2">
        <h3 className="text-base sm:text-lg font-semibold text-gray-800 dark:text-white truncate">{supplier.companyName}</h3>
        <Tag color="blue">{supplier.validation}</Tag>
      </div>
      <p className="text-sm text-gray-600 dark:text-gray-300 mb-1 truncate"><strong>Code:</strong> {supplier.supplierCode}</p>
      <p className="text-sm text-gray-600 dark:text-gray-300 mb-1"><strong>Risk:</strong> {renderTag(supplier.riskClassification)}</p>
      <p className="text-sm text-gray-600 dark:text-gray-300 mb-1"><strong>Validation:</strong> {renderTag(supplier.validation)}</p>
      <div className="mt-2 flex gap-2">
        <Tooltip title="View">
          <Button icon={<EyeOutlined />} onClick={() => setViewingSupplier(supplier)} size="small" className="text-blue-600" />
        </Tooltip>
      </div>
    </div>
  );

  const total = data.length;
  const approved = data.filter((s) => s.validation === "Approved").length;
  const rejected = data.filter((s) => s.validation === "Rejected").length;
  const pending = data.filter((s) => s.validation === "Pending").length;
  const highRisk = data.filter((s) => s.riskClassification === "High").length;

  // Sample trend data for widgets
  const trendData = [
    { value: 10 },
    { value: 15 },
    { value: 12 },
    { value: 18 },
    { value: 20 },
  ];

  return (
    <>
      <div className="px-2 sm:px-4 md:px-6 lg:px-8 py-4">
        <h2 className="text-xl sm:text-2xl md:text-3xl font-bold text-gray-800 dark:text-white mb-2">
          Supplier Approvals <span className="text-gray-500 text-base sm:text-lg font-medium">({data.length})</span>
        </h2>
        <p className="text-gray-600 dark:text-gray-300 text-sm sm:text-base mb-4">
          View and monitor supplier approval statuses.
        </p>
        <Breadcrumb
          className="mb-4 sm:mb-6 mt-2 text-sm text-gray-500 dark:text-gray-400"
          separator=">"
          items={[
            {
              title: (
                <Tooltip title="Dashboard">
                  <HomeOutlined />
                </Tooltip>
              ),
              href: "/dashboard",
            },
            {
              title: "Master Data",
            },
            {
              title: "Supplier Approvals",
            },
          ]}
        />

        <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-4 mt-3 mb-6">
          <WidgetCard
            title="Total Approvals"
            value={total}
            icon={<AppstoreOutlined />}
            color="green"
            percentage={12}
            description="All supplier approval records"
            trendData={trendData}
          />
          <WidgetCard
            title="Approved"
            value={approved}
            icon={<CheckCircleTwoTone twoToneColor="#52c41a" />}
            color="blue"
            percentage={8}
            description="Suppliers with approved status"
            trendData={trendData}
          />
          <WidgetCard
            title="Rejected"
            value={rejected}
            icon={<StopOutlined />}
            color="red"
            percentage={-3}
            description="Suppliers with rejected status"
            trendData={trendData}
          />
          <WidgetCard
            title="Pending Approvals"
            value={pending}
            icon={<WarningOutlined />}
            color="orange"
            percentage={5}
            description="Suppliers awaiting approval"
            trendData={trendData}
          />
          {/* <WidgetCard
            title="High Risk"
            value={highRisk}
            icon={<WarningOutlined />}
            color="orange"
            percentage={5}
            description="High-risk suppliers"
            trendData={trendData}
          /> */}
        </div>

        <div className="mt-4">
          <Card title="Supplier Approval List" className="p-0 mb-6 shadow-sm">
            <div className="p-4 flex flex-col gap-4">
              <div className="flex flex-col sm:flex-row justify-between items-stretch sm:items-center gap-4">
                <Input
                  prefix={<SearchOutlined />}
                  allowClear
                  placeholder="Search Supplier or Code"
                  value={searchTerm}
                  onChange={(e) => setSearchTerm(e.target.value)}
                  className="w-full sm:w-64 border-gray-300 dark:border-gray-600 rounded-md shadow-sm"
                />
                <Dropdown
                  className="w-full sm:w-auto border-gray-300 dark:border-gray-600"
                  menu={{
                    items: [
                      { label: "Export CSV", key: "csv" },
                      { label: "Export PDF", key: "pdf" },
                    ],
                  }}
                >
                  <Button icon={<DownloadOutlined />}>Export</Button>
                </Dropdown>
              </div>
              <div className="flex flex-col gap-4 lg:flex-row lg:justify-between lg:items-center">
                <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-4 w-full lg:w-auto">
                  <Select
                    placeholder="Risk Level"
                    className="w-full max-w-[160px]"
                    onChange={(value) =>
                      setAdvancedFilters((prev) => ({ ...prev, riskClassification: value }))
                    }
                    allowClear
                    options={riskOptions.map((v) => ({ label: v, value: v }))}
                  />
                  <Select
                    placeholder="Validation Status"
                    className="w-full max-w-[160px]"
                    onChange={(value) =>
                      setAdvancedFilters((prev) => ({ ...prev, validation: value }))
                    }
                    allowClear
                    options={validationOptions.map((v) => ({ label: v, value: v }))}
                  />
                </div>
                <div className="flex flex-col sm:flex-row flex-wrap gap-4 justify-start lg:justify-end w-full lg:w-auto">
                  <ColumnVisibilityToggle
                    columns={columnKeys}
                    labels={labelsMap}
                    storageKey="supplier_approval_table_visibility"
                    onChange={handleVisibleColumnChange}
                  />
                  <Button
                    onClick={handleBulkDelete}
                    disabled={selectedRows.length === 0}
                    danger
                    className="w-full sm:w-auto"
                  >
                    Bulk Delete
                  </Button>
                  <div className="hidden sm:block">
                    <Switch
                      checkedChildren={<AppstoreOutlined />}
                      unCheckedChildren={<TableOutlined />}
                      checked={isGridView}
                      onChange={setIsGridView}
                      className="w-full sm:w-auto"
                    />
                  </div>
                </div>
              </div>

              {(advancedFilters.riskClassification || advancedFilters.validation) && (
                <div className="flex flex-wrap gap-2 mt-2">
                  {advancedFilters.riskClassification && (
                    <Tag
                      closable
                      onClose={() =>
                        setAdvancedFilters((prev) => ({
                          ...prev,
                          riskClassification: undefined,
                        }))
                      }
                    >
                      Risk Level: {advancedFilters.riskClassification}
                    </Tag>
                  )}
                  {advancedFilters.validation && (
                    <Tag
                      closable
                      onClose={() =>
                        setAdvancedFilters((prev) => ({
                          ...prev,
                          validation: undefined,
                        }))
                      }
                    >
                      Validation: {advancedFilters.validation}
                    </Tag>
                  )}
                </div>
              )}
            </div>

            {isGridView || isMobile ? (
              <div className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-4 p-4">
                {paginatedData.map((supplier) => (
                  <SupplierCard key={supplier.id} supplier={supplier} />
                ))}
              </div>
            ) : (
              <div className="overflow-auto">
                <DataTable<SupplierApproval>
                  columns={filteredColumns}
                  dataSource={paginatedData}
                  rowKey="id"
                  scrollX="max-content"
                  className="mt-4"
                />
              </div>
            )}
            <div className="flex flex-col sm:flex-row justify-between items-center px-4 py-2 text-sm text-gray-600 dark:text-gray-300">
              <div className="mb-2 sm:mb-0">
                Showing <strong>{(currentPage - 1) * pageSize + 1}</strong> to{" "}
                <strong>{Math.min(currentPage * pageSize, filteredData.length)}</strong> of{" "}
                <strong>{filteredData.length}</strong> approvals
              </div>
              <Pagination
                current={currentPage}
                pageSize={pageSize}
                total={filteredData.length}
                onChange={(page, size) => {
                  setCurrentPage(page);
                  setPageSize(size);
                }}
                showSizeChanger
                pageSizeOptions={["10", "30", "50", "100"]}
                size="small"
              />
            </div>
          </Card>
        </div>

        <motion.div
          initial={{ opacity: 0, scale: 0.95 }}
          animate={{ opacity: 1, scale: 1 }}
          exit={{ opacity: 0, scale: 0.95 }}
        >
          <Modal
            title="Supplier Details"
            open={!!viewingSupplier}
            onCancel={() => setViewingSupplier(null)}
            footer={null}
            className="w-full max-w-[90vw] sm:max-w-[500px] top-4"
          >
            {viewingSupplier && (
              <div className="space-y-2 text-sm sm:text-base">
                <p><strong>Supplier Code:</strong> {viewingSupplier.supplierCode}</p>
                <p><strong>Company Name:</strong> <span className="text-green-600">{viewingSupplier.companyName}</span></p>
                <p><strong>Contact Person:</strong> {viewingSupplier.contactPerson || "-"}</p>
                <p><strong>Email:</strong> {viewingSupplier.email || "-"}</p>
                <p><strong>Phone:</strong> {viewingSupplier.phone || "-"}</p>
                <p><strong>Risk Classification:</strong> {viewingSupplier.riskClassification}</p>
                <p><strong>Validation:</strong> {viewingSupplier.validation}</p>
                 <p><strong>Created_at:</strong> {viewingSupplier.created_at}</p>
              </div>
            )}
          </Modal>
        </motion.div>
      </div>
    </>
  );
}