"use client";

import React, { useCallback, useState } from "react";
import {
  Button,
  Card,
  Col,
  DatePicker,
  Form,
  Input,
  Modal,
  Row,
  Select,
  Space,
  Tooltip,
  Tag,
  Breadcrumb,
  Switch,
  Pagination,
  Dropdown,
} from "antd";
import {
  PlusOutlined,
  EditOutlined,
  DeleteOutlined,
  EyeOutlined,
  HomeOutlined,
  TableOutlined,
  AppstoreOutlined,
  SearchOutlined,
  DownloadOutlined,
  CheckCircleTwoTone,
  WarningOutlined,
  ClockCircleOutlined,
  CheckCircleOutlined,
  CloseCircleOutlined,
} from "@ant-design/icons";
import type { ColumnsType } from "antd/es/table";
import DataTable from "../components/common/DataTable";
import WidgetCard from "../components/common/WidgetCard";
import ColumnVisibilityToggle from "../components/common/ManageColumn";
import { useIsMobile } from "@/data/useIsMobile";
import type { SupplierAction } from "../types/SupplierAction";
import { toast } from "sonner";
import { motion } from "framer-motion";
import dayjs from "dayjs";

const statusOptions = ["Pending", "InProgress", "Completed", "Closed"] as const;

export default function SupplierActionPage() {
  const [form] = Form.useForm();
  const [actions, setActions] = useState<SupplierAction[]>([]);
  const [isModalOpen, setIsModalOpen] = useState(false);
  const [editingAction, setEditingAction] = useState<SupplierAction | null>(null);
  const [viewingAction, setViewingAction] = useState<SupplierAction | null>(null);
  const [searchTerm, setSearchTerm] = useState("");
  const [isGridView, setIsGridView] = useState(false);
  const [selectedRows, setSelectedRows] = useState<string[]>([]);
  const [advancedFilters, setAdvancedFilters] = useState({
    status: undefined,
    unitName: undefined,
  });
  const [dateRange, setDateRange] = useState<[dayjs.Dayjs, dayjs.Dayjs] | null>(null);
  const [currentPage, setCurrentPage] = useState(1);
  const [pageSize, setPageSize] = useState(10);
  const isMobile = useIsMobile();

  const openAddModal = () => {
    form.resetFields();
    setEditingAction(null);
    setIsModalOpen(true);
  };

  const openEditModal = (action: SupplierAction) => {
    form.setFieldsValue({
      ...action,
      dateIdentified: action.dateIdentified ? dayjs(action.dateIdentified) : undefined,
    });
    setEditingAction(action);
    setIsModalOpen(true);
  };

  const handleDelete = (id: string) => {
    Modal.confirm({
      title: "Are you sure you want to delete this action?",
      onOk: () => {
        setActions((prev) => prev.filter((a) => a.id !== id));
        toast.success("Action deleted successfully");
      },
      onCancel: () => {},
    });
  };

  const handleBulkDelete = () => {
    Modal.confirm({
      title: `Are you sure you want to delete ${selectedRows.length} actions?`,
      onOk: () => {
        setActions((prev) => prev.filter((a) => !selectedRows.includes(a.id.toString())));
        setSelectedRows([]);
        toast.success("Actions deleted successfully");
      },
    });
  };

  const handleFormSubmit = () => {
    form.validateFields().then((values) => {
      const data: SupplierAction = {
        ...values,
        id: editingAction ? editingAction.id : Date.now().toString(),
        dateIdentified: dayjs(values.dateIdentified).format("YYYY-MM-DD"),
        created_at: editingAction ? editingAction.created_at : dayjs().toISOString(),
      };
      if (editingAction) {
        setActions((prev) => prev.map((a) => (a.id === editingAction.id ? data : a)));
        toast.success("Action updated successfully");
      } else {
        setActions((prev) => [...prev, data]);
        toast.success("Action added successfully");
      }
      setIsModalOpen(false);
      form.resetFields();
    });
  };

  const renderTag = (value: string) => {
    const colorMap: Record<string, string> = {
      Pending: "gold",
      InProgress: "blue",
      Completed: "green",
      Closed: "red",
    };

    const iconMap: Record<string, React.ReactNode> = {
      Pending: <WarningOutlined />,
      InProgress: <ClockCircleOutlined />,
      Completed: <CheckCircleOutlined />,
      Closed: <CloseCircleOutlined />,
    };

    return (
      <Tag icon={iconMap[value]} color={colorMap[value] || "default"}>
        {value}
      </Tag>
    );
  };

  const filteredActions = actions.filter((action) => {
    const matchesSearch =
      action.supplier.toLowerCase().includes(searchTerm.toLowerCase()) ||
      action.materialDescription.toLowerCase().includes(searchTerm.toLowerCase());
    const matchesDate =
      !dateRange ||
      (action.created_at &&
        (!dateRange[0] ||
          dayjs(action.created_at).isSame(dateRange[0]) ||
          dayjs(action.created_at).isAfter(dateRange[0])) &&
        (!dateRange[1] ||
          dayjs(action.created_at).isSame(dateRange[1]) ||
          dayjs(action.created_at).isBefore(dateRange[1])));
    const matchesAdvanced =
      (!advancedFilters.status || action.status === advancedFilters.status) &&
      (!advancedFilters.unitName || action.unitName === advancedFilters.unitName);
    return matchesSearch && matchesDate && matchesAdvanced;
  });

  const paginatedActions = filteredActions.slice(
    (currentPage - 1) * pageSize,
    currentPage * pageSize
  );

  const columns: ColumnsType<SupplierAction> = [
    {
      title: (
        <input
          type="checkbox"
          onChange={(e) => {
            const checked = e.target.checked;
            setSelectedRows(checked ? filteredActions.map((a) => a.id.toString()) : []);
          }}
          className="w-4 h-4"
        />
      ),
      key: "selection",
      render: (_, record) => (
        <input
          type="checkbox"
          checked={selectedRows.includes(record.id.toString())}
          onChange={(e) => {
            setSelectedRows((prev) =>
              e.target.checked
                ? [...prev, record.id.toString()]
                : prev.filter((id) => id !== record.id.toString())
            );
          }}
          className="w-4 h-4"
        />
      ),
      width: 44,
    },
    {
      title: "Unit",
      dataIndex: "unitName",
      key: "unitName",
      sorter: (a, b) => a.unitName.localeCompare(b.unitName),
      filterSearch: true,
      onFilter: (value, record) =>
        record.unitName.toLowerCase().includes((value as string).toLowerCase()),
      filters: Array.from(new Set(actions.map((a) => a.unitName))).map((value) => ({
        text: value,
        value: value,
      })),
    },
    {
      title: "Supplier",
      dataIndex: "supplier",
      key: "supplier",
      sorter: (a, b) => a.supplier.localeCompare(b.supplier),
      render: (text) => <span className="text-green-600 font-medium">{text}</span>,
      filterSearch: true,
      onFilter: (value, record) =>
        record.supplier.toLowerCase().includes((value as string).toLowerCase()),
      filters: Array.from(new Set(actions.map((a) => a.supplier))).map((value) => ({
        text: value,
        value: value,
      })),
    },
    {
      title: "Material",
      dataIndex: "materialDescription",
      key: "materialDescription",
      sorter: (a, b) => a.materialDescription.localeCompare(b.materialDescription),
      filterSearch: true,
      onFilter: (value, record) =>
        record.materialDescription.toLowerCase().includes((value as string).toLowerCase()),
      filters: Array.from(new Set(actions.map((a) => a.materialDescription))).map((value) => ({
        text: value,
        value: value,
      })),
    },
    {
      title: "Invoice No.",
      dataIndex: "invoiceNo",
      key: "invoiceNo",
      sorter: (a, b) => a.invoiceNo.localeCompare(b.invoiceNo),
      render: (text) => (
        <Tooltip title={text}>
          <span className="block truncate max-w-[100px] sm:max-w-[150px] md:max-w-[180px]">{text}</span>
        </Tooltip>
      ),
    },
    {
      title: "Supplier Lot No.",
      dataIndex: "supplierLotNo",
      key: "supplierLotNo",
      render: (text) => (
        <Tooltip title={text}>
          <span className="block truncate max-w-[100px] sm:max-w-[150px] md:max-w-[180px]">{text}</span>
        </Tooltip>
      ),
    },
    {
      title: "Date Identified",
      dataIndex: "dateIdentified",
      key: "dateIdentified",
      sorter: (a, b) => dayjs(a.dateIdentified).unix() - dayjs(b.dateIdentified).unix(),
      render: (text) => dayjs(text).format("YYYY-MM-DD"),
    },
    {
      title: "CAPA No.",
      dataIndex: "capaNo",
      key: "capaNo",
      render: (text) => (
        <Tooltip title={text}>
          <span className="block truncate max-w-[100px] sm:max-w-[150px] md:max-w-[180px]">{text}</span>
        </Tooltip>
      ),
    },
    {
      title: "Status",
      dataIndex: "status",
      key: "status",
      filters: statusOptions.map((status) => ({ text: status, value: status })),
      onFilter: (value, record) => record.status === value,
      render: renderTag,
      sorter: (a, b) => a.status.localeCompare(b.status),
    },
    {
      title: "Actions",
      key: "actions",
      fixed: "right",
      render: (_, record) => (
        <Space size="small">
          <Tooltip title="View">
            <Button
              icon={<EyeOutlined />}
              onClick={() => setViewingAction(record)}
              size="small"
              className="text-blue-600"
            />
          </Tooltip>
          <Tooltip title="Edit">
            <Button
              icon={<EditOutlined />}
              onClick={() => openEditModal(record)}
              size="small"
              type="primary"
              className="bg-blue-600 hover:bg-blue-700"
            />
          </Tooltip>
          <Tooltip title="Delete">
            <Button
              icon={<DeleteOutlined />}
              onClick={() => handleDelete(record.id)}
              danger
              size="small"
              className="text-red-600"
            />
          </Tooltip>
        </Space>
      ),
    },
  ];

  const allColumns = [
    { key: "selection", label: "Select" },
    { key: "unitName", label: "Unit" },
    { key: "supplier", label: "Supplier" },
    { key: "materialDescription", label: "Material" },
    { key: "invoiceNo", label: "Invoice No." },
    { key: "supplierLotNo", label: "Supplier Lot No." },
    { key: "dateIdentified", label: "Date Identified" },
    { key: "capaNo", label: "CAPA No." },
    { key: "status", label: "Status" },
    { key: "actions", label: "Actions" },
  ];

  const columnKeys = allColumns.map((c) => c.key);
  const labelsMap = Object.fromEntries(allColumns.map((c) => [c.key, c.label]));
  const [visibleColumnKeys, setVisibleColumnKeys] = useState<string[]>(columnKeys);

  const handleVisibleColumnChange = useCallback((cols: string[]) => {
    setVisibleColumnKeys(cols);
  }, []);

  const filteredColumns = columns.filter(
    (col) => typeof col.key === "string" && visibleColumnKeys.includes(col.key)
  );

  const ActionCard = ({ action }: { action: SupplierAction }) => (
    <div className="border rounded-xl p-4 shadow-sm hover:shadow-md bg-white dark:bg-gray-900 transition-all w-full">
      <div className="flex flex-col sm:flex-row justify-between items-start sm:items-center mb-2 gap-2">
        <h3 className="text-base sm:text-lg font-semibold text-gray-800 dark:text-white truncate">{action.supplier}</h3>
        <Tag color="blue">{action.status}</Tag>
      </div>
      <p className="text-sm text-gray-600 dark:text-gray-300 mb-1 truncate"><strong>Material:</strong> {action.materialDescription}</p>
      <p className="text-sm text-gray-600 dark:text-gray-300 mb-1"><strong>Status:</strong> {renderTag(action.status)}</p>
      <p className="text-sm text-gray-500 dark:text-gray-400 text-xs"><strong>CAPA No.:</strong> {action.capaNo}</p>
      <div className="mt-2 flex gap-2">
        <Tooltip title="View">
          <Button icon={<EyeOutlined />} onClick={() => setViewingAction(action)} size="small" className="text-blue-600" />
        </Tooltip>
        <Tooltip title="Edit">
          <Button icon={<EditOutlined />} onClick={() => openEditModal(action)} size="small" type="primary" className="bg-blue-600 hover:bg-blue-700" />
        </Tooltip>
        <Tooltip title="Delete">
          <Button icon={<DeleteOutlined />} onClick={() => handleDelete(action.id)} size="small" danger className="text-red-600" />
        </Tooltip>
      </div>
    </div>
  );

  const total = actions.length;
  const pending = actions.filter((a) => a.status === "Pending").length;
  const inProgress = actions.filter((a) => a.status === "InProgress").length;
  const completed = actions.filter((a) => a.status === "Completed").length;

  // Sample trend data for widgets
  const trendData = [
    { value: 10 },
    { value: 15 },
    { value: 12 },
    { value: 18 },
    { value: 20 },
  ];

  return (
    <>
      <div className="px-2 sm:px-4 md:px-6 lg:px-8 py-4">
        <h2 className="text-xl sm:text-2xl md:text-3xl font-bold text-gray-800 dark:text-white mb-2">
          Supplier Actions <span className="text-gray-500 text-base sm:text-lg font-medium">({actions.length})</span>
        </h2>
        <p className="text-gray-600 dark:text-gray-300 text-sm sm:text-base mb-4">
          Manage supplier corrective and preventive actions.
        </p>
        <Breadcrumb
          className="mb-4 sm:mb-6 mt-2 text-sm text-gray-500 dark:text-gray-400"
          separator=">"
          items={[
            {
              title: (
                <Tooltip title="Dashboard">
                  <HomeOutlined />
                </Tooltip>
              ),
              href: "/dashboard",
            },
            {
              title: "Master Data",
            },
            {
              title: "Supplier Actions",
            },
          ]}
        />

        <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-4 mt-3 mb-6">
          <WidgetCard
            title="Total Actions"
            value={total}
            icon={<AppstoreOutlined />}
            color="green"
            percentage={12}
            description="All corrective and preventive actions"
            trendData={trendData}
            className="shadow-[0_0_10px_rgba(0,128,255,0.2)] dark:shadow-[0_0_10px_rgba(0,128,255,0.3)]"
          />
          <WidgetCard
            title="Pending Actions"
            value={pending}
            icon={<WarningOutlined />}
            color="orange"
            percentage={5}
            description="Actions awaiting action"
            trendData={trendData}
            className="shadow-[0_0_10px_rgba(255,165,0,0.2)] dark:shadow-[0_0_10px_rgba(255,165,0,0.3)]"
          />
          <WidgetCard
            title="In Progress"
            value={inProgress}
            icon={<ClockCircleOutlined />}
            color="blue"
            percentage={8}
            description="Actions currently in progress"
            trendData={trendData}
            className="shadow-[0_0_10px_rgba(0,0,255,0.2)] dark:shadow-[0_0_10px_rgba(0,0,255,0.3)]"
          />
          <WidgetCard
            title="Completed Actions"
            value={completed}
            icon={<CheckCircleTwoTone twoToneColor="#52c41a" />}
            color="green"
            percentage={3}
            description="Successfully completed actions"
            trendData={trendData}
            className="shadow-[0_0_10px_rgba(0,255,0,0.2)] dark:shadow-[0_0_10px_rgba(0,255,0,0.3)]"
          />
        </div>

        <div className="mt-4">
          <Card title="Supplier Action List" className="p-0 mb-6 shadow-sm">
            <div className="p-4 flex flex-col gap-4">
              <div className="flex flex-col sm:flex-row justify-between items-stretch sm:items-center gap-4">
                <Input
                  prefix={<SearchOutlined />}
                  allowClear
                  placeholder="Search Supplier or Material"
                  value={searchTerm}
                  onChange={(e) => setSearchTerm(e.target.value)}
                  className="w-full sm:w-64 border-gray-300 dark:border-gray-600 rounded-md shadow-sm"
                />
                <Dropdown
                  className="w-full sm:w-auto border-gray-300 dark:border-gray-600"
                  menu={{
                    items: [
                      { label: "Export CSV", key: "csv" },
                      { label: "Export PDF", key: "pdf" },
                    ],
                  }}
                >
                  <Button icon={<DownloadOutlined />}>Export</Button>
                </Dropdown>
                <Button
                  type="primary"
                  icon={<PlusOutlined />}
                  onClick={openAddModal}
                  className="bg-blue-600 hover:bg-blue-700 w-full sm:w-auto"
                >
                  Add Action
                </Button>
              </div>
              <div className="flex flex-col gap-4 lg:flex-row lg:justify-between lg:items-center">
                <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-4 w-full lg:w-auto">
                  <Select
                    placeholder="Status"
                    className="w-full max-w-[160px]"
                    onChange={(value) =>
                      setAdvancedFilters((prev) => ({ ...prev, status: value }))
                    }
                    allowClear
                    options={statusOptions.map((v) => ({ label: v, value: v }))}
                  />
                  <Select
                    placeholder="Unit Name"
                    className="w-full max-w-[160px]"
                    onChange={(value) =>
                      setAdvancedFilters((prev) => ({ ...prev, unitName: value }))
                    }
                    allowClear
                    options={Array.from(new Set(actions.map((a) => a.unitName))).map((value) => ({
                      label: value,
                      value: value,
                    }))}
                  />
                </div>
                <div className="flex flex-col sm:flex-row flex-wrap gap-4 justify-start lg:justify-end w-full lg:w-auto">
                  <ColumnVisibilityToggle
                    columns={columnKeys}
                    labels={labelsMap}
                    storageKey="supplier_action_table_visibility"
                    onChange={handleVisibleColumnChange}
                  />
                  <DatePicker.RangePicker
                    value={dateRange}
                    onChange={(dates) =>
                      setDateRange(dates as [dayjs.Dayjs, dayjs.Dayjs] | null)
                    }
                    className="w-full sm:w-auto border-gray-300 dark:border-gray-600"
                  />
                  <Button
                    onClick={handleBulkDelete}
                    disabled={selectedRows.length === 0}
                    danger
                    className="w-full sm:w-auto"
                  >
                    Bulk Delete
                  </Button>
                  <div className="hidden sm:block">
                    <Switch
                      checkedChildren={<AppstoreOutlined />}
                      unCheckedChildren={<TableOutlined />}
                      checked={isGridView}
                      onChange={setIsGridView}
                      className="w-full sm:w-auto"
                    />
                  </div>
                </div>
              </div>

              {(advancedFilters.status || advancedFilters.unitName) && (
                <div className="flex flex-wrap gap-2 mt-2">
                  {advancedFilters.status && (
                    <Tag
                      closable
                      onClose={() =>
                        setAdvancedFilters((prev) => ({
                          ...prev,
                          status: undefined,
                        }))
                      }
                    >
                      Status: {advancedFilters.status}
                    </Tag>
                  )}
                  {advancedFilters.unitName && (
                    <Tag
                      closable
                      onClose={() =>
                        setAdvancedFilters((prev) => ({
                          ...prev,
                          unitName: undefined,
                        }))
                      }
                    >
                      Unit: {advancedFilters.unitName}
                    </Tag>
                  )}
                </div>
              )}
            </div>

            {isGridView || isMobile ? (
              <div className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-4 p-4">
                {paginatedActions.map((action) => (
                  <ActionCard key={action.id} action={action} />
                ))}
              </div>
            ) : (
              <div className="overflow-auto">
                <DataTable<SupplierAction>
                  columns={filteredColumns}
                  dataSource={paginatedActions}
                  rowKey="id"
                  scrollX="max-content"
                  className="mt-4"
                />
              </div>
            )}
            <div className="flex flex-col sm:flex-row justify-between items-center px-4 py-2 text-sm text-gray-600 dark:text-gray-300">
              <div className="mb-2 sm:mb-0">
                Showing <strong>{(currentPage - 1) * pageSize + 1}</strong> to{" "}
                <strong>{Math.min(currentPage * pageSize, filteredActions.length)}</strong> of{" "}
                <strong>{filteredActions.length}</strong> actions
              </div>
              <Pagination
                current={currentPage}
                pageSize={pageSize}
                total={filteredActions.length}
                onChange={(page, size) => {
                  setCurrentPage(page);
                  setPageSize(size);
                }}
                showSizeChanger
                pageSizeOptions={["10", "30", "50", "100"]}
                size="small"
              />
            </div>
          </Card>
        </div>

        <motion.div
          initial={{ opacity: 0, scale: 0.95 }}
          animate={{ opacity: 1, scale: 1 }}
          exit={{ opacity: 0, scale: 0.95 }}
        >
          <Modal
            title={editingAction ? "Edit Supplier Action" : "Add Supplier Action"}
            open={isModalOpen}
            onOk={handleFormSubmit}
            onCancel={() => {
              setIsModalOpen(false);
              form.resetFields();
            }}
            destroyOnHidden
            className="w-full max-w-[90vw] sm:max-w-[600px] md:max-w-[800px] lg:max-w-[1000px] top-4"
          >
            <Form form={form} layout="vertical" className="px-2 sm:px-4">
              <Row gutter={[16, 16]}>
                <Col xs={24} sm={12}>
                  <Form.Item name="unitName" label="Unit Name" rules={[{ required: true, message: "Unit Name is required" }]}>
                    <Input />
                  </Form.Item>
                </Col>
                <Col xs={24} sm={12}>
                  <Form.Item name="supplier" label="Supplier" rules={[{ required: true, message: "Supplier is required" }]}>
                    <Input />
                  </Form.Item>
                </Col>
                <Col xs={24}>
                  <Form.Item name="materialDescription" label="Material Description" rules={[{ required: true, message: "Material Description is required" }]}>
                    <Input />
                  </Form.Item>
                </Col>
                <Col xs={24} sm={12}>
                  <Form.Item name="invoiceNo" label="Invoice No." rules={[{ required: true, message: "Invoice No. is required" }]}>
                    <Input />
                  </Form.Item>
                </Col>
                <Col xs={24} sm={12}>
                  <Form.Item name="supplierLotNo" label="Supplier Lot No." rules={[{ required: true, message: "Supplier Lot No. is required" }]}>
                    <Input />
                  </Form.Item>
                </Col>
                <Col xs={24} sm={12}>
                  <Form.Item name="dateIdentified" label="Date Identified" rules={[{ required: true, message: "Date Identified is required" }]}>
                    <DatePicker className="w-full" />
                  </Form.Item>
                </Col>
                <Col xs={24}>
                  <Form.Item name="nonConformanceDetails" label="Non Conformance Details" rules={[{ required: true, message: "Non Conformance Details are required" }]}>
                    <Input.TextArea rows={2} />
                  </Form.Item>
                </Col>
                <Col xs={24} sm={12}>
                  <Form.Item name="capaNo" label="CAPA No." rules={[{ required: true, message: "CAPA No. is required" }]}>
                    <Input />
                  </Form.Item>
                </Col>
                <Col xs={24} sm={12}>
                  <Form.Item name="status" label="Status" rules={[{ required: true, message: "Status is required" }]}>
                    <Select options={statusOptions.map((v) => ({ label: v, value: v }))} />
                  </Form.Item>
                </Col>
              </Row>
            </Form>
          </Modal>
        </motion.div>

        <Modal
          title="Supplier Action Details"
          open={!!viewingAction}
          onCancel={() => setViewingAction(null)}
          footer={null}
          className="w-full max-w-[90vw] sm:max-w-[500px] top-4"
        >
          {viewingAction && (
            <div className="space-y-2 text-sm sm:text-base">
              <p><strong>Unit Name:</strong> {viewingAction.unitName}</p>
              <p><strong>Supplier:</strong> <span className="text-green-600">{viewingAction.supplier}</span></p>
              <p><strong>Material:</strong> {viewingAction.materialDescription}</p>
              <p><strong>Invoice No.:</strong> {viewingAction.invoiceNo}</p>
              <p><strong>Supplier Lot No.:</strong> {viewingAction.supplierLotNo}</p>
              <p><strong>Date Identified:</strong> {dayjs(viewingAction.dateIdentified).format("YYYY-MM-DD")}</p>
              <p><strong>Non Conformance:</strong> {viewingAction.nonConformanceDetails}</p>
              <p><strong>CAPA No.:</strong> {viewingAction.capaNo}</p>
              <p><strong>Status:</strong> {viewingAction.status}</p>
              <p><strong>Created:</strong> {dayjs(viewingAction.created_at).format("YYYY-MM-DD")}</p>
            </div>
          )}
        </Modal>
      </div>
    </>
  );
}