"use client";

import React, { useCallback, useEffect, useState } from "react";
import {
  Button,
  Form,
  Input,
  Modal,
  Select,
  Space,
  Row,
  Col,
  Tooltip,
  Tag,
  Breadcrumb,
  Card,
  DatePicker,
  Switch,
  Pagination,
  Dropdown,
} from "antd";
import {
  PlusOutlined,
  EditOutlined,
  DeleteOutlined,
  HomeOutlined,
  TableOutlined,
  AppstoreOutlined,
  CheckCircleTwoTone,
  StopOutlined,
  WarningOutlined,
  EyeOutlined,
  SearchOutlined,
  DownloadOutlined,
  InfoCircleOutlined,
  ExclamationCircleOutlined,
  CheckCircleOutlined,
  PauseCircleOutlined,
  FireOutlined,
  MinusCircleOutlined,
  CloseCircleOutlined,
} from "@ant-design/icons";
import type { ColumnsType } from "antd/es/table";
import { Product } from "../types/Product";
import DataTable from "../components/common/DataTable";
import dayjs from "dayjs";
import { FilterValue } from "antd/es/table/interface";
import { toast } from "sonner";
import { motion } from "framer-motion";
import WidgetCard from "../components/common/WidgetCard";
import ColumnVisibilityToggle from "../components/common/ManageColumn";
import { useIsMobile } from "@/data/useIsMobile";
import { useRouter } from "next/navigation";

const { RangePicker } = DatePicker;
const classificationOptions = ["Class I", "Class II", "Class III"] as const;
const riskLevelOptions = ["Low", "Medium", "High"] as const;
const statusOptions = ["Active", "Inactive", "Discontinued"] as const;

export default function ProductsPage() {
  const [products, setProducts] = useState<Product[]>([]);
  const [isModalOpen, setIsModalOpen] = useState(false);
  const [editingProduct, setEditingProduct] = useState<Product | null>(null);
  const [form] = Form.useForm();
  const [searchTerm, setSearchTerm] = useState("");
  const [dateRange, setDateRange] = useState<[dayjs.Dayjs, dayjs.Dayjs] | null>(null);
  const [filters, setFilters] = useState<Record<string, FilterValue | null>>({});
  const [isGridView, setIsGridView] = useState(false);
  const [selectedRows, setSelectedRows] = useState<string[]>([]);
  const [advancedFilters, setAdvancedFilters] = useState({
    classification: undefined,
    riskLevel: undefined,
    status: undefined,
  });
  const [currentPage, setCurrentPage] = useState(1);
  const [pageSize, setPageSize] = useState(10);
  const [savedFilters, setSavedFilters] = useState<{ name: string; filters: typeof advancedFilters }[]>([]);
  const [viewingProduct, setViewingProduct] = useState<Product | null>(null);
  const [loading, setLoading] = useState(false);
  const isMobile = useIsMobile();
    const router = useRouter();

  // const handleView = (product: Product) => {
  //   setViewingProduct(product);
  // };
   const handleView = (product: Product) => {
    router.push(`/products/${product.id}`);
  };
    useEffect(() => {
    localStorage.setItem("products", JSON.stringify(products));
  }, [products]);

  const openAddModal = () => {
    setEditingProduct(null);
    form.resetFields();
    setIsModalOpen(true);
  };

  const openEditModal = (product: Product) => {
    setEditingProduct(product);
    form.setFieldsValue(product);
    setIsModalOpen(true);
  };

  const handleDelete = (id: string) => {
    Modal.confirm({
      title: "Are you sure you want to delete this product?",
      onOk: () => {
        setProducts((prev) => prev.filter((p) => p.id !== id));
        toast.success("Product deleted successfully");
      },
      onCancel: () => { },
    });
  };

  const handleBulkDelete = () => {
    Modal.confirm({
      title: `Are you sure you want to delete ${selectedRows.length} products?`,
      onOk: () => {
        setProducts((prev) => prev.filter((p) => !selectedRows.includes(p.id.toString())));
        setSelectedRows([]);
        toast.success("Products deleted successfully");
      },
    });
  };

  const handleFormSubmit = () => {
    form.validateFields().then((values: Product) => {
      const capitalized = {
        ...values,
        productName: values.productName.charAt(0).toUpperCase() + values.productName.slice(1),
      };

      if (editingProduct) {
        setProducts((prev) =>
          prev.map((p) => (p.id === editingProduct.id ? { ...editingProduct, ...capitalized } : p))
        );
        toast.success("Product updated successfully");
      } else {
        const newProduct: Product = {
          ...capitalized,
          created_at: dayjs().toISOString(),
          id: Date.now().toString(),
        };
        setProducts((prev) => [...prev, newProduct]);
        toast.success("Product added successfully");
      }

      setIsModalOpen(false);
      form.resetFields();
    });
  };

  const renderTag = (value: string) => {
    const colorMap: Record<string, string> = {
      "Class I": "blue",
      "Class II": "volcano",
      "Class III": "purple",
      Low: "green",
      Medium: "orange",
      High: "red",
      Active: "green",
      Inactive: "gold",
      Discontinued: "red",
    };

    const iconMap: Record<string, React.ReactNode> = {
      "Class I": <InfoCircleOutlined />,
      "Class II": <WarningOutlined />,
      "Class III": <ExclamationCircleOutlined />,
      Low: <CheckCircleOutlined />,
      Medium: <PauseCircleOutlined />,
      High: <FireOutlined />,
      Active: <CheckCircleOutlined />,
      Inactive: <MinusCircleOutlined />,
      Discontinued: <CloseCircleOutlined />,
    };

    return (
      <Tag icon={iconMap[value]} color={colorMap[value] || "default"}>
        {value}
      </Tag>
    );
  };

  const filteredProducts = products.filter((product) => {
    const matchesSearch =
      product.productName.toLowerCase().includes(searchTerm.toLowerCase()) ||
      product.productCode.toLowerCase().includes(searchTerm.toLowerCase());
    const matchesDate =
      !dateRange ||
      (product.created_at &&
        (!dateRange[0] || dayjs(product.created_at).isSame(dateRange[0]) || dayjs(product.created_at).isAfter(dateRange[0])) &&
        (!dateRange[1] || dayjs(product.created_at).isSame(dateRange[1]) || dayjs(product.created_at).isBefore(dateRange[1])));
    const matchesAdvanced =
      (!advancedFilters.classification || product.classification === advancedFilters.classification) &&
      (!advancedFilters.riskLevel || product.riskLevel === advancedFilters.riskLevel) &&
      (!advancedFilters.status || product.status === advancedFilters.status);
    return matchesSearch && matchesDate && matchesAdvanced;
  });

  const paginatedProducts = filteredProducts.slice(
    (currentPage - 1) * pageSize,
    currentPage * pageSize
  );

  const columns: ColumnsType<Product> = [
    {
      title: (
        <input
          type="checkbox"
          onChange={(e) => {
            const checked = e.target.checked;
            setSelectedRows(checked ? filteredProducts.map((p) => p.id.toString()) : []);
          }}
          className="w-4 h-4"
        />
      ),
      key: "selection",
      render: (_, record) => (
        <input
          type="checkbox"
          checked={selectedRows.includes(record.id.toString())}
          onChange={(e) => {
            setSelectedRows((prev) =>
              e.target.checked ? [...prev, record.id.toString()] : prev.filter((id) => id !== record.id.toString())
            );
          }}
          className="w-4 h-4"
        />
      ),
      width: 44
    },
    {
      title: "Code",
      dataIndex: "productCode",
      key: "productCode",
      sorter: (a, b) => a.productCode.localeCompare(b.productCode),

    },
    {
      title: "Name",
      dataIndex: "productName",
      key: "productName",
      sorter: (a, b) => a.productName.localeCompare(b.productName),
      render: (text) => <span className="text-green-600 font-medium">{text}</span>,
      filterSearch: true,
      onFilter: (value, record) =>
        record.productName.toLowerCase().includes((value as string).toLowerCase()),
      filters: products.map((p) => ({ text: p.productName, value: p.productName })),

    },
    {
      title: "Category",
      dataIndex: "category",
      key: "category",

    },
    {
      title: "Classification",
      dataIndex: "classification",
      key: "classification",
      render: renderTag,
      filters: Array.from(new Set(products.map((p) => p.classification))).map((value) => ({
        text: value,
        value: value,
      })),
      onFilter: (value, record) => record.classification === value,

    },
    {
      title: "Risk Level",
      dataIndex: "riskLevel",
      key: "riskLevel",
      render: renderTag,
      filters: Array.from(new Set(products.map((p) => p.riskLevel))).map((value) => ({
        text: value,
        value: value,
      })),
      onFilter: (value, record) => record.riskLevel === value,

    },
    {
      title: "Status",
      dataIndex: "status",
      key: "status",
      render: renderTag,
      filters: Array.from(new Set(products.map((p) => p.status))).map((value) => ({
        text: value,
        value: value,
      })),
      onFilter: (value, record) => record.status === value,

    },
    {
      title: "Description",
      dataIndex: "description",
      key: "description",
      render: (text) => (
        <Tooltip title={text}>
          <span className="block truncate max-w-[100px] sm:max-w-[150px] md:max-w-[180px]">{text}</span>
        </Tooltip>
      ),

    },
    {
      title: "Specifications",
      dataIndex: "specifications",
      key: "specifications",
      render: (text) => (
        <Tooltip title={text}>
          <span className="block truncate max-w-[100px] sm:max-w-[150px] md:max-w-[180px]">{text}</span>
        </Tooltip>
      ),

    },
    {
      title: "Actions",
      key: "actions",
      fixed: "right",
      render: (_, record) => (
        <Space size="small">
          <Tooltip title="View">
            <Button
              icon={<EyeOutlined />}
              onClick={() => handleView(record)}
              size="small"
              className="text-blue-600"
            />
          </Tooltip>
          <Tooltip title="Edit">
            <Button
              icon={<EditOutlined />}
              onClick={() => openEditModal(record)}
              size="small"
              type="primary"
              className="bg-blue-600 hover:bg-blue-700"
            />
          </Tooltip>
          <Tooltip title="Delete">
            <Button
              icon={<DeleteOutlined />}
              onClick={() => handleDelete(record.id.toString())}
              danger
              size="small"
              className="text-red-600"
            />
          </Tooltip>
        </Space>
      ),

    },
  ];

  const allColumns = [
    { key: "selection", label: "Select" },
    { key: "productCode", label: "Code" },
    { key: "productName", label: "Name" },
    { key: "category", label: "Category" },
    { key: "classification", label: "Classification" },
    { key: "riskLevel", label: "Risk Level" },
    { key: "created_at", label: "Created_at" },
    { key: "updated_at", label: "Updated_at" },
    { key: "status", label: "Status" },
    { key: "description", label: "Description" },
    { key: "specifications", label: "Specifications" },
    { key: "actions", label: "Action" },
  ];

  const columnKeys = allColumns.map((c) => c.key);
  const labelsMap = Object.fromEntries(allColumns.map((c) => [c.key, c.label]));
  const [visibleColumnKeys, setVisibleColumnKeys] = useState<string[]>(columnKeys);

  const handleVisibleColumnChange = useCallback((cols: string[]) => {
    setVisibleColumnKeys(cols);
  }, []);

  const filteredColumns = columns.filter(
    (col) => typeof col.key === "string" && visibleColumnKeys.includes(col.key)
  );

  const ProductCard = ({ product }: { product: Product }) => (
    <div className="border rounded-xl p-4 shadow-sm hover:shadow-md bg-white dark:bg-gray-900 transition-all w-full">
      <div className="flex flex-col sm:flex-row justify-between items-start sm:items-center mb-2 gap-2">
        <h3 className="text-base sm:text-lg font-semibold text-gray-800 dark:text-white truncate">{product.productName}</h3>
        <Tag color="blue">{product.category}</Tag>
      </div>
      <p className="text-sm text-gray-600 dark:text-gray-300 mb-1 truncate"><strong>Code:</strong> {product.productCode}</p>
      <p className="text-sm text-gray-600 dark:text-gray-300 mb-1"><strong>Status:</strong> {renderTag(product.status)}</p>
      <p className="text-sm text-gray-600 dark:text-gray-300 mb-1"><strong>Classification:</strong> {renderTag(product.classification)}</p>
      <p className="text-sm text-gray-500 dark:text-gray-400 text-xs"><strong>Created:</strong> {dayjs(product.created_at).format("YYYY-MM-DD")}</p>
      <div className="mt-2 flex gap-2">
        <Tooltip title="View">
          <Button icon={<EyeOutlined />} onClick={() => handleView(product)} size="small" className="text-blue-600" />
        </Tooltip>
        <Tooltip title="Edit">
          <Button icon={<EditOutlined />} onClick={() => openEditModal(product)} size="small" type="primary" className="bg-blue-600 hover:bg-blue-700" />
        </Tooltip>
        <Tooltip title="Delete">
          <Button icon={<DeleteOutlined />} onClick={() => handleDelete(product.id.toString())} size="small" danger className="text-red-600" />
        </Tooltip>
      </div>
    </div>
  );

  const total = products.length;
  const active = products.filter((p) => p.status === "Active").length;
  const discontinued = products.filter((p) => p.status === "Discontinued").length;
  const highRisk = products.filter((p) => p.riskLevel === "High").length;

  return (
    <>

    <div className="px-2 sm:px-4 md:px-6 lg:px-8 py-4">
      <h2 className="text-xl sm:text-2xl md:text-3xl font-bold text-gray-800 dark:text-white mb-2">
        Products <span className="text-gray-500 text-base sm:text-lg font-medium">({products.length})</span>
      </h2>
      <p className="text-gray-600 dark:text-gray-300 text-sm sm:text-base mb-4">
        Manage product master data used throughout the QMS system.
      </p>
      <Breadcrumb
        className="mb-4 sm:mb-6 mt-2 text-sm text-gray-500 dark:text-gray-400"
        separator=">"
        items={[
          {
            title: (
              <Tooltip title="Dashboard">
                <HomeOutlined />
              </Tooltip>
            ),
            href: "/dashboard",
          },
          {
            title: "Master Data",
          },
          {
            title: "Products",
          },
        ]}
      />


      <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-4 mt-3 mb-6">
        <WidgetCard
          title="Total Products"
          value={total}
          icon={<AppstoreOutlined />}
          color="green"
          percentage={12}
          description="All registered medical products"
        />
        <WidgetCard
          title="Active"
          value={active}
          icon={<CheckCircleTwoTone twoToneColor="#52c41a" />}
          color="blue"
          percentage={8}
          description="Currently active in production"
        />
        <WidgetCard
          title="Discontinued"
          value={discontinued}
          icon={<StopOutlined />}
          color="red"
          percentage={-3}
          description="Products no longer in use"
        />
        <WidgetCard
          title="High Risk"
          value={highRisk}
          icon={<WarningOutlined />}
          color="orange"
          percentage={5}
          description="Class III or high-risk items"
        />
      </div>

      <div className="mt-4">
        <Card title="Product List" className="p-0 mb-6 shadow-sm">
          <div className="p-4 flex flex-col gap-4">
            <div className="flex flex-col sm:flex-row justify-between items-stretch sm:items-center gap-4">
              <Input
                prefix={<SearchOutlined />}
                allowClear
                placeholder="Search Product by name or code"
                value={searchTerm}
                onChange={(e) => setSearchTerm(e.target.value)}
                className="w-full sm:w-64 border-gray-300 dark:border-gray-600 rounded-md shadow-sm"
              />
              <Dropdown
                className="w-full sm:w-auto border-gray-300 dark:border-gray-600"
                menu={{
                  items: [
                    { label: "Export CSV", key: "csv" },
                    { label: "Export PDF", key: "pdf" },
                  ],
                  // onClick: ({ key }) => handleExport(key),
                }}
              >
                <Button icon={<DownloadOutlined />}>Export</Button>
              </Dropdown>
              <Button
                type="primary"
                icon={<PlusOutlined />}
                onClick={openAddModal}
                className="bg-blue-600 hover:bg-blue-700 w-full sm:w-auto"
              >
                Add Product
              </Button>
            </div>
            <div className="flex flex-col gap-4 lg:flex-row lg:justify-between lg:items-center">
              <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-4 w-full lg:w-auto">
                <Select
                  placeholder="Classification"
                  className="w-full max-w-[160px]"
                  onChange={(value) =>
                    setAdvancedFilters((prev) => ({ ...prev, classification: value }))
                  }
                  allowClear
                  options={classificationOptions.map((v) => ({ label: v, value: v }))}
                />
                <Select
                  placeholder="Risk Level"
                  className="w-full max-w-[160px]"
                  onChange={(value) =>
                    setAdvancedFilters((prev) => ({ ...prev, riskLevel: value }))
                  }
                  allowClear
                  options={riskLevelOptions.map((v) => ({ label: v, value: v }))}
                />
                <Select
                  placeholder="Status"
                  className="w-full max-w-[160px]"
                  onChange={(value) =>
                    setAdvancedFilters((prev) => ({ ...prev, status: value }))
                  }
                  allowClear
                  options={statusOptions.map((v) => ({ label: v, value: v }))}
                />
              </div>
              <div className="flex flex-col sm:flex-row flex-wrap gap-4 justify-start lg:justify-end w-full lg:w-auto">
                <ColumnVisibilityToggle
                  columns={columnKeys}
                  labels={labelsMap}
                  storageKey="product_table_visibility"
                  onChange={handleVisibleColumnChange}
                />
                <RangePicker
                  value={dateRange}
                  onChange={(dates) =>
                    setDateRange(dates as [dayjs.Dayjs, dayjs.Dayjs] | null)
                  }
                  className="w-full sm:w-auto border-gray-300 dark:border-gray-600"
                />
                <Button
                  onClick={handleBulkDelete}
                  disabled={selectedRows.length === 0}
                  danger
                  className="w-full sm:w-auto"
                >
                  Bulk Delete
                </Button>
                <div className="hidden sm:block">
                  <Switch
                    checkedChildren={<AppstoreOutlined />}
                    unCheckedChildren={<TableOutlined />}
                    checked={isGridView}
                    onChange={setIsGridView}
                    className="w-full sm:w-auto"
                  />
                </div>
              </div>
            </div>

            {(advancedFilters.classification ||
              advancedFilters.riskLevel ||
              advancedFilters.status) && (
                <div className="flex flex-wrap gap-2 mt-2">
                  {advancedFilters.classification && (
                    <Tag
                      closable
                      onClose={() =>
                        setAdvancedFilters((prev) => ({
                          ...prev,
                          classification: undefined,
                        }))
                      }
                    >
                      Classification: {advancedFilters.classification}
                    </Tag>
                  )}
                  {advancedFilters.riskLevel && (
                    <Tag
                      closable
                      onClose={() =>
                        setAdvancedFilters((prev) => ({
                          ...prev,
                          riskLevel: undefined,
                        }))
                      }
                    >
                      Risk Level: {advancedFilters.riskLevel}
                    </Tag>
                  )}
                  {advancedFilters.status && (
                    <Tag
                      closable
                      onClose={() =>
                        setAdvancedFilters((prev) => ({
                          ...prev,
                          status: undefined,
                        }))
                      }
                    >
                      Status: {advancedFilters.status}
                    </Tag>
                  )}
                </div>
              )}
          </div>

          {isGridView || isMobile ? (
            <div className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-4 p-4">
              {paginatedProducts.map((product) => (
                <ProductCard key={product.id} product={product} />
              ))}

            </div>
          ) : (
            <div className=" overflow-auto">
              <DataTable<Product>
                columns={filteredColumns}
                dataSource={paginatedProducts}
                rowKey="id"
                scrollX="max-content"
                className=" mt-4"
              //        pagination={{
              //   current: currentPage,
              //   pageSize,
              //   total: filteredProducts.length,
              //   onChange: (page, size) => {
              //     setCurrentPage(page);
              //     setPageSize(size);
              //   },
              //   showSizeChanger: true,
              //   pageSizeOptions: ['10', '30', '50', '100'],
              //   responsive: true,
              // }}

              />
            </div>
          )}
          <div className="flex flex-col sm:flex-row justify-between items-center px-4 py-2 text-sm text-gray-600 dark:text-gray-300">
            <div className="mb-2 sm:mb-0">
              Showing{" "}
              <strong>
                {(currentPage - 1) * pageSize + 1}
              </strong>{" "}
              to{" "}
              <strong>
                {Math.min(currentPage * pageSize, filteredProducts.length)}
              </strong>{" "}
              of{" "}
              <strong>{filteredProducts.length}</strong> products
            </div>

            <Pagination
              current={currentPage}
              pageSize={pageSize}
              total={filteredProducts.length}
              onChange={(page, size) => {
                setCurrentPage(page);
                setPageSize(size);
              }}
              showSizeChanger
              pageSizeOptions={['10', '30', '50', '100']}
              size="small"
            />
          </div>

        </Card>
      </div>

      <motion.div initial={{ opacity: 0, scale: 0.95 }} animate={{ opacity: 1, scale: 1 }} exit={{ opacity: 0, scale: 0.95 }}>
        <Modal
          title={editingProduct ? "Edit Product" : "Add Product"}
          open={isModalOpen}
          onOk={handleFormSubmit}
          onCancel={() => {
            setIsModalOpen(false);
            form.resetFields();
          }}
          destroyOnHidden
          className="w-full max-w-[90vw] sm:max-w-[600px] md:max-w-[800px] lg:max-w-[1000px] top-4"
        >
          <Form form={form} layout="vertical" className="px-2 sm:px-4">
            <Row gutter={[16, 16]}>
              <Col xs={24} sm={12}>
                <Form.Item
                  name="productCode"
                  label="Product Code"
                  rules={[{ required: true, message: "Product code is required" }]}
                >
                  <Input />
                </Form.Item>
              </Col>
              <Col xs={24} sm={12}>
                <Form.Item
                  name="productName"
                  label="Product Name"
                  rules={[{ required: true, message: "Product name is required" }]}
                >
                  <Input />
                </Form.Item>
              </Col>
              <Col xs={24}>
                <Form.Item name="description" label="Description">
                  <Input.TextArea rows={3} />
                </Form.Item>
              </Col>
              <Col xs={24} sm={12}>
                <Form.Item name="category" label="Category">
                  <Input />
                </Form.Item>
              </Col>
              <Col xs={24} sm={12}>
                <Form.Item
                  name="classification"
                  label="Classification"
                  rules={[{ required: true, message: "Classification is required" }]}
                >
                  <Select options={classificationOptions.map((v) => ({ label: v, value: v }))} />
                </Form.Item>
              </Col>
              <Col xs={24} sm={12}>
                <Form.Item
                  name="riskLevel"
                  label="Risk Level"
                  rules={[{ required: true, message: "Risk level is required" }]}
                >
                  <Select options={riskLevelOptions.map((v) => ({ label: v, value: v }))} />
                </Form.Item>
              </Col>
              <Col xs={24} sm={12}>
                <Form.Item name="specifications" label="Specifications">
                  <Input.TextArea rows={2} />
                </Form.Item>
              </Col>
              <Col xs={24} sm={12}>
                <Form.Item
                  name="status"
                  label="Status"
                  rules={[{ required: true, message: "Status is required" }]}
                >
                  <Select options={statusOptions.map((v) => ({ label: v, value: v }))} />
                </Form.Item>
              </Col>
            </Row>
          </Form>
        </Modal>
      </motion.div>

      <Modal
        title="Product Details"
        open={!!viewingProduct}
        onCancel={() => setViewingProduct(null)}
        footer={null}
        className="w-full max-w-[90vw] sm:max-w-[500px] top-4"
      >
        {viewingProduct && (
          <div className="space-y-2 text-sm sm:text-base">
            <p><strong>Code:</strong> {viewingProduct.productCode}</p>
            <p><strong>Name:</strong> <span className="text-green-600">{viewingProduct.productName}</span></p>
            <p><strong>Category:</strong> {viewingProduct.category}</p>
            <p><strong>Description:</strong> {viewingProduct.description}</p>
            <p><strong>Specifications:</strong> {viewingProduct.specifications}</p>
            <p><strong>Classification:</strong> {viewingProduct.classification}</p>
            <p><strong>Risk Level:</strong> {viewingProduct.riskLevel}</p>
            <p><strong>Status:</strong> {viewingProduct.status}</p>
            <p><strong>Created:</strong> {dayjs(viewingProduct.created_at).format("YYYY-MM-DD")}</p>
          </div>
        )}
      </Modal>
    </div>
    </>
  );
}