// src/app/layout.tsx
import type { Metadata, Viewport } from 'next';
import { Geist, Geist_Mono } from 'next/font/google';
import './globals.css';
//import { Toaster } from 'react-hot-toast'; // ✅ Import toast
import { AuthProvider } from '@/context/AuthContext';
import LayoutWrapper from './components/LayoutWrapper';
import { Toaster } from 'sonner';
import { Suspense } from 'react';
import Head from 'next/head';
import TopRouteLoader from './components/common/ClientTopRouteLoader';
import { SearchProvider } from '@/context/SearchContext';

const geistSans = Geist({
  variable: '--font-geist-sans',
  subsets: ['latin'],
  display: 'swap',
  weight: ['400', '500', '600'],
});

const geistMono = Geist_Mono({
  variable: '--font-geist-mono',
  subsets: ['latin'],
  display: 'swap',
  weight: ['400'],
});

export const metadata: Metadata = {
  metadataBase: new URL('https://yourdomain.com'), 
  title: 'QMS System',
  description: 'Quality Management System for your organization',
  openGraph: {
    title: 'QMS Dashboard',
    description: 'Quality Management System for your organization',
    url: 'https://yourdomain.com',
    siteName: 'QMS',
    images: [
      {
        url: '/og-image.png',
        width: 800,
        height: 600,
        alt: 'QMS Dashboard',
      },
    ],
    type: 'website',
  },
  icons: {
    icon: [
      { url: '/favicon.ico', type: 'image/x-icon' },
      { url: '/favicon-32x32.png', sizes: '32x32', type: 'image/png' },
      { url: '/favicon-16x16.png', sizes: '16x16', type: 'image/png' },
    ],
  },
};

export const viewport: Viewport = {
  width: 'device-width',
  initialScale: 1,
  maximumScale: 1,
  themeColor: '#ffffff',
};

export default function RootLayout({ children }: { children: React.ReactNode }) {
  return (
    <html lang="en" className={`${geistSans.variable} ${geistMono.variable}`}>
  <Head>
  <meta name="viewport" content="width=device-width, initial-scale=1" />
  <script
    dangerouslySetInnerHTML={{
      __html: `
        (function() {
          try {
            const dark = localStorage.getItem('dark-mode');
            const prefers = window.matchMedia('(prefers-color-scheme: dark)').matches;
            const shouldApplyDark = dark === 'true' || (dark === null && prefers);
            if (shouldApplyDark) {
              document.documentElement.classList.add('dark');
            } else {
              document.documentElement.classList.remove('dark');
            }
          } catch (e) {}
        })();
      `,
    }}
  />
</Head>
      <body className="min-h-screen bg-gray-50 text-gray-900 antialiased selection:bg-blue-100 selection:text-blue-900">
        {/* <Toaster position="top-right" /> ✅ Toast Component */}
            <TopRouteLoader />
        <Toaster
          position="bottom-right"
          expand={true}
          richColors={true}
          closeButton={true}
          duration={3000}
          toastOptions={{
            style: {
              fontSize: '0.875rem',
              fontWeight: 500,
            },
          }}
        />
        {/* <Suspense fallback={<div className="text-center p-6">Loading...</div>}> */}
        {/* <Suspense
  fallback={
    <div className="p-6 space-y-4">
      <div className="h-6 bg-gray-200 dark:bg-gray-700 rounded w-1/3 animate-pulse"></div>
      <div className="h-4 bg-gray-200 dark:bg-gray-700 rounded w-full animate-pulse"></div>
      <div className="h-4 bg-gray-200 dark:bg-gray-700 rounded w-5/6 animate-pulse"></div>
      <div className="h-4 bg-gray-200 dark:bg-gray-700 rounded w-2/3 animate-pulse"></div>
    </div>
  }
> */}
         <Suspense
  fallback={
    <div className="flex items-center justify-center h-screen bg-white dark:bg-gray-900">
      <div className="flex flex-col items-center space-y-4">
        <svg
          className="animate-spin h-8 w-8 text-primary"
          xmlns="http://www.w3.org/2000/svg"
          fill="none"
          viewBox="0 0 24 24"
        >
          <circle
            className="opacity-25"
            cx="12"
            cy="12"
            r="10"
            stroke="currentColor"
            strokeWidth="4"
          ></circle>
          <path
            className="opacity-75"
            fill="currentColor"
            d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4z"
          ></path>
        </svg>
        <span className="text-sm text-gray-500 dark:text-gray-400">Loading app...</span>
      </div>
    </div>
  }
>
  <AuthProvider>
    <SearchProvider>
      <LayoutWrapper>{children}</LayoutWrapper>
    </SearchProvider>
  </AuthProvider>
</Suspense>
      </body>
    </html>
  );
}
