"use client";

import React, { useCallback, useEffect, useMemo, useState } from "react";
import Button from "antd/es/button";
import "antd/es/button/style";
import Form from "antd/es/form";
import "antd/es/form/style";
import Input from "antd/es/input";
import "antd/es/input/style";
import Modal from "antd/es/modal";
import "antd/es/modal/style";
import Select from "antd/es/select";
import "antd/es/select/style";
import Space from "antd/es/space";
import "antd/es/space/style";
import Tooltip from "antd/es/tooltip";
import "antd/es/tooltip/style";
import Breadcrumb from "antd/es/breadcrumb";
import "antd/es/breadcrumb/style";
import Card from "antd/es/card";
import "antd/es/card/style";
import DatePicker from "antd/es/date-picker";
import "antd/es/date-picker/style";
import Switch from "antd/es/switch";
import "antd/es/switch/style";
import Table from "antd/es/table";
import "antd/es/table/style";
import Tag from "antd/es/tag";
import "antd/es/tag/style";
import Pagination from "antd/es/pagination";
import "antd/es/pagination/style";
import { Dropdown, InputNumber } from "antd";
// ✅ correct
import {
  PlusOutlined,
  EditOutlined,
  DeleteOutlined,
  EyeOutlined,
  HomeOutlined,
  TableOutlined,
  AppstoreOutlined,
  SearchOutlined,
  DownloadOutlined,
  CheckCircleTwoTone,
  StarOutlined,
  CheckCircleOutlined,
} from "@ant-design/icons";
import type { ColumnsType } from "antd/es/table";
import DataTable from "../components/common/DataTable";
import dynamic from "next/dynamic";
import { motion } from "framer-motion";
import { toast } from "sonner";
import dayjs from "@/data/utils/dayjs";
import { useIsMobile } from "@/data/useIsMobile";
import {
  createSupplierRating,
  getAllSupplierRatings,
  updateSupplierRating,
  deleteSupplierRating,
  SupplierRating,
} from "@/app/api/supplierrating/supplierrating";
import {
  getAllSupplierApprovals,
  SupplierApproval,
} from "@/app/api/supplierapproval/supplierapproval";
import { UserCheck, UserPlus } from "lucide-react";
import { FileLineChartIcon } from "lucide-react";
//import ConfirmDialog from "../components/common/ConfirmDialog";

const WidgetCard = dynamic(() => import("../components/common/WidgetCard"), {
  ssr: false,
});
WidgetCard.displayName = "WidgetCard";

const ColumnVisibilityToggle = dynamic(
  () => import("../components/common/ManageColumn"),
  { ssr: false }
);
ColumnVisibilityToggle.displayName = "ColumnVisibilityToggle";

const ConfirmDialog = dynamic(
  () => import("../components/common/ConfirmDialog"),
  { ssr: false }
);
ConfirmDialog.displayName = "ConfirmDialog";

export default function SupplierRatingPage() {
  const [form] = Form.useForm();
  const [ratings, setRatings] = useState<SupplierRating[]>([]);
  const [suppliers, setSuppliers] = useState<SupplierApproval[]>([]);
  const [isModalOpen, setIsModalOpen] = useState(false);
  const [editingRating, setEditingRating] = useState<SupplierRating | null>(
    null
  );
  const [viewingRating, setViewingRating] = useState<SupplierRating | null>(
    null
  );
  const [confirmDialog, setConfirmDialog] = useState({
    open: false,
    onConfirm: () => {},
    title: "",
    message: "",
  });
  const [searchTerm, setSearchTerm] = useState("");
  const [isGridView, setIsGridView] = useState(false);
  const [isSubmitting, setIsSubmitting] = useState(false);
  const [selectedRows, setSelectedRows] = useState<string[]>([]);
  const [advancedFilters, setAdvancedFilters] = useState({
    materialDescription: undefined,
  });
  const [dateRange, setDateRange] = useState<[dayjs.Dayjs, dayjs.Dayjs] | null>(
    null
  );
  const [currentPage, setCurrentPage] = useState(1);
  const [pageSize, setPageSize] = useState(10);
  const [loading, setLoading] = useState(false);
  const isMobile = useIsMobile();

  const fetchData = useCallback(async () => {
    setLoading(true);
    try {
      const ratingsData = await getAllSupplierRatings();
      setRatings(Array.isArray(ratingsData) ? ratingsData : []);
      const suppliersData = await getAllSupplierApprovals();
      setSuppliers(
        Array.isArray(suppliersData)
          ? suppliersData.filter((s) => s.status === "Approved")
          : []
      );
    } catch (error) {
      toast.error("Failed to fetch data");
    } finally {
      setLoading(false);
    }
  }, []); // ✅ stable reference

  useEffect(() => {
    fetchData();
  }, [fetchData]);

  const openAddModal = useCallback(() => {
    form.resetFields();
    setEditingRating(null);
    setIsModalOpen(true);
  }, [form]);

  const openEditModal = useCallback(
    (rating: SupplierRating) => {
      form.setFieldsValue({
        ...rating,
        ratingDate: rating.ratingDate ? dayjs(rating.ratingDate) : undefined,
      });
      setEditingRating(rating);
      setIsModalOpen(true);
    },
    [form]
  );

  const handleDelete = useCallback(
    (supplier_id: string) => {
      setConfirmDialog({
        open: true,
        title: "Delete Rating",
        message: "Are you sure you want to delete this rating?",
        onConfirm: async () => {
          try {
            await deleteSupplierRating(supplier_id);
            setRatings((prev) =>
              prev.filter((r) => r.supplier_id !== supplier_id)
            );
            toast.success("Rating deleted successfully");
          } catch {
            toast.error("Failed to delete rating");
          } finally {
            setConfirmDialog((prev) => ({ ...prev, open: false }));
          }
        },
      });
    },
    [setRatings, setConfirmDialog]
  );

  const handleBulkDelete = useCallback(() => {
    if (selectedRows.length === 0) return;
    setConfirmDialog({
      open: true,
      title: "Delete Multiple Ratings",
      message: `Are you sure you want to delete ${selectedRows.length} ratings?`,
      onConfirm: async () => {
        try {
          await Promise.all(
            selectedRows.map((supplier_id) => deleteSupplierRating(supplier_id))
          );
          setRatings((prev) =>
            prev.filter((r) => !selectedRows.includes(r.supplier_id!))
          );
          setSelectedRows([]);
          toast.success("Ratings deleted successfully");
        } catch {
          toast.error("Failed to delete ratings");
        } finally {
          setConfirmDialog((prev) => ({ ...prev, open: false }));
        }
      },
    });
  }, [selectedRows, setRatings, setSelectedRows, setConfirmDialog]);

  const computeDerivedValues = useCallback(
    (
      entry: Omit<
        SupplierRating,
        "acceptedPercent" | "overallScore" | "created_at"
      >
    ): SupplierRating => {
      const acceptedPercent = entry.lotReceived
        ? (entry.lotAccepted / entry.lotReceived) * 100
        : 0;
      const overallScore =
        (entry.qualityRating || 0) +
        (entry.deliveryRating || 0) +
        (entry.serviceRating || 0) +
        (entry.priceRating || 0);
      return {
        ...entry,
        acceptedPercent: +acceptedPercent.toFixed(2),
        overallScore,
        ratingDate: entry.ratingDate
          ? dayjs(entry.ratingDate).format("YYYY-MM-DD")
          : dayjs().format("YYYY-MM-DD"),
      };
    },
    []
  );

  const handleFormSubmit = useCallback(async () => {
    setIsSubmitting(true);
    try {
      const values = await form.validateFields();

      // Merge form values with existing record (to preserve fields not in the form)
      const payload = computeDerivedValues({
        ...(editingRating ?? {}),
        ...values,
      } as Omit<SupplierRating, "acceptedPercent" | "overallScore" | "created_at">);

      if (editingRating?.supplier_id) {
        // ✅ UPDATE
        const updated = await updateSupplierRating(
          editingRating.supplier_id,
          payload
        );
        setRatings((prev) =>
          (prev ?? []).map((r) =>
            r.supplier_id === editingRating.supplier_id
              ? { ...r, ...updated }
              : r
          )
        );
        toast.success("Rating updated successfully");
      } else {
        // ✅ CREATE
        const created = await createSupplierRating(payload);
        setRatings((prev) => [...(prev ?? []), created]);
        toast.success("Rating added successfully");
      }

      setIsModalOpen(false);
      form.resetFields();
      fetchData();
    } catch (err) {
      console.error(err);
      toast.error("Failed to save rating");
    } finally {
      setIsSubmitting(false);
    }
  }, [form, computeDerivedValues, fetchData, editingRating]);

  const filteredRatings = useMemo(
    () =>
      (ratings ?? []).filter((rating) => {
        const supplier = suppliers.find(
          (s) => s.supplier_id === rating.supplier_id
        );
        const matchesSearch =
          (supplier?.supplier_name ?? "")
            .toLowerCase()
            .includes(searchTerm.toLowerCase()) ||
          (rating.materialDescription ?? "")
            .toLowerCase()
            .includes(searchTerm.toLowerCase()) ||
          (rating.rating_By ?? "")
            .toLowerCase()
            .includes(searchTerm.toLowerCase());
        const matchesDate =
          !dateRange ||
          (rating.created_at &&
            (!dateRange[0] ||
              dayjs(rating.created_at).isSame(dateRange[0]) ||
              dayjs(rating.created_at).isAfter(dateRange[0])) &&
            (!dateRange[1] ||
              dayjs(rating.created_at).isSame(dateRange[1]) ||
              dayjs(rating.created_at).isBefore(dateRange[1])));
        const matchesAdvanced =
          !advancedFilters.materialDescription ||
          rating.materialDescription === advancedFilters.materialDescription;
        return matchesSearch && matchesDate && matchesAdvanced;
      }),
    [ratings, searchTerm, dateRange, advancedFilters, suppliers]
  );

  const paginatedRatings = useMemo(
    () =>
      filteredRatings.slice(
        (currentPage - 1) * pageSize,
        currentPage * pageSize
      ),
    [filteredRatings, currentPage, pageSize]
  );

  const columns = useMemo<ColumnsType<SupplierRating>>(
    () => [
      {
        title: (
          <input
            type="checkbox"
            onChange={(e) =>
              setSelectedRows(
                e.target.checked
                  ? filteredRatings.map((r) => r.supplier_id!)
                  : []
              )
            }
            className="w-4 h-4"
          />
        ),
        key: "selection",
        render: (_, record) => (
          <input
            type="checkbox"
            checked={selectedRows.includes(record.supplier_id!)}
            onChange={(e) =>
              setSelectedRows((prev) =>
                e.target.checked
                  ? [...prev, record.supplier_id!]
                  : prev.filter((id) => id !== record.supplier_id)
              )
            }
            className="w-4 h-4"
          />
        ),
        width: 44,
      },
      {
        title: "Supplier",
        dataIndex: "supplier_id",
        key: "supplier_id",
        sorter: (a, b) => a.supplier_id.localeCompare(b.supplier_id),
        render: (text) => {
          const supplier = suppliers.find((s) => s.supplier_id === text);
          return (
            <span className="text-green-600 font-medium">
              {supplier?.supplier_name || text}
            </span>
          );
        },
        filterSearch: true,
        onFilter: (value, record) =>
          record.supplier_id
            .toLowerCase()
            .includes((value as string).toLowerCase()),
      },
      {
        title: "Material",
        dataIndex: "materialDescription",
        key: "materialDescription",
        sorter: (a, b) =>
          (a.materialDescription ?? "").localeCompare(
            b.materialDescription ?? ""
          ),
        filterSearch: true,
        onFilter: (value, record) =>
          (record.materialDescription ?? "")
            .toLowerCase()
            .includes((value as string).toLowerCase()),
      },
      {
        title: "Lot Received",
        dataIndex: "lotReceived",
        key: "lotReceived",
        sorter: (a, b) =>
          Number(a.lotReceived ?? 0) - Number(b.lotReceived ?? 0),
      },
      {
        title: "Lot Accepted",
        dataIndex: "lotAccepted",
        key: "lotAccepted",
        sorter: (a, b) =>
          Number(a.lotAccepted ?? 0) - Number(b.lotAccepted ?? 0),
      },
      {
        title: "Accepted %",
        dataIndex: "acceptedPercent",
        key: "acceptedPercent",
        render: (v) => `${v}%`,
        sorter: (a, b) =>
          Number(a.acceptedPercent ?? 0) - Number(b.acceptedPercent ?? 0),
      },
      {
        title: "Overall Score",
        dataIndex: "overallscore",
        key: "overallScore",
        sorter: (a, b) =>
          Number(a.overallScore ?? 0) - Number(b.overallScore ?? 0),
        render: (text) => (
          <span className="flex items-center gap-1">
            <StarOutlined className="text-yellow-500" /> {text}
          </span>
        ),
      },
      {
        title: "Rating Date",
        dataIndex: "ratingDate",
        key: "ratingDate",
        sorter: (a, b) =>
          new Date(a.ratingDate ?? 0).getTime() -
          new Date(b.ratingDate ?? 0).getTime(),
        render: (text) => dayjs(text).format("YYYY-MM-DD"),
      },
      {
        title: "On Time Delivery",
        dataIndex: "onTimeDelivery",
        key: "onTimeDelivery",
        sorter: (a, b) =>
          Number(a.onTimeDelivery ?? 0) - Number(b.onTimeDelivery ?? 0),
      },
      {
        title: "Short Delivery",
        dataIndex: "shortDelivery",
        key: "shortDelivery",
        sorter: (a, b) =>
          Number(a.shortDelivery ?? 0) - Number(b.shortDelivery ?? 0),
      },
      {
        title: "Rated By",
        dataIndex: "rating_By",
        key: "rating_By",
        sorter: (a, b) => (a.rating_By || "").localeCompare(b.rating_By || ""),
        render: (text) => <span>{text || "-"}</span>,
      },
      {
        title: "Risk Level",
        dataIndex: "risklevel",
        key: "risklevel",
        render: (text) => <Tag color="purple">{text || "-"}</Tag>,
      },
      {
        title: "Category",
        dataIndex: "category",
        key: "category",
        render: (text) => <Tag color="green">{text || "-"}</Tag>,
      },
      {
        title: "Rating Scale",
        dataIndex: "ratingScale",
        key: "ratingScale",
        render: (text) => <Tag color="green">{text || "-"}</Tag>,
      },
      {
        title: "Remark",
        dataIndex: "remark",
        key: "remark",
        render: (text) => text || "-",
      },
      {
        title: "Approved By",
        dataIndex: "approved_By",
        key: "approved_By",
        sorter: (a, b) =>
          (a.approved_By || "").localeCompare(b.approved_By || ""),
        render: (text) => <span>{text || "-"}</span>,
      },
      {
        title: "Created At",
        dataIndex: "created_at",
        key: "created_at",
        render: (value) =>
          value
            ? `${dayjs(value).format("DD-MM-YYYY hh:mm A")} (${dayjs(
                value
              ).fromNow()})`
            : "-",
      },
      {
        title: "Updated At",
        dataIndex: "updated_at",
        key: "updated_at",
        render: (value) =>
          value
            ? `${dayjs(value).format("DD-MM-YYYY hh:mm A")} (${dayjs(
                value
              ).fromNow()})`
            : "-",
      },
      {
        title: "Actions",
        key: "actions",
        fixed: "right",
        render: (_, record) => (
          <Space size="small">
            <Tooltip title="View">
              <Button
                icon={<EyeOutlined />}
                onClick={() => setViewingRating(record)}
                size="small"
                className="text-blue-600"
              />
            </Tooltip>
            <Tooltip title="Edit">
              <Button
                icon={<EditOutlined />}
                onClick={() => openEditModal(record)}
                size="small"
                type="primary"
                className="bg-blue-600 hover:bg-blue-700"
              />
            </Tooltip>
            <Tooltip title="Delete">
              <Button
                icon={<DeleteOutlined />}
                onClick={() => handleDelete(record.supplier_id!)}
                danger
                size="small"
                className="text-red-600"
              />
            </Tooltip>
          </Space>
        ),
      },
    ],
    [filteredRatings, suppliers, selectedRows, openEditModal, handleDelete]
  );

  const allColumns = useMemo(
    () => [
      { key: "selection", label: "Select" },
      { key: "supplier_id", label: "Supplier" },
      { key: "materialDescription", label: "Material" },
      { key: "lotReceived", label: "Lot Received" },
      { key: "lotAccepted", label: "Lot Accepted" },
      { key: "acceptedPercent", label: "Accepted %" },
      { key: "overallScore", label: "Overall Score" },
      { key: "ratingDate", label: "Rating Date" },
      { key: "onTimeDelivery", label: "On Time Delivery" },
      { key: "shortDelivery", label: "Short Delivery" },
      { key: "risklevel", label: "Risk Level" },
      { key: "category", label: "Category" },
      { key: "ratingScale", label: "Rating Scale" },
      { key: "remark", label: "Remark" },
      { key: "rating_By", label: "Rated By" },
      { key: "approved_By", label: "Approved By" },
      { key: "created_at", label: "Created At" },
      { key: "updated_at", label: "Updated At" },
      { key: "actions", label: "Actions" },
    ],
    []
  );

  const columnKeys = useMemo(() => allColumns.map((c) => c.key), [allColumns]);
  const labelsMap = useMemo(
    () => Object.fromEntries(allColumns.map((c) => [c.key, c.label])),
    [allColumns]
  );
  const [visibleColumnKeys, setVisibleColumnKeys] =
    useState<string[]>(columnKeys);

  const handleVisibleColumnChange = useCallback((cols: string[]) => {
    setVisibleColumnKeys(cols);
  }, []);

  const filteredColumns = useMemo(
    () =>
      columns.filter(
        (col) =>
          typeof col.key === "string" && visibleColumnKeys.includes(col.key)
      ),
    [columns, visibleColumnKeys]
  );

  // const RatingCard = useCallback(
  //   ({ rating }: { rating: SupplierRating }) => {
  //     const supplier = suppliers.find((s) => s.supplier_id === rating.supplier_id);
  //     return (
  //       <div className="border rounded-xl p-4 shadow-sm hover:shadow-md bg-white dark:bg-gray-900 transition-all w-full bg-gradient-to-br from-gray-50 to-gray-100 dark:from-gray-800 dark:to-gray-900">
  //         <div className="flex flex-col sm:flex-row justify-between items-start sm:items-center mb-2 gap-2">
  //           <h3 className="text-base sm:text-lg font-semibold text-gray-800 dark:text-white truncate">
  //             {supplier?.supplier_name || rating.supplier_id || "N/A"}
  //           </h3>
  //           <Tag color="blue">
  //             {rating.overallScore} <StarOutlined />
  //           </Tag>
  //         </div>
  //         <p className="text-sm text-gray-600 dark:text-gray-300 mb-1 truncate">
  //           <strong>Material:</strong> {rating.materialDescription || "N/A"}
  //         </p>
  //         <p className="text-sm text-gray-600 dark:text-gray-300 mb-1">
  //           <strong>Lots:</strong> Received {rating.lotReceived || 0}, Accepted{" "}
  //           {rating.lotAccepted || 0} ({rating.acceptedPercent || 0}%)
  //         </p>
  //         <p className="text-sm text-gray-600 dark:text-gray-300 mb-1">
  //           <strong>Ratings:</strong> Quality {rating.qualityRating || 0}/40, Delivery{" "}
  //           {rating.deliveryRating || 0}/30, Service {rating.serviceRating || 0}/20, Price{" "}
  //           {rating.priceRating || 0}/10
  //         </p>
  //         <p className="text-sm text-gray-600 dark:text-gray-300 mb-1">
  //           <strong>Date:</strong> {rating.ratingDate ? dayjs(rating.ratingDate).format("YYYY-MM-DD") : "N/A"}
  //         </p>
  //         <p className="text-sm text-gray-600 dark:text-gray-300 mb-1">
  //           <strong>Delivery:</strong> On-Time {rating.onTimeDelivery || 0}, Short{" "}
  //           {rating.shortDelivery || 0}
  //         </p>
  //         <p className="text-sm text-gray-600 dark:text-gray-300 mb-1 truncate">
  //           <strong>Rated By:</strong> {rating.rating_By || "N/A"}
  //         </p>
  //         <p className="text-sm text-gray-600 dark:text-gray-300 mb-1 truncate">
  //           <strong>Approved By:</strong> {rating.approved_By || "N/A"}
  //         </p>
  //         <p className="text-sm text-gray-500 dark:text-gray-400 text-xs">
  //           <strong>Created:</strong>{" "}
  //           {rating.created_at ? dayjs(rating.created_at).format("YYYY-MM-DD") : "N/A"}
  //         </p>
  //         <p className="text-sm text-gray-500 dark:text-gray-400 text-xs">
  //           <strong>Updated:</strong>{" "}
  //           {rating.updated_at ? dayjs(rating.updated_at).format("YYYY-MM-DD") : "N/A"}
  //         </p>
  //         <p className="text-sm text-gray-500 dark:text-gray-400 text-xs">
  //           <strong>Score:</strong> {rating.overallScore || 0}{" "}
  //           <StarOutlined className="text-yellow-500" />
  //         </p>
  //         <div className="mt-2 flex gap-2">
  //           <Tooltip title="View">
  //             <Button
  //               icon={<EyeOutlined />}
  //               onClick={() => setViewingRating(rating)}
  //               size="small"
  //               className="text-blue-600"
  //             />
  //           </Tooltip>
  //           <Tooltip title="Edit">
  //             <Button
  //               icon={<EditOutlined />}
  //               onClick={() => openEditModal(rating)}
  //               size="small"
  //               type="primary"
  //               className="bg-blue-600 hover:bg-blue-700"
  //             />
  //           </Tooltip>
  //           <Tooltip title="Delete">
  //             <Button
  //               icon={<DeleteOutlined />}
  //               onClick={() => handleDelete(rating.supplier_id!)}
  //               size="small"
  //               danger
  //               className="text-red-600"
  //             />
  //           </Tooltip>
  //         </div>
  //       </div>
  //     );
  //   },
  //   [suppliers, openEditModal, handleDelete]
  // );
  const RatingCard = React.memo(({ rating }: { rating: SupplierRating }) => {
    const supplier = suppliers.find(
      (s) => s.supplier_id === rating.supplier_id
    );

    return (
      <div
        className="border rounded-xl p-4 shadow-sm hover:shadow-md 
                    bg-white dark:bg-gray-900 transition-all w-full 
                    bg-gradient-to-br from-gray-50 to-gray-100 
                    dark:from-gray-800 dark:to-gray-900 flex flex-col"
      >
        {/* Header */}
        <div className="flex flex-col sm:flex-row justify-between items-start sm:items-center mb-4">
          <h3 className="text-base sm:text-lg font-semibold text-gray-800 dark:text-white truncate">
            {supplier?.supplier_name || rating.supplier_id || "N/A"}
          </h3>
          <Tag color="blue">
            {rating.overallScore} <StarOutlined />
          </Tag>
        </div>

        {/* Line by Line Info */}
        <div className="text-sm space-y-2">
          <p className="text-gray-600 dark:text-gray-300">
            <strong>Material:</strong> {rating.materialDescription || "N/A"}
          </p>
          <p className="text-gray-600 dark:text-gray-300">
            <strong>Lots:</strong> Received {rating.lotReceived || 0}, Accepted{" "}
            {rating.lotAccepted || 0} ({rating.acceptedPercent || 0}%)
          </p>
          <p className="text-gray-600 dark:text-gray-300">
            <strong>Ratings:</strong> Quality {rating.qualityRating || 0}/40,
            Delivery {rating.deliveryRating || 0}/30, Service{" "}
            {rating.serviceRating || 0}/20, Price {rating.priceRating || 0}/10
          </p>
          <p className="text-gray-600 dark:text-gray-300">
            <strong>Date:</strong>{" "}
            {rating.ratingDate
              ? dayjs(rating.ratingDate).format("YYYY-MM-DD")
              : "N/A"}
          </p>
          <p className="text-gray-600 dark:text-gray-300">
            <strong>Delivery:</strong> On-Time {rating.onTimeDelivery || 0},
            Short {rating.shortDelivery || 0}
          </p>
          <p className="text-gray-600 dark:text-gray-300">
            <strong>Rating Scale:</strong> {rating.ratingScale || "N/A"}
          </p>
          <p className="text-gray-600 dark:text-gray-300">
            <strong>Remark:</strong> {rating.remark || "N/A"}
          </p>
          <p className="text-gray-600 dark:text-gray-300">
            <strong>Rated By:</strong> {rating.rating_By || "N/A"}
          </p>
          <p className="text-gray-600 dark:text-gray-300">
            <strong>Approved By:</strong> {rating.approved_By || "N/A"}
          </p>
          <p className="text-gray-500 dark:text-gray-400 text-xs">
            <strong>Created:</strong>{" "}
            {rating.created_at
              ? dayjs(rating.created_at).format("YYYY-MM-DD")
              : "N/A"}
          </p>
          <p className="text-gray-500 dark:text-gray-400 text-xs">
            <strong>Updated:</strong>{" "}
            {rating.updated_at
              ? dayjs(rating.updated_at).format("YYYY-MM-DD")
              : "N/A"}
          </p>
          <p className="text-gray-500 dark:text-gray-400 text-xs">
            <strong>Overall Score:</strong> {rating.overallScore || 0}{" "}
            <StarOutlined className="text-yellow-500" />
          </p>
        </div>

        {/* Actions */}
        <div className="mt-3 flex flex-wrap gap-2">
          <Tooltip title="View">
            <Button
              icon={<EyeOutlined />}
              onClick={() => setViewingRating(rating)}
              size="small"
              className="text-blue-600"
            />
          </Tooltip>
          <Tooltip title="Edit">
            <Button
              icon={<EditOutlined />}
              onClick={() => openEditModal(rating)}
              size="small"
              type="primary"
              className="bg-blue-600 hover:bg-blue-700"
            />
          </Tooltip>
          <Tooltip title="Delete">
            <Button
              icon={<DeleteOutlined />}
              onClick={() => handleDelete(rating.supplier_id!)}
              size="small"
              danger
              className="text-red-600"
            />
          </Tooltip>
        </div>
      </div>
    );
  });

  RatingCard.displayName = "RatingCard";
  const total = filteredRatings.length;
  const highQuality = useMemo(
    () => filteredRatings.filter((r) => (r.qualityRating || 0) >= 30).length,
    [filteredRatings]
  );
  const highAcceptance = useMemo(
    () => filteredRatings.filter((r) => (r.acceptedPercent || 0) >= 80).length,
    [filteredRatings]
  );
  const highScore = useMemo(
    () => filteredRatings.filter((r) => (r.overallScore || 0) >= 80).length,
    [filteredRatings]
  );

  const trendData = useMemo(
    () => [
      { value: 10 },
      { value: 15 },
      { value: 12 },
      { value: 18 },
      { value: 20 },
    ],
    []
  );

  return (
    <div className="px-2 sm:px-2 md:px-2 lg:px-2 md:py-2 lg:py-2 py-2 sm:py-2">
      <h2 className="flex items-center gap-2 text-xl sm:text-2xl md:text-3xl font-bold text-gray-800 dark:text-white mb-2">
        <FileLineChartIcon className="w-7 h-7 text-green-600 dark:text-green-400 flex-shrink-0" />
        <span>Supplier Ratings</span>
        <span className="text-gray-500 text-base sm:text-lg font-medium">
          ({total})
        </span>
      </h2>
      <p className="text-gray-600 dark:text-gray-300 text-sm sm:text-base mb-4">
        Evaluate supplier performance based on delivery, quality, and service.
      </p>
      <Breadcrumb
        className="mb-4 sm:mb-6 mt-2 text-sm text-gray-500 dark:text-gray-400"
        separator=">"
        items={[
          {
            title: (
              <Tooltip title="Dashboard">
                <HomeOutlined />
              </Tooltip>
            ),
            href: "/dashboard",
          },
          { title: "Master Data" },
          { title: "Supplier Ratings" },
        ]}
      />

      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4 mt-3 mb-6">
        <WidgetCard
          title="Total Ratings"
          value={total}
          icon={<AppstoreOutlined />}
          color="green"
          percentage={12}
          description="All supplier rating records"
          trendData={trendData}
          className="bg-gradient-to-br from-gray-50 to-gray-100 dark:from-gray-800 dark:to-gray-900"
        />
        <WidgetCard
          title="High Quality"
          value={highQuality}
          icon={<CheckCircleTwoTone twoToneColor="#52c41a" />}
          color="blue"
          percentage={8}
          description="Suppliers with quality rating ≥ 30"
          trendData={trendData}
          className="bg-gradient-to-br from-gray-50 to-gray-100 dark:from-gray-800 dark:to-gray-900"
        />
        <WidgetCard
          title="High Acceptance"
          value={highAcceptance}
          icon={<CheckCircleOutlined />}
          color="orange"
          percentage={5}
          description="Suppliers with ≥ 80% acceptance"
          trendData={trendData}
          className="bg-gradient-to-br from-gray-50 to-gray-100 dark:from-gray-800 dark:to-gray-900"
        />
        <WidgetCard
          title="Top Performers"
          value={highScore}
          icon={<StarOutlined />}
          color="red"
          percentage={3}
          description="Suppliers with overall score ≥ 80"
          trendData={trendData}
          className="bg-gradient-to-br from-gray-50 to-gray-100 dark:from-gray-800 dark:to-gray-900"
        />
      </div>

      <div className="mt-4">
        <Card title="Supplier Rating List" className="p-0 mb-6 shadow-sm">
          <div className="p-4 flex flex-col gap-4">
            <div className="flex flex-col sm:flex-row justify-between items-stretch sm:items-center gap-4">
              <Input
                prefix={<SearchOutlined />}
                allowClear
                placeholder="Search Supplier or Material"
                value={searchTerm}
                onChange={(e) => setSearchTerm(e.target.value)}
                className="w-full sm:w-64 border-gray-300 dark:border-gray-600 rounded-md shadow-sm"
              />
              <Dropdown
                className="w-full sm:w-auto border-gray-300 dark:border-gray-600"
                menu={{
                  items: [
                    { label: "Export CSV", key: "csv" },
                    { label: "Export PDF", key: "pdf" },
                  ],
                }}
              >
                <Button icon={<DownloadOutlined />}>Export</Button>
              </Dropdown>
              <Button
                type="primary"
                icon={<PlusOutlined />}
                onClick={openAddModal}
                className="bg-blue-600 hover:bg-blue-700 w-full sm:w-auto"
              >
                Add Rating
              </Button>
            </div>
            <div className="flex flex-col gap-4 lg:flex-row lg:justify-between lg:items-center">
              <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-4 w-full lg:w-auto">
                <Select
                  placeholder="Material Description"
                  className="w-full max-w-[160px]"
                  onChange={(value) =>
                    setAdvancedFilters((prev) => ({
                      ...prev,
                      materialDescription: value,
                    }))
                  }
                  allowClear
                  options={Array.from(
                    new Set((ratings ?? []).map((r) => r.materialDescription))
                  )
                    .filter((v): v is string => !!v)
                    .map((value) => ({ label: value, value }))}
                />
              </div>
              <div className="flex flex-col sm:flex-row flex-wrap gap-4 justify-start lg:justify-end w-full lg:w-auto">
                <ColumnVisibilityToggle
                  columns={columnKeys}
                  labels={labelsMap}
                  storageKey="supplier_rating_table_visibility"
                  onChange={handleVisibleColumnChange}
                />
                <DatePicker.RangePicker
                  value={dateRange}
                  onChange={(dates) =>
                    setDateRange(dates as [dayjs.Dayjs, dayjs.Dayjs] | null)
                  }
                  className="w-full sm:w-auto border-gray-300 dark:border-gray-600"
                />
                <Button
                  onClick={handleBulkDelete}
                  disabled={selectedRows.length === 0}
                  danger
                  className="w-full sm:w-auto"
                >
                  Bulk Delete
                </Button>
                <div className="hidden sm:block">
                  <Switch
                    checkedChildren={<AppstoreOutlined />}
                    unCheckedChildren={<TableOutlined />}
                    checked={isGridView}
                    onChange={setIsGridView}
                    className="w-full sm:w-auto"
                  />
                </div>
              </div>
            </div>

            {advancedFilters.materialDescription && (
              <div className="flex flex-wrap gap-2 mt-2">
                <Tag
                  closable
                  onClose={() =>
                    setAdvancedFilters((prev) => ({
                      ...prev,
                      materialDescription: undefined,
                    }))
                  }
                >
                  Material: {advancedFilters.materialDescription}
                </Tag>
              </div>
            )}
          </div>

          {isGridView || isMobile ? (
            <div className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 lg:grid-cols-3 gap-4 p-4">
              {(paginatedRatings ?? []).map((rating) => (
                <RatingCard key={rating.supplier_id} rating={rating} />
              ))}
            </div>
          ) : (
            <div className="overflow-auto">
              <DataTable<SupplierRating>
                columns={filteredColumns}
                dataSource={paginatedRatings}
                rowKey="supplier_id"
                scrollX="max-content"
                className="mt-4"
                pagination={false}
                loading={loading}
              />
            </div>
          )}
          <div className="flex flex-col sm:flex-row justify-between items-center px-4 py-2 text-sm text-gray-600 dark:text-gray-300">
            <div className="mb-2 sm:mb-0">
              Showing <strong>{(currentPage - 1) * pageSize + 1}</strong> to{" "}
              <strong>
                {Math.min(currentPage * pageSize, filteredRatings.length)}
              </strong>{" "}
              of <strong>{filteredRatings.length}</strong> ratings
            </div>
            <Pagination
              current={currentPage}
              pageSize={pageSize}
              total={filteredRatings.length}
              onChange={(page, size) => {
                setCurrentPage(page);
                setPageSize(size);
              }}
              showSizeChanger
              pageSizeOptions={["10", "30", "50", "100"]}
              size="small"
            />
          </div>
        </Card>
      </div>

      <motion.div
        initial={{ opacity: 0, scale: 0.95 }}
        animate={{ opacity: 1, scale: 1 }}
        exit={{ opacity: 0, scale: 0.95 }}
      >
        <Modal
          title={
            <div className="flex items-center gap-2 border-b pb-2 border-gray-500 dark:border-gray-600">
              {editingRating ? (
                <UserCheck className="w-5 h-5 text-blue-600" /> // Edit icon
              ) : (
                <UserPlus className="w-5 h-5 text-green-600" /> // Add icon
              )}
              <span>
                {editingRating ? "Edit Supplier Rating" : "Add Supplier Rating"}
              </span>
            </div>
          }
          open={isModalOpen}
          onOk={handleFormSubmit}
          onCancel={() => {
            setIsModalOpen(false);
            form.resetFields();
          }}
          destroyOnHidden
          width="95%" // responsive width
          className="!top-5 font-semibold"
          styles={{
            body: { maxHeight: "80vh", overflowY: "auto" }, // ✅ fixed line
          }}
          okButtonProps={{ loading: isSubmitting }}
        >
          <Form form={form} layout="vertical" className="w-full px-4 space-y-4">
            {/* Supplier Info */}
            <Card title="Supplier Info" size="small" className="shadow-sm">
              <div className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-4">
                <Form.Item
                  name="supplier_id"
                  label="Supplier"
                  rules={[{ required: true, message: "Supplier is required" }]}
                >
                  <Select
                    showSearch
                    placeholder="Select a supplier"
                    optionFilterProp="children"
                    filterOption={(input, option) =>
                      (option?.label as string)
                        ?.toLowerCase()
                        .includes(input.toLowerCase())
                    }
                    options={(suppliers ?? []).map((s) => ({
                      value: s.supplier_id,
                      label: `${s.supplier_name} (${
                        s.contactPerson || "N/A"
                      }, ${s.supplier_type || "N/A"})`,
                    }))}
                  />
                </Form.Item>

                <Form.Item
                  name="materialDescription"
                  label="Material Description"
                  rules={[
                    {
                      required: true,
                      message: "Material Description is required",
                    },
                  ]}
                >
                  <Input />
                </Form.Item>

                <Form.Item
                  name="lotReceived"
                  label="Lot Received"
                  rules={[
                    { required: true, message: "Lot Received is required" },
                  ]}
                >
                  <InputNumber className="w-full" min={0} />
                </Form.Item>

                <Form.Item
                  name="lotAccepted"
                  label="Lot Accepted (>=100)"
                  rules={[
                    { required: true, message: "Lot Accepted is required" },
                  ]}
                >
                  <InputNumber className="w-full" min={0} />
                </Form.Item>
              </div>
            </Card>

            {/* Ratings */}
            <Card title="Ratings" size="small" className="shadow-sm">
              <div className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-4">
                <Form.Item
                  name="qualityRating"
                  label="Quality Rating (40 pts)"
                  rules={[
                    { required: true, message: "Quality Rating is required" },
                  ]}
                >
                  <InputNumber className="w-full" max={40} min={0} />
                </Form.Item>

                <Form.Item
                  name="deliveryRating"
                  label="Delivery Rating (30 pts)"
                  rules={[
                    { required: true, message: "Delivery Rating is required" },
                  ]}
                >
                  <InputNumber className="w-full" max={30} min={0} />
                </Form.Item>

                <Form.Item
                  name="serviceRating"
                  label="Service Rating (20 pts)"
                  rules={[
                    { required: true, message: "Service Rating is required" },
                  ]}
                >
                  <InputNumber className="w-full" max={20} min={0} />
                </Form.Item>

                <Form.Item
                  name="priceRating"
                  label="Price Rating (10 pts)"
                  rules={[
                    { required: true, message: "Price Rating is required" },
                  ]}
                >
                  <InputNumber className="w-full" max={10} min={0} />
                </Form.Item>

                <Form.Item
                  name="ratingDate"
                  label="Rating Date"
                  rules={[
                    { required: true, message: "Rating Date is required" },
                  ]}
                >
                  <DatePicker className="w-full" />
                </Form.Item>
              </div>
            </Card>

            {/* Delivery */}
            <Card
              title="Delivery Performance"
              size="small"
              className="shadow-sm"
            >
              <div className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-4">
                <Form.Item name="onTimeDelivery" label="On Time Delivery">
                  <InputNumber className="w-full" min={0} />
                </Form.Item>

                <Form.Item name="shortDelivery" label="Short Delivery">
                  <InputNumber className="w-full" min={0} />
                </Form.Item>

                <Form.Item
                  name="ratingScale"
                  label="Rating Scale"
                  rules={[
                    { required: true, message: "Rating Scale is required" },
                  ]}
                >
                  <Select
                    options={[
                      { value: "excellent", label: "Excellent" },
                      { value: "good", label: "Good" },
                      { value: "average", label: "Average" },
                      { value: "poor", label: "Poor" },
                    ]}
                  />
                </Form.Item>
              </div>
            </Card>

            {/* Remark */}
            <Card title="Remarks" size="small" className="shadow-sm">
              <Form.Item name="remark" label="Remark">
                <Input.TextArea rows={3} />
              </Form.Item>
            </Card>
          </Form>
        </Modal>
      </motion.div>

      <Modal
        title="Supplier Rating Details"
        open={!!viewingRating}
        onCancel={() => setViewingRating(null)}
        footer={null}
        className="w-full max-w-[90vw] sm:max-w-[500px] top-4"
      >
        {viewingRating && (
          <div className="space-y-2 text-sm sm:text-base">
            <p>
              <strong>Supplier:</strong>{" "}
              <span className="text-green-600">
                {suppliers.find(
                  (s) => s.supplier_id === viewingRating.supplier_id
                )?.supplier_name || viewingRating.supplier_id}
              </span>
            </p>
            <p>
              <strong>Material:</strong> {viewingRating.materialDescription}
            </p>
            <p>
              <strong>Lot Received:</strong> {viewingRating.lotReceived}
            </p>
            <p>
              <strong>Lot Accepted:</strong> {viewingRating.lotAccepted}
            </p>
            <p>
              <strong>Accepted %:</strong> {viewingRating.acceptedPercent}%
            </p>
            <p>
              <strong>Quality:</strong> {viewingRating.qualityRating}
            </p>
            <p>
              <strong>Delivery:</strong> {viewingRating.deliveryRating}
            </p>
            <p>
              <strong>Service:</strong> {viewingRating.serviceRating}
            </p>
            <p>
              <strong>Price:</strong> {viewingRating.priceRating}
            </p>
            <p>
              <strong>Overall Score:</strong> {viewingRating.overallScore}{" "}
              <StarOutlined className="text-yellow-500" />
            </p>
            <p>
              <strong>Rating Date:</strong>{" "}
              {dayjs(viewingRating.ratingDate).format("YYYY-MM-DD")}
            </p>
            <p>
              <strong>On Time Delivery:</strong>{" "}
              {viewingRating.onTimeDelivery || "-"}
            </p>
            <p>
              <strong>Short Delivery:</strong>{" "}
              {viewingRating.shortDelivery || "-"}
            </p>
            <p>
              <strong>Rating Scale:</strong> {viewingRating.ratingScale || "-"}
            </p>
            <p>
              <strong>Remark:</strong> {viewingRating.remark || "-"}
            </p>
            <p>
              <strong>Risk Level:</strong> {viewingRating.risklevel || "-"}
            </p>
            <p>
              <strong>Category:</strong> {viewingRating.category || "-"}
            </p>

            <p>
              <strong>Rated By:</strong> {viewingRating.rating_By || "-"}
            </p>
            <p>
              <strong>Approved By:</strong> {viewingRating.approved_By || "-"}
            </p>
            <p>
              <strong>Created:</strong>{" "}
              {dayjs(viewingRating.created_at).format("YYYY-MM-DD")}
            </p>
            <p>
              <strong>Updated:</strong>{" "}
              {dayjs(viewingRating.updated_at).format("YYYY-MM-DD")}
            </p>
          </div>
        )}
      </Modal>
      <ConfirmDialog
        open={confirmDialog.open}
        title={confirmDialog.title}
        message={confirmDialog.message}
        onClose={() => setConfirmDialog((prev) => ({ ...prev, open: false }))}
        onConfirm={confirmDialog.onConfirm}
      />
    </div>
  );
}
