"use client";

import React, { useCallback, useEffect, useState } from "react";
import Button from "antd/es/button";
import "antd/es/button/style";
import Card from "antd/es/card";
import "antd/es/card/style";
import DatePicker from "antd/es/date-picker";
import "antd/es/date-picker/style";
import Form from "antd/es/form";
import "antd/es/form/style";
import Input from "antd/es/input";
import "antd/es/input/style";
import Modal from "antd/es/modal";
import "antd/es/modal/style";
import Select from "antd/es/select";
import "antd/es/select/style";
import Space from "antd/es/space";
import "antd/es/space/style";
import Tooltip from "antd/es/tooltip";
import "antd/es/tooltip/style";
import Tag from "antd/es/tag";
import "antd/es/tag/style";
import Breadcrumb from "antd/es/breadcrumb";
import "antd/es/breadcrumb/style";
import Switch from "antd/es/switch";
import "antd/es/switch/style";
import Pagination from "antd/es/pagination";
import "antd/es/pagination/style";
import Dropdown from "antd/es/dropdown";
import "antd/es/dropdown/style";
import Upload from "antd/es/upload";
import "antd/es/upload/style";
import {
  PlusOutlined,
  EditOutlined,
  DeleteOutlined,
  EyeOutlined,
  HomeOutlined,
  TableOutlined,
  AppstoreOutlined,
  SearchOutlined,
  DownloadOutlined,
  CheckCircleTwoTone,
  WarningOutlined,
  ClockCircleOutlined,
  CheckCircleOutlined,
  CloseCircleOutlined,
  FileOutlined,
} from "@ant-design/icons";
import type { ColumnsType } from "antd/es/table";
import DataTable from "../components/common/DataTable";
import { useIsMobile } from "@/data/useIsMobile";
import { toast } from "sonner";
import { motion } from "framer-motion";
import dayjs from "@/data/utils/dayjs";
import dynamic from "next/dynamic";
import {
  createSupplierAction,
  deleteSupplierAction,
  downloadSupplierAction,
  getAllSupplierActions,
  SupplierAction,
  updateSupplierAction,
} from "../api/supplieraction/supplieraction";
import { getAllSuppliers, Supplier } from "../api/suppliers/suppliers";
import { getAllProducts, Product } from "../api/products/product";
import ConfirmDialog from "../components/common/ConfirmDialog";
import GlassCard from "../components/common/GlassCard";
import { Edit2, PlusSquare } from "lucide-react";
import { ClipboardListIcon } from "lucide-react";

const WidgetCard = dynamic(() => import("../components/common/WidgetCard"), {
  ssr: false,
});

const ColumnVisibilityToggle = dynamic(
  () => import("../components/common/ManageColumn"),
  {
    ssr: false,
  }
);

const statusOptions = ["Pending", "InProgress", "Completed", "Closed"] as const;
const priorityOptions = ["Low", "Medium", "High", "Critical"] as const;
const departmentOptions = [
  "Quality",
  "Procurement",
  "Production",
  "Other",
] as const;
const actionTypeOptions = ["Corrective", "Preventive"] as const;

export default function SupplierActionPage() {
  const [form] = Form.useForm();
  const [actions, setActions] = useState<SupplierAction[]>([]);
  const [suppliers, setSuppliers] = useState<Supplier[]>([]);
  const [products, setProducts] = useState<Product[]>([]);
  const [isModalOpen, setIsModalOpen] = useState(false);
  const [editingAction, setEditingAction] = useState<SupplierAction | null>(
    null
  );
  const [viewingAction, setViewingAction] = useState<SupplierAction | null>(
    null
  );
  const [confirmDialog, setConfirmDialog] = useState({
    open: false,
    title: "",
    message: "",
    onConfirm: () => {},
  });

  const [searchTerm, setSearchTerm] = useState("");
  const [isGridView, setIsGridView] = useState(false);
  const [selectedRows, setSelectedRows] = useState<string[]>([]);
  const [advancedFilters, setAdvancedFilters] = useState({
    Status: undefined,
    unitName: undefined,
    priorityLevel: undefined,
    department: undefined,
    action_type: undefined,
  });
  const [dateRange, setDateRange] = useState<[dayjs.Dayjs, dayjs.Dayjs] | null>(
    null
  );
  const [currentPage, setCurrentPage] = useState(1);
  const [pageSize, setPageSize] = useState(10);
  const [loading, setLoading] = useState(false);
  const [fileList, setFileList] = useState<
    import("antd/es/upload/interface").UploadFile[]
  >([]);
  const isMobile = useIsMobile();

  // Fetch suppliers, products, and actions on mount
  useEffect(() => {
    const fetchData = async () => {
      setLoading(true);
      try {
        const [supplierData, productData, actionData] = await Promise.all([
          getAllSuppliers(),
          getAllProducts(),
          getAllSupplierActions(),
        ]);
        setSuppliers(supplierData);
        setProducts(productData);
        setActions(actionData);
      } catch (error) {
        toast.error("Failed to fetch data");
      } finally {
        setLoading(false);
      }
    };
    fetchData();
  }, []);

  const openAddModal = () => {
    form.resetFields();
    setEditingAction(null);
    setFileList([]);
    setIsModalOpen(true);
  };

  const openEditModal = (action: SupplierAction) => {
    form.setFieldsValue({
      ...action,
      dateIdentified: action.dateIdentified
        ? dayjs(action.dateIdentified)
        : undefined,
      dateResolved: action.dateResolved
        ? dayjs(action.dateResolved)
        : undefined,
      targetdate: action.targetdate ? dayjs(action.targetdate) : undefined,
      duedate: action.duedate ? dayjs(action.duedate) : undefined,
    });
    setEditingAction(action);
    setFileList(
      action.evidence_file && typeof action.evidence_file === "string"
        ? [
            {
              uid: "-1",
              name: action.evidence_file.split("/").pop() || "evidence",
              status: "done",
              url: action.evidence_file,
            },
          ]
        : []
    );
    setIsModalOpen(true);
  };

  const handleDelete = (supplier_id: string, product_id: string) => {
    setConfirmDialog({
      open: true,
      title: "Delete Supplier Action",
      message: "Are you sure you want to delete this action?",
      onConfirm: async () => {
        try {
          await deleteSupplierAction(supplier_id, product_id);
          toast.success("Action deleted successfully");
          setActions((prev) =>
            prev.filter(
              (a) =>
                !(a.supplier_id === supplier_id && a.product_id === product_id)
            )
          );
        } catch {
          toast.error("Failed to delete action");
        }
      },
    });
  };

  const handleBulkDelete = () => {
    if (selectedRows.length === 0) return;
    setConfirmDialog({
      open: true,
      title: `Delete Multiple Supplier Actions`,
      message: `Are you sure you want to delete ${selectedRows.length} actions?`,
      onConfirm: async () => {
        try {
          for (const rowId of selectedRows) {
            const [supplier_id, product_id] = rowId.split("-");
            await deleteSupplierAction(supplier_id, product_id);
          }
          toast.success("Actions deleted successfully");
          setSelectedRows([]);
          setActions((prev) =>
            prev.filter(
              (a) => !selectedRows.includes(`${a.supplier_id}-${a.product_id}`)
            )
          );
        } catch (error) {
          toast.error("Failed to delete actions");
        }
      },
    });
  };

  const handleFormSubmit = async () => {
    try {
      const values = await form.validateFields();
      const data: SupplierAction = {
        ...values,
        dateIdentified: dayjs(values.dateIdentified).format("YYYY-MM-DD"),
        dateResolved: dayjs(values.dateResolved).format("YYYY-MM-DD"),
        targetdate: dayjs(values.targetdate).format("YYYY-MM-DD"),
        duedate: dayjs(values.duedate).format("YYYY-MM-DD"),
        evidence_file: fileList[0].originFileObj as File | undefined,
      };

      if (editingAction) {
        await updateSupplierAction(
          editingAction.supplier_id,
          editingAction.product_id,
          data
        );
        toast.success("Action updated successfully");
        setActions((prev) =>
          prev.map((a) =>
            a.supplier_id === editingAction.supplier_id &&
            a.product_id === editingAction.product_id
              ? { ...a, ...values } // merge updated fields
              : a
          )
        );
      } else {
        const newAction = await createSupplierAction(data);
        toast.success("Action added successfully");
        setActions((prev) => [...prev, newAction]);
      }
      setIsModalOpen(false);
      form.resetFields();
      setFileList([]);
      const actionData = await getAllSupplierActions();
      setActions(actionData);
    } catch (error) {
      toast.error("Failed to save action");
    }
  };
  const handleDownload = async (supplier_id: string, product_id: string) => {
    try {
      await downloadSupplierAction(supplier_id, product_id);
      toast.success("Document downloaded");
    } catch (error) {
      toast.error("Failed to download document");
    }
  };

  const renderTag = (value: string) => {
    const colorMap: Record<string, string> = {
      Pending: "gold",
      InProgress: "blue",
      Completed: "green",
      Closed: "red",
    };

    const iconMap: Record<string, React.ReactNode> = {
      Pending: <WarningOutlined />,
      InProgress: <ClockCircleOutlined />,
      Completed: <CheckCircleOutlined />,
      Closed: <CloseCircleOutlined />,
    };

    return (
      <Tag icon={iconMap[value]} color={colorMap[value] || "default"}>
        {value}
      </Tag>
    );
  };

  const getSupplierDisplay = (supplier_id: string) => {
    const supplier = suppliers.find((s) => s.supplier_id === supplier_id);
    return supplier
      ? `${supplier.company_name} (${supplier.contact_person || "N/A"})`
      : supplier_id;
  };

  const getProductDisplay = (product_id: string) => {
    const product = products.find((p) => p.product_id === product_id);
    return product
      ? `${product.product_name} (${product.supplier_name || "N/A"})`
      : product_id;
  };

  const filteredActions = actions.filter((action) => {
    const supplier = suppliers.find(
      (s) => s.supplier_id === action.supplier_id
    );
    const product = products.find((p) => p.product_id === action.product_id);

    const matchesSearch =
      (supplier?.company_name
        ?.toLowerCase()
        .includes(searchTerm.toLowerCase()) ??
        false) ||
      (supplier?.contact_person
        ?.toLowerCase()
        .includes(searchTerm.toLowerCase()) ??
        false) ||
      (product?.product_name
        ?.toLowerCase()
        .includes(searchTerm.toLowerCase()) ??
        false) ||
      (product?.supplier_name
        ?.toLowerCase()
        .includes(searchTerm.toLowerCase()) ??
        false) ||
      (action.unitName?.toLowerCase().includes(searchTerm.toLowerCase()) ??
        false) ||
      (action.materialDescription
        ?.toLowerCase()
        .includes(searchTerm.toLowerCase()) ??
        false) ||
      (action.supplier_id?.toLowerCase().includes(searchTerm.toLowerCase()) ??
        false) ||
      (action.product_id?.toLowerCase().includes(searchTerm.toLowerCase()) ??
        false);

    const matchesDate =
      !dateRange ||
      (action.created_at &&
        (!dateRange[0] ||
          dayjs(action.created_at).isSame(dateRange[0]) ||
          dayjs(action.created_at).isAfter(dateRange[0])) &&
        (!dateRange[1] ||
          dayjs(action.created_at).isSame(dateRange[1]) ||
          dayjs(action.created_at).isBefore(dateRange[1])));

    const matchesAdvanced =
      (!advancedFilters.Status || action.Status === advancedFilters.Status) &&
      (!advancedFilters.unitName ||
        action.unitName === advancedFilters.unitName) &&
      (!advancedFilters.priorityLevel ||
        action.priorityLevel === advancedFilters.priorityLevel) &&
      (!advancedFilters.department ||
        action.department === advancedFilters.department) &&
      (!advancedFilters.action_type ||
        action.action_type === advancedFilters.action_type);

    return matchesSearch && matchesDate && matchesAdvanced;
  });

  const paginatedActions = filteredActions.slice(
    (currentPage - 1) * pageSize,
    currentPage * pageSize
  );

  const columns: ColumnsType<SupplierAction> = [
    {
      title: (
        <input
          type="checkbox"
          onChange={(e) => {
            const checked = e.target.checked;
            setSelectedRows(
              checked
                ? filteredActions.map((a) => `${a.supplier_id}-${a.product_id}`)
                : []
            );
          }}
          className="w-4 h-4"
        />
      ),
      key: "selection",
      render: (_, record) => (
        <input
          type="checkbox"
          checked={selectedRows.includes(
            `${record.supplier_id}-${record.product_id}`
          )}
          onChange={(e) => {
            const rowId = `${record.supplier_id}-${record.product_id}`;
            setSelectedRows((prev) =>
              e.target.checked
                ? [...prev, rowId]
                : prev.filter((id) => id !== rowId)
            );
          }}
          className="w-4 h-4"
        />
      ),
      width: 44,
    },
    {
      title: "Supplier",
      dataIndex: "supplier_id",
      key: "supplier_id",
      sorter: (a, b) => {
        const supplierA = suppliers.find(
          (s) => s.supplier_id === a.supplier_id
        );
        const supplierB = suppliers.find(
          (s) => s.supplier_id === b.supplier_id
        );
        return (supplierA?.company_name || a.supplier_id).localeCompare(
          supplierB?.company_name || b.supplier_id
        );
      },
      render: (supplier_id) => {
        const supplier = suppliers.find((s) => s.supplier_id === supplier_id);
        return (
          <span className="text-green-600 font-medium">
            {supplier
              ? `${supplier.company_name} (${supplier.contact_person || "N/A"})`
              : supplier_id}
          </span>
        );
      },
    },
    {
      title: "Product",
      dataIndex: "product_id",
      key: "product_id",
      sorter: (a, b) => {
        const productA = products.find((p) => p.product_id === a.product_id);
        const productB = products.find((p) => p.product_id === b.product_id);
        return (productA?.product_name || a.product_id).localeCompare(
          productB?.product_name || b.product_id
        );
      },
      render: (product_id) => {
        const product = products.find((p) => p.product_id === product_id);
        return product
          ? `${product.product_name} (${product.supplier_name || "N/A"})`
          : product_id;
      },
    },
    {
      title: "Unit",
      dataIndex: "unitName",
      key: "unitName",
      sorter: (a, b) => a.unitName.localeCompare(b.unitName),
      filterSearch: true,
      onFilter: (value, record) =>
        record.unitName.toLowerCase().includes((value as string).toLowerCase()),
      filters: Array.from(new Set(actions.map((a) => a.unitName))).map(
        (value) => ({
          text: value,
          value: value,
        })
      ),
    },
    {
      title: "Material",
      dataIndex: "materialDescription",
      key: "materialDescription",
      sorter: (a, b) =>
        (a.materialDescription ?? "").localeCompare(
          b.materialDescription ?? ""
        ),
      filterSearch: true,
      onFilter: (value, record) =>
        (record.materialDescription ?? "")
          .toLowerCase()
          .includes((value as string).toLowerCase()),
      filters: Array.from(
        new Set(actions.map((a) => a.materialDescription ?? ""))
      ).map((value) => ({
        text: value,
        value: value,
      })),
    },
    {
      title: "Invoice No.",
      dataIndex: "invoiceNo",
      key: "invoiceNo",
      sorter: (a, b) => a.invoiceNo.localeCompare(b.invoiceNo),
      render: (text) => (
        <Tooltip title={text}>
          <span className="block truncate max-w-[100px] sm:max-w-[150px] md:max-w-[180px]">
            {text}
          </span>
        </Tooltip>
      ),
    },
    {
      title: "Supplier Lot No.",
      dataIndex: "supplierLotNo",
      key: "supplierLotNo",
      render: (text) => (
        <Tooltip title={text}>
          <span className="block truncate max-w-[100px] sm:max-w-[150px] md:max-w-[180px]">
            {text}
          </span>
        </Tooltip>
      ),
    },
    {
      title: "Date Identified",
      dataIndex: "dateIdentified",
      key: "dateIdentified",
      sorter: (a, b) =>
        dayjs(a.dateIdentified).unix() - dayjs(b.dateIdentified).unix(),
      render: (text) => dayjs(text).format("YYYY-MM-DD"),
    },
    {
      title: "Date Resolved",
      dataIndex: "dateResolved",
      key: "dateResolved",
      sorter: (a, b) =>
        dayjs(a.dateResolved).unix() - dayjs(b.dateResolved).unix(),
      render: (text) => dayjs(text).format("YYYY-MM-DD"),
    },
    {
      title: "CAPA No.",
      dataIndex: "capaNo",
      key: "capaNo",
      render: (text) => (
        <Tooltip title={text}>
          <span className="block truncate max-w-[100px] sm:max-w-[150px] md:max-w-[180px]">
            {text}
          </span>
        </Tooltip>
      ),
    },
    {
      title: "Non Conformance",
      dataIndex: "nonConformanceDetails",
      key: "nonConformanceDetails",
      sorter: (a, b) =>
        (a.nonConformanceDetails ?? "").localeCompare(
          b.nonConformanceDetails ?? ""
        ),
      render: (text) => (
        <Tooltip title={text}>
          <span className="block truncate max-w-[100px] sm:max-w-[150px] md:max-w-[180px]">
            {text}
          </span>
        </Tooltip>
      ),
    },
    {
      title: "Description",
      dataIndex: "description",
      key: "description",
      sorter: (a, b) =>
        (a.description ?? "").localeCompare(b.description ?? ""),
      render: (text) => (
        <Tooltip title={text}>
          <span className="block truncate max-w-[100px] sm:max-w-[150px] md:max-w-[180px]">
            {text}
          </span>
        </Tooltip>
      ),
    },
    {
      title: "Root Cause Analysis",
      dataIndex: "root_cause_analysis",
      key: "root_cause_analysis",
      sorter: (a, b) =>
        (a.root_cause_analysis ?? "").localeCompare(
          b.root_cause_analysis ?? ""
        ),
      render: (text) => (
        <Tooltip title={text}>
          <span className="block truncate max-w-[100px] sm:max-w-[150px] md:max-w-[180px]">
            {text}
          </span>
        </Tooltip>
      ),
    },
    {
      title: "Action Taken",
      dataIndex: "action_taken",
      key: "action_taken",
      sorter: (a, b) =>
        (a.action_taken ?? "").localeCompare(b.action_taken ?? ""),
      render: (text) => (
        <Tooltip title={text}>
          <span className="block truncate max-w-[100px] sm:max-w-[150px] md:max-w-[180px]">
            {text}
          </span>
        </Tooltip>
      ),
    },
    {
      title: "Status",
      dataIndex: "Status",
      key: "Status",
      filters: statusOptions.map((status) => ({ text: status, value: status })),
      onFilter: (value, record) => record.Status === value,
      render: renderTag,
      sorter: (a, b) => a.Status.localeCompare(b.Status),
    },
    {
      title: "Priority",
      dataIndex: "priorityLevel",
      key: "priorityLevel",
      sorter: (a, b) => a.priorityLevel.localeCompare(b.priorityLevel),
    },
    {
      title: "Department",
      dataIndex: "department",
      key: "department",
      sorter: (a, b) => a.department.localeCompare(b.department),
    },
    {
      title: "Action Type",
      dataIndex: "action_type",
      key: "action_type",
      sorter: (a, b) => a.action_type.localeCompare(b.action_type),
    },
    {
      title: "Responsible",
      dataIndex: "responsible_person",
      key: "responsible_person",
    },
    {
      title: "Target Date",
      dataIndex: "targetdate",
      key: "targetdate",
      render: (text) => dayjs(text).format("YYYY-MM-DD"),
    },
    {
      title: "Due Date",
      dataIndex: "duedate",
      key: "duedate",
      render: (text) => dayjs(text).format("YYYY-MM-DD"),
    },
    {
      title: "Evidence",
      key: "evidence",
      render: (_, record) => (
        <Tooltip title="Download Evidence">
          <Button
            icon={<DownloadOutlined />}
            size="small"
            className="text-green-600"
            onClick={() =>
              handleDownload(record.supplier_id, record.product_id)
            }
          />
        </Tooltip>
      ),
    },
    {
      title: "Actions",
      key: "actions",
      fixed: "right",
      render: (_, record) => (
        <Space size="small">
          <Tooltip title="View">
            <Button
              icon={<EyeOutlined />}
              onClick={() => setViewingAction(record)}
              size="small"
              className="text-blue-600"
            />
          </Tooltip>
          <Tooltip title="Edit">
            <Button
              icon={<EditOutlined />}
              onClick={() => openEditModal(record)}
              size="small"
              type="primary"
              className="bg-blue-600 hover:bg-blue-700"
            />
          </Tooltip>
          <Tooltip title="Delete">
            <Button
              icon={<DeleteOutlined />}
              onClick={() =>
                handleDelete(record.supplier_id, record.product_id)
              }
              danger
              size="small"
              className="text-red-600"
            />
          </Tooltip>
          <Tooltip title="Download PDF">
            <Button
              icon={<DownloadOutlined />}
              onClick={() =>
                handleDownload(record.supplier_id, record.product_id)
              }
              size="small"
              className="text-green-600"
            />
          </Tooltip>
        </Space>
      ),
    },
  ];

  const allColumns = columns.map((col) => ({
    key: col.key as string,
    label: col.title as string,
  }));

  const columnKeys = allColumns.map((c) => c.key);
  const labelsMap = Object.fromEntries(allColumns.map((c) => [c.key, c.label]));
  const [visibleColumnKeys, setVisibleColumnKeys] =
    useState<string[]>(columnKeys);

  const handleVisibleColumnChange = useCallback((cols: string[]) => {
    setVisibleColumnKeys(cols);
  }, []);

  const filteredColumns = columns.filter(
    (col) => typeof col.key === "string" && visibleColumnKeys.includes(col.key)
  );

  const ActionCard = React.memo(({ action }: { action: SupplierAction }) => (
    <GlassCard gradient blurStrength="lg" hoverLift color="success">
      <div
        className="border rounded-xl p-4 shadow-sm hover:shadow-md 
                 bg-white dark:bg-gray-900 transition-all w-full 
                 bg-gradient-to-br from-gray-50 to-gray-100 
                 dark:from-gray-800 dark:to-gray-900 flex flex-col"
      >
        {/* Header */}
        <div className="flex flex-col sm:flex-row justify-between items-start sm:items-center mb-4">
          <h3 className="text-base sm:text-lg font-semibold text-gray-800 dark:text-white truncate">
            {getSupplierDisplay(action.supplier_id)} -{" "}
            {getProductDisplay(action.product_id)}
          </h3>
          <Tag color="blue">{action.Status}</Tag>
        </div>

        {/* Line by Line Info */}
        <div className="text-sm space-y-2">
          <p className="text-gray-600 dark:text-gray-300">
            <strong>Unit:</strong> {action.unitName}
          </p>
          <p className="text-gray-600 dark:text-gray-300">
            <strong>Status:</strong> {renderTag(action.Status)}
          </p>
          <p className="text-gray-500 dark:text-gray-400 text-xs">
            <strong>CAPA No.:</strong> {action.capaNo}
          </p>
          <p className="text-gray-600 dark:text-gray-300">
            <strong>Priority:</strong> {action.priorityLevel}
          </p>
          <p className="text-gray-600 dark:text-gray-300">
            <strong>Department:</strong> {action.department}
          </p>
          <p className="text-gray-600 dark:text-gray-300">
            <strong>Responsible:</strong> {action.responsible_person}
          </p>
          <p className="text-gray-600 dark:text-gray-300">
            <strong>Target Date:</strong>{" "}
            {dayjs(action.targetdate).format("YYYY-MM-DD")}
          </p>
          <p className="text-gray-600 dark:text-gray-300">
            <strong>Due Date:</strong>{" "}
            {dayjs(action.duedate).format("YYYY-MM-DD")}
          </p>

          {action.description && (
            <p className="text-gray-600 dark:text-gray-300">
              <strong>Description:</strong> {action.description}
            </p>
          )}
          {action.root_cause_analysis && (
            <p className="text-gray-600 dark:text-gray-300">
              <strong>Root Cause:</strong> {action.root_cause_analysis}
            </p>
          )}
          {action.action_taken && (
            <p className="text-gray-600 dark:text-gray-300">
              <strong>Action Taken:</strong> {action.action_taken}
            </p>
          )}
        </div>

        {/* Evidence File */}
        {action.evidence_file && (
          <div className="mt-2">
            <p className="text-gray-600 dark:text-gray-300">
              <strong>Evidence:</strong>{" "}
              <Button
                type="link"
                size="small"
                icon={<DownloadOutlined />}
                className="text-green-600"
                onClick={() =>
                  handleDownload(action.supplier_id, action.product_id)
                }
              >
                Download
              </Button>
            </p>
          </div>
        )}

        {/* Footer Actions */}
        <div className="mt-3 flex flex-wrap gap-2">
          <Tooltip title="View">
            <Button
              icon={<EyeOutlined />}
              onClick={() => setViewingAction(action)}
              size="small"
              className="text-blue-600"
            />
          </Tooltip>
          <Tooltip title="Edit">
            <Button
              icon={<EditOutlined />}
              onClick={() => openEditModal(action)}
              size="small"
              type="primary"
              className="bg-blue-600 hover:bg-blue-700"
            />
          </Tooltip>
          <Tooltip title="Delete">
            <Button
              icon={<DeleteOutlined />}
              onClick={() =>
                handleDelete(action.supplier_id, action.product_id)
              }
              size="small"
              danger
              className="text-red-600"
            />
          </Tooltip>
          {action.evidence_file && (
            <Tooltip title="Download Evidence">
              <Button
                icon={<DownloadOutlined />}
                onClick={() =>
                  handleDownload(action.supplier_id, action.product_id)
                }
                size="small"
                className="text-green-600"
              />
            </Tooltip>
          )}
        </div>
      </div>
    </GlassCard>
  ));

  ActionCard.displayName = "ActionCard";

  const total = actions.length;
  const pending = actions.filter((a) => a.Status === "Pending").length;
  const inProgress = actions.filter((a) => a.Status === "InProgress").length;
  const completed = actions.filter((a) => a.Status === "Completed").length;

  const trendData = [
    { value: 10 },
    { value: 15 },
    { value: 12 },
    { value: 18 },
    { value: 20 },
  ];

  return (
    <>
      <div className="px-2 sm:px-2 md:px-2 lg:px-2 md:py-2 lg:py-2 py-2 sm:py-2">
        <h2 className="flex items-center gap-2 text-xl sm:text-2xl md:text-3xl font-bold text-gray-800 dark:text-white mb-2">
          <ClipboardListIcon className="w-7 h-7 text-green-600 dark:text-green-400 flex-shrink-0" />
          <span>Supplier Actions</span>
          <span className="text-gray-500 text-base sm:text-lg font-medium">
            ({actions.length})
          </span>
        </h2>
        <p className="text-gray-600 dark:text-gray-300 text-sm sm:text-base mb-4">
          Manage supplier corrective and preventive actions.
        </p>
        <Breadcrumb
          className="mb-4 sm:mb-6 mt-2 text-sm text-gray-500 dark:text-gray-400"
          separator=">"
          items={[
            {
              title: (
                <Tooltip title="Dashboard">
                  <HomeOutlined />
                </Tooltip>
              ),
              href: "/hrm/dashboard",
            },
            {
              title: "Master Data",
            },
            {
              title: "Supplier Actions",
            },
          ]}
        />

        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4 mt-3 mb-6">
          <WidgetCard
            title="Total Actions"
            value={total}
            icon={<AppstoreOutlined />}
            color="green"
            percentage={12}
            description="All corrective and preventive actions"
            trendData={trendData}
            className="shadow-[0_0_10px_rgba(0,128,255,0.2)] dark:shadow-[0_0_10px_rgba(0,128,255,0.3)]"
          />
          <WidgetCard
            title="Pending Actions"
            value={pending}
            icon={<WarningOutlined />}
            color="orange"
            percentage={5}
            description="Actions awaiting action"
            trendData={trendData}
            className="shadow-[0_0_10px_rgba(255,165,0,0.2)] dark:shadow-[0_0_10px_rgba(255,165,0,0.3)]"
          />
          <WidgetCard
            title="In Progress"
            value={inProgress}
            icon={<ClockCircleOutlined />}
            color="blue"
            percentage={8}
            description="Actions currently in progress"
            trendData={trendData}
            className="shadow-[0_0_10px_rgba(0,0,255,0.2)] dark:shadow-[0_0_10px_rgba(0,0,255,0.3)]"
          />
          <WidgetCard
            title="Completed Actions"
            value={completed}
            icon={<CheckCircleTwoTone twoToneColor="#52c41a" />}
            color="green"
            percentage={3}
            description="Successfully completed actions"
            trendData={trendData}
            className="shadow-[0_0_10px_rgba(0,255,0,0.2)] dark:shadow-[0_0_10px_rgba(0,255,0,0.3)]"
          />
        </div>

        <div className="mt-4">
          <Card title="Supplier Action List" className="p-0 mb-6 shadow-sm">
            <div className="p-4 flex flex-col gap-4">
              <div className="flex flex-col sm:flex-row justify-between items-stretch sm:items-center gap-4">
                <Input
                  prefix={<SearchOutlined />}
                  allowClear
                  placeholder="Search Supplier, Product, Unit or Material"
                  value={searchTerm}
                  onChange={(e) => setSearchTerm(e.target.value)}
                  className="w-full sm:w-64 border-gray-300 dark:border-gray-600 rounded-md shadow-sm"
                />
                <Dropdown
                  className="w-full sm:w-auto border-gray-300 dark:border-gray-600"
                  menu={{
                    items: [
                      { label: "Export CSV", key: "csv" },
                      { label: "Export PDF", key: "pdf" },
                    ],
                  }}
                >
                  <Button icon={<DownloadOutlined />}>Export</Button>
                </Dropdown>
                <Button
                  type="primary"
                  icon={<PlusOutlined />}
                  onClick={openAddModal}
                  className="bg-blue-600 hover:bg-blue-700 w-full sm:w-auto"
                >
                  Add Action
                </Button>
              </div>
              <div className="flex flex-col gap-4 lg:flex-row lg:justify-between lg:items-center">
                <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-5 gap-4 w-full lg:w-auto">
                  <Select
                    placeholder="Status"
                    className="w-full max-w-[160px]"
                    onChange={(value) =>
                      setAdvancedFilters((prev) => ({ ...prev, Status: value }))
                    }
                    allowClear
                    options={statusOptions.map((v) => ({ label: v, value: v }))}
                  />
                  <Select
                    placeholder="Unit Name"
                    className="w-full max-w-[160px]"
                    onChange={(value) =>
                      setAdvancedFilters((prev) => ({
                        ...prev,
                        unitName: value,
                      }))
                    }
                    allowClear
                    options={Array.from(
                      new Set(actions.map((a) => a.unitName))
                    ).map((value) => ({
                      label: value,
                      value: value,
                    }))}
                  />
                  <Select
                    placeholder="Priority"
                    className="w-full max-w-[160px]"
                    onChange={(value) =>
                      setAdvancedFilters((prev) => ({
                        ...prev,
                        priorityLevel: value,
                      }))
                    }
                    allowClear
                    options={priorityOptions.map((v) => ({
                      label: v,
                      value: v,
                    }))}
                  />
                  <Select
                    placeholder="Department"
                    className="w-full max-w-[160px]"
                    onChange={(value) =>
                      setAdvancedFilters((prev) => ({
                        ...prev,
                        department: value,
                      }))
                    }
                    allowClear
                    options={departmentOptions.map((v) => ({
                      label: v,
                      value: v,
                    }))}
                  />
                  <Select
                    placeholder="Action Type"
                    className="w-full max-w-[160px]"
                    onChange={(value) =>
                      setAdvancedFilters((prev) => ({
                        ...prev,
                        action_type: value,
                      }))
                    }
                    allowClear
                    options={actionTypeOptions.map((v) => ({
                      label: v,
                      value: v,
                    }))}
                  />
                </div>
                <div className="flex flex-col sm:flex-row flex-wrap gap-4 justify-start lg:justify-end w-full lg:w-auto">
                  <ColumnVisibilityToggle
                    columns={columnKeys}
                    labels={labelsMap}
                    storageKey="supplier_action_table_visibility"
                    onChange={handleVisibleColumnChange}
                  />
                  <DatePicker.RangePicker
                    value={dateRange}
                    onChange={(dates) =>
                      setDateRange(dates as [dayjs.Dayjs, dayjs.Dayjs] | null)
                    }
                    className="w-full sm:w-auto border-gray-300 dark:border-gray-600"
                  />
                  <Button
                    onClick={handleBulkDelete}
                    disabled={selectedRows.length === 0}
                    danger
                    className="w-full sm:w-auto"
                  >
                    Bulk Delete
                  </Button>
                  <div className="hidden sm:block">
                    <Switch
                      checkedChildren={<AppstoreOutlined />}
                      unCheckedChildren={<TableOutlined />}
                      checked={isGridView}
                      onChange={setIsGridView}
                      className="w-full sm:w-auto"
                    />
                  </div>
                </div>
              </div>

              {(advancedFilters.Status ||
                advancedFilters.unitName ||
                advancedFilters.priorityLevel ||
                advancedFilters.department ||
                advancedFilters.action_type) && (
                <div className="flex flex-wrap gap-2 mt-2">
                  {advancedFilters.Status && (
                    <Tag
                      closable
                      onClose={() =>
                        setAdvancedFilters((prev) => ({
                          ...prev,
                          Status: undefined,
                        }))
                      }
                    >
                      Status: {advancedFilters.Status}
                    </Tag>
                  )}
                  {advancedFilters.unitName && (
                    <Tag
                      closable
                      onClose={() =>
                        setAdvancedFilters((prev) => ({
                          ...prev,
                          unitName: undefined,
                        }))
                      }
                    >
                      Unit: {advancedFilters.unitName}
                    </Tag>
                  )}
                  {advancedFilters.priorityLevel && (
                    <Tag
                      closable
                      onClose={() =>
                        setAdvancedFilters((prev) => ({
                          ...prev,
                          priorityLevel: undefined,
                        }))
                      }
                    >
                      Priority: {advancedFilters.priorityLevel}
                    </Tag>
                  )}
                  {advancedFilters.department && (
                    <Tag
                      closable
                      onClose={() =>
                        setAdvancedFilters((prev) => ({
                          ...prev,
                          department: undefined,
                        }))
                      }
                    >
                      Department: {advancedFilters.department}
                    </Tag>
                  )}
                  {advancedFilters.action_type && (
                    <Tag
                      closable
                      onClose={() =>
                        setAdvancedFilters((prev) => ({
                          ...prev,
                          action_type: undefined,
                        }))
                      }
                    >
                      Action Type: {advancedFilters.action_type}
                    </Tag>
                  )}
                </div>
              )}
            </div>

            {isGridView || isMobile ? (
              <div className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 lg:grid-cols-3 gap-4 p-4">
                {paginatedActions.map((action) => (
                  <ActionCard
                    key={`${action.supplier_id}-${action.product_id}`}
                    action={action}
                  />
                ))}
              </div>
            ) : (
              <div className="overflow-auto">
                <DataTable<SupplierAction>
                  columns={filteredColumns}
                  dataSource={paginatedActions}
                  rowKey={(record) =>
                    `${record.supplier_id}-${record.product_id}`
                  }
                  scrollX="max-content"
                  className="mt-4"
                  pagination={false}
                  loading={loading}
                />
              </div>
            )}
            <div className="flex flex-col sm:flex-row justify-between items-center px-4 py-2 text-sm text-gray-600 dark:text-gray-300">
              <div className="mb-2 sm:mb-0">
                Showing <strong>{(currentPage - 1) * pageSize + 1}</strong> to{" "}
                <strong>
                  {Math.min(currentPage * pageSize, filteredActions.length)}
                </strong>{" "}
                of <strong>{filteredActions.length}</strong> actions
              </div>
              <Pagination
                current={currentPage}
                pageSize={pageSize}
                total={filteredActions.length}
                onChange={(page, size) => {
                  setCurrentPage(page);
                  setPageSize(size);
                }}
                showSizeChanger
                pageSizeOptions={["10", "30", "50", "100"]}
                size="small"
              />
            </div>
          </Card>
        </div>

        <motion.div
          initial={{ opacity: 0, scale: 0.95 }}
          animate={{ opacity: 1, scale: 1 }}
          exit={{ opacity: 0, scale: 0.95 }}
        >
          <Modal
            title={
              <div className="flex items-center gap-2 border-b pb-2 border-gray-300 dark:border-gray-600">
                {editingAction ? (
                  <>
                    <Edit2 className="w-5 h-5 text-blue-600" />
                    <span className="font-semibold text-gray-800 dark:text-gray-200">
                      Edit Supplier Action
                    </span>
                  </>
                ) : (
                  <>
                    <PlusSquare className="w-5 h-5 text-green-600" />
                    <span className="font-semibold text-gray-800 dark:text-gray-200">
                      Add Supplier Action
                    </span>
                  </>
                )}
              </div>
            }
            open={isModalOpen}
            onOk={handleFormSubmit}
            onCancel={() => {
              setIsModalOpen(false);
              form.resetFields();
              setFileList([]);
            }}
            destroyOnHidden
            width="95%" // responsive width
            className="!top-5 font-semibold"
            styles={{
              body: { maxHeight: "80vh", overflowY: "auto" }, // ✅ fixed line
            }}
          >
            <Form
              form={form}
              layout="vertical"
              className="w-full px-4 space-y-10"
            >
              {/* Supplier & Product Section */}
              <div>
                <h3 className="text-lg font-semibold text-gray-800 dark:text-gray-200 mb-4">
                  Supplier & Product
                </h3>
                <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-4">
                  <Form.Item
                    name="supplier_id"
                    label="Supplier"
                    rules={[
                      { required: true, message: "Supplier is required" },
                    ]}
                  >
                    <Select
                      showSearch
                      optionFilterProp="label"
                      options={suppliers.map((supplier) => ({
                        label: `${supplier.company_name} (${
                          supplier.contact_person || "N/A"
                        })`,
                        value: supplier.supplier_id,
                      }))}
                    />
                  </Form.Item>

                  <Form.Item
                    name="product_id"
                    label="Product"
                    rules={[{ required: true, message: "Product is required" }]}
                  >
                    <Select
                      showSearch
                      optionFilterProp="label"
                      options={products.map((product) => ({
                        label: `${product.product_name} (${
                          product.supplier_name || "N/A"
                        })`,
                        value: product.product_id,
                      }))}
                    />
                  </Form.Item>

                  <Form.Item
                    name="unitName"
                    label="Unit Name"
                    rules={[{ required: true }]}
                  >
                    <Input />
                  </Form.Item>

                  <Form.Item
                    name="materialDescription"
                    label="Material Description"
                  >
                    <Input />
                  </Form.Item>
                </div>
              </div>

              {/* Identification & CAPA Section */}
              <div>
                <h3 className="text-lg font-semibold text-gray-800 dark:text-gray-200 mb-4">
                  Identification & CAPA
                </h3>
                <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-4">
                  <Form.Item
                    name="invoiceNo"
                    label="Invoice No."
                    rules={[{ required: true }]}
                  >
                    <Input />
                  </Form.Item>

                  <Form.Item
                    name="supplierLotNo"
                    label="Supplier Lot No."
                    rules={[{ required: true }]}
                  >
                    <Input />
                  </Form.Item>

                  <Form.Item
                    name="dateIdentified"
                    label="Date Identified"
                    rules={[{ required: true }]}
                  >
                    <DatePicker className="w-full" />
                  </Form.Item>

                  <Form.Item
                    name="dateResolved"
                    label="Date Resolved"
                    rules={[{ required: true }]}
                  >
                    <DatePicker className="w-full" />
                  </Form.Item>

                  <div className="col-span-full">
                    <Form.Item
                      name="nonConformanceDetails"
                      label="Non Conformance Details"
                      rules={[{ required: true }]}
                    >
                      <Input.TextArea rows={2} />
                    </Form.Item>
                  </div>

                  <Form.Item
                    name="capaNo"
                    label="CAPA No."
                    rules={[{ required: true }]}
                  >
                    <Input />
                  </Form.Item>
                </div>
              </div>

              {/* Status & Priority Section */}
              <div>
                <h3 className="text-lg font-semibold text-gray-800 dark:text-gray-200 mb-4">
                  Status & Responsibility
                </h3>
                <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-4">
                  <Form.Item
                    name="Status"
                    label="Status"
                    rules={[{ required: true }]}
                  >
                    <Select
                      options={statusOptions.map((v) => ({
                        label: v,
                        value: v,
                      }))}
                    />
                  </Form.Item>

                  <Form.Item
                    name="priorityLevel"
                    label="Priority Level"
                    rules={[{ required: true }]}
                  >
                    <Select
                      options={priorityOptions.map((v) => ({
                        label: v,
                        value: v,
                      }))}
                    />
                  </Form.Item>

                  <Form.Item
                    name="department"
                    label="Department"
                    rules={[{ required: true }]}
                  >
                    <Select
                      options={departmentOptions.map((v) => ({
                        label: v,
                        value: v,
                      }))}
                    />
                  </Form.Item>

                  <Form.Item
                    name="action_type"
                    label="Action Type"
                    rules={[{ required: true }]}
                  >
                    <Select
                      options={actionTypeOptions.map((v) => ({
                        label: v,
                        value: v,
                      }))}
                    />
                  </Form.Item>

                  <div className="col-span-full">
                    <Form.Item
                      name="description"
                      label="Description"
                      rules={[{ required: true }]}
                    >
                      <Input.TextArea rows={2} />
                    </Form.Item>
                  </div>

                  <div className="col-span-full">
                    <Form.Item
                      name="root_cause_analysis"
                      label="Root Cause Analysis"
                      rules={[{ required: true }]}
                    >
                      <Input.TextArea rows={2} />
                    </Form.Item>
                  </div>
                </div>
              </div>

              {/* Action & Dates Section */}
              <div>
                <h3 className="text-lg font-semibold text-gray-800 dark:text-gray-200 mb-4">
                  Action & Dates
                </h3>
                <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-4">
                  <Form.Item
                    name="responsible_person"
                    label="Responsible Person"
                    rules={[{ required: true }]}
                  >
                    <Input />
                  </Form.Item>

                  <Form.Item
                    name="targetdate"
                    label="Target Date"
                    rules={[{ required: true }]}
                  >
                    <DatePicker className="w-full" />
                  </Form.Item>

                  <Form.Item
                    name="duedate"
                    label="Due Date"
                    rules={[{ required: true }]}
                  >
                    <DatePicker className="w-full" />
                  </Form.Item>

                  <div className="col-span-full">
                    <Form.Item
                      name="action_taken"
                      label="Action Taken"
                      rules={[{ required: true }]}
                    >
                      <Input.TextArea rows={2} />
                    </Form.Item>
                  </div>
                </div>
              </div>

              {/* File Upload Section */}
              <div>
                <h3 className="text-lg font-semibold text-gray-800 dark:text-gray-200 mb-4">
                  Evidence File
                </h3>
                <Form.Item className="col-span-full lg:col-span-2">
                  <Upload
                    fileList={fileList}
                    onChange={({ fileList: newFileList }) =>
                      setFileList(newFileList)
                    }
                    beforeUpload={() => false}
                    maxCount={1}
                  >
                    <Button icon={<FileOutlined />}>Upload Evidence</Button>
                  </Upload>
                </Form.Item>
              </div>
            </Form>
          </Modal>
        </motion.div>

        <Modal
          title="Supplier Action Details"
          open={!!viewingAction}
          onCancel={() => setViewingAction(null)}
          footer={null}
          className="w-full max-w-[90vw] sm:max-w-[500px] top-4"
        >
          {viewingAction && (
            <div className="space-y-2 text-sm sm:text-base">
              <p>
                <strong>Supplier:</strong>{" "}
                <span className="text-green-600">
                  {getSupplierDisplay(viewingAction.supplier_id)}
                </span>
              </p>
              <p>
                <strong>Product:</strong>{" "}
                {getProductDisplay(viewingAction.product_id)}
              </p>
              <p>
                <strong>Unit Name:</strong> {viewingAction.unitName}
              </p>
              <p>
                <strong>Material:</strong>{" "}
                {viewingAction.materialDescription ?? "N/A"}
              </p>
              <p>
                <strong>Invoice No.:</strong> {viewingAction.invoiceNo}
              </p>
              <p>
                <strong>Supplier Lot No.:</strong> {viewingAction.supplierLotNo}
              </p>
              <p>
                <strong>Date Identified:</strong>{" "}
                {dayjs(viewingAction.dateIdentified).format("YYYY-MM-DD")}
              </p>
              <p>
                <strong>Date Resolved:</strong>{" "}
                {dayjs(viewingAction.dateResolved).format("YYYY-MM-DD")}
              </p>
              <p>
                <strong>Non Conformance:</strong>{" "}
                {viewingAction.nonConformanceDetails}
              </p>
              <p>
                <strong>CAPA No.:</strong> {viewingAction.capaNo}
              </p>
              <p>
                <strong>Status:</strong> {viewingAction.Status}
              </p>
              <p>
                <strong>Priority:</strong> {viewingAction.priorityLevel}
              </p>
              <p>
                <strong>Department:</strong> {viewingAction.department}
              </p>
              <p>
                <strong>Action Type:</strong> {viewingAction.action_type}
              </p>
              <p>
                <strong>Description:</strong> {viewingAction.description}
              </p>
              <p>
                <strong>Root Cause:</strong> {viewingAction.root_cause_analysis}
              </p>
              <p>
                <strong>Responsible:</strong> {viewingAction.responsible_person}
              </p>
              <p>
                <strong>Target Date:</strong>{" "}
                {dayjs(viewingAction.targetdate).format("YYYY-MM-DD")}
              </p>
              <p>
                <strong>Due Date:</strong>{" "}
                {dayjs(viewingAction.duedate).format("YYYY-MM-DD")}
              </p>
              <p>
                <strong>Action Taken:</strong> {viewingAction.action_taken}
              </p>
              <p>
                <strong>Evidence:</strong>{" "}
                {viewingAction.evidence_file ? (
                  <a
                    href={viewingAction.evidence_file as string}
                    target="_blank"
                    rel="noopener noreferrer"
                  >
                    View Evidence
                  </a>
                ) : (
                  "None"
                )}
              </p>
              <p>
                <strong>Created:</strong>{" "}
                {dayjs(viewingAction.created_at).format("YYYY-MM-DD")}
              </p>
              <p>
                <strong>Updated:</strong>{" "}
                {viewingAction.updated_at
                  ? dayjs(viewingAction.updated_at).format("YYYY-MM-DD")
                  : "N/A"}
              </p>
            </div>
          )}
        </Modal>
        <ConfirmDialog
          open={confirmDialog.open}
          title={confirmDialog.title}
          message={confirmDialog.message}
          onClose={() => setConfirmDialog({ ...confirmDialog, open: false })}
          onConfirm={confirmDialog.onConfirm}
        />
      </div>
    </>
  );
}
