"use client";

import React, { useEffect, useState } from "react";
import {
  getPurchaseRequisitions,
  PurchaseRequisition,
} from "../api/purchaserequisition/purchaserequisition";
import {
  getAllPurchaseOrders,
  PurchaseOrder,
} from "../api/purchaseorder/purchaseorder";
import Chart from "react-apexcharts";
import { ApexOptions } from "apexcharts";
import ReactApexChart from "react-apexcharts";
import {
  CheckCircle,
  Clock,
  FileText,
  ShoppingCart,
  XCircle,
  Calendar,
  History,
  BarChart2,
  Package,
  LayoutGrid,
  FileCheck,
  PlusCircle,
} from "lucide-react";
import CountUp from "react-countup";
import { FaInfoCircle } from "react-icons/fa";
import { useRouter } from "next/navigation";
const statusOptions = [
  "Draft",
  "Sent",
  "Confirmed",
  "Delivered",
  "Closed",
  "Cancelled",
] as const;

export default function PurchasePage() {
  const [loading, setLoading] = useState(true);
  const [requisitions, setRequisitions] = useState<PurchaseRequisition[]>([]);
  const [orders, setOrders] = useState<PurchaseOrder[]>([]);
  const [timeFilter, setTimeFilter] = useState("Last 30 Days");
  const router = useRouter();
  {
    /** for fetching purchase requisitions and orders */
  }
  useEffect(() => {
    const fetchData = async () => {
      try {
        const [reqData, orderData] = await Promise.all([
          getPurchaseRequisitions(),
          getAllPurchaseOrders(),
        ]);

        setRequisitions(
          reqData.sort(
            (a, b) =>
              new Date(b.created_at ?? "").getTime() -
              new Date(a.created_at ?? "").getTime()
          )
        );

        setOrders(
          orderData
            .sort(
              (a, b) =>
                new Date(b.po_date ?? "").getTime() -
                new Date(a.po_date ?? "").getTime()
            )
            .map((order) => ({
              ...order,
              supported_file:
                typeof order.supported_file === "string"
                  ? null
                  : order.supported_file,
            }))
        );
      } catch (err) {
        console.error("Error fetching purchase data:", err);
      } finally {
        setLoading(false);
      }
    };
    fetchData();
  }, []);

  {
    /** for charting the status of purchase orders */
  }
  const statusCounts = statusOptions.map(
    (status) => orders.filter((o) => o.Status === status).length
  );

  const chartOptions: ApexOptions = {
    chart: {
      type: "pie", // ⬅️ type-safe
      height: 300,
    },
    labels: [...statusOptions],
    legend: {
      position: "bottom",
      labels: {
        colors: "#9CA3AF", // Tailwind gray-400
      },
    },
    responsive: [
      {
        breakpoint: 640,
        options: {
          chart: { height: 250 },
          legend: { position: "bottom" },
        },
      },
    ],
  };

  {
    /** for charting the status of purchase requisitions */
  }
  const counts = requisitions.reduce(
    (acc, req) => {
      if (req.Status === "Pending") acc.Pending++;
      if (req.Status === "Approved") acc.Approved++;
      if (req.Status === "Rejected") acc.Rejected++;
      return acc;
    },
    { Pending: 0, Approved: 0, Rejected: 0 }
  );

  const requisitionChart: ApexOptions = {
    chart: {
      type: "donut",
      height: 240,
      toolbar: { show: false },
    },
    colors: ["#F59E0B", "#3B82F6", "#EF4444"], // Pending=yellow, Approved=blue, Rejected=red
    labels: ["Pending", "Approved", "Rejected"],
    plotOptions: {
      pie: {
        donut: {
          size: "60%",
          labels: { show: false },
        },
      },
    },
    stroke: {
      lineCap: "round",
      show: true,
      width: 0,
      colors: ["#fff"],
    },
    dataLabels: { enabled: false },
    legend: {
      show: false,
    },
    responsive: [
      {
        breakpoint: 480,
        options: { chart: { height: 180 } },
      },
    ],
  };

  const requisitionSeries = [counts.Pending, counts.Approved, counts.Rejected];

  {
    /** for calculating month-over-month changes in requisition statuses */
  }
  const startOfMonth = new Date(
    new Date().getFullYear(),
    new Date().getMonth(),
    1
  );
  const startOfLastMonth = new Date(
    new Date().getFullYear(),
    new Date().getMonth() - 1,
    1
  );
  const endOfLastMonth = new Date(
    new Date().getFullYear(),
    new Date().getMonth(),
    0
  );

  const thisMonthReqs = requisitions.filter(
    (r) => r.created_at && new Date(r.created_at) >= startOfMonth
  );
  const lastMonthReqs = requisitions.filter(
    (r) =>
      r.created_at &&
      new Date(r.created_at) >= startOfLastMonth &&
      new Date(r.created_at) <= endOfLastMonth
  );

  const thisMonthCounts = thisMonthReqs.reduce(
    (acc, req) => {
      if (req.Status === "Pending") acc.Pending++;
      if (req.Status === "Approved") acc.Approved++;
      if (req.Status === "Rejected") acc.Rejected++;
      return acc;
    },
    { Pending: 0, Approved: 0, Rejected: 0 }
  );

  const getChange = (thisVal: number, lastVal: number) => {
    if (lastVal === 0) return thisVal > 0 ? "+100%" : "0%";
    const diff = ((thisVal - lastVal) / lastVal) * 100;
    return `${diff > 0 ? "+" : ""}${diff.toFixed(1)}%`;
  };

  {
    /** for calculating month-over-month changes in purchaseorder statuses */
  }

  const thisMonthOrders = orders.filter(
    (o) => o.po_date && new Date(o.po_date) >= startOfMonth
  );
  const lastMonthOrders = orders.filter(
    (o) =>
      o.po_date &&
      new Date(o.po_date) >= startOfLastMonth &&
      new Date(o.po_date) <= endOfLastMonth
  );

  const thisMonthOrderCounts = thisMonthOrders.reduce(
    (acc, order) => {
      if (order.Status === "Draft") acc.Draft++;
      if (order.Status === "Sent") acc.Sent++;
      if (order.Status === "Confirmed") acc.Confirmed++;
      if (order.Status === "Delivered") acc.Delivered++;
      if (order.Status === "Closed") acc.Closed++;
      if (order.Status === "Cancelled") acc.Cancelled++;
      return acc;
    },
    { Draft: 0, Sent: 0, Confirmed: 0, Delivered: 0, Closed: 0, Cancelled: 0 }
  );

  {
    /** group requisitions by department */
  }
  const deptCounts: Record<string, number> = {};
  requisitions.forEach((req) => {
    if (req.department) {
      deptCounts[req.department] = (deptCounts[req.department] || 0) + 1;
    }
  });

  const deptSeries = [
    {
      name: "Requisitions",
      data: Object.values(deptCounts),
    },
  ];

  const deptChart: ApexOptions = {
    chart: {
      type: "bar",
      height: 275,
      toolbar: { show: false },
    },
    plotOptions: {
      bar: {
        borderRadius: 6,
        columnWidth: "45%",
      },
    },
    colors: ["#10B981"],
    dataLabels: {
      enabled: true,
      style: { fontSize: "12px", fontWeight: "bold", colors: ["#ffffff"] }, // Force white
    },
    xaxis: {
      categories: Object.keys(deptCounts),
      labels: {
        style: {
          fontSize: "12px",
          fontWeight: 500,
          colors: Array(Object.keys(deptCounts).length).fill("#ffffff"), // 👈 force white on all labels
        },
      },
    },
    yaxis: {
      min: 0,
      tickAmount: 5,
      labels: {
        style: {
          fontSize: "12px",
          colors: ["#ffffff"], // 👈 force white
        },
      },
    },
    grid: {
      borderColor: "#374151", // dark gray grid for dark mode
      strokeDashArray: 4,
    },
  };

  {
    /** filter upcoming purchase orders */
  }
  const upcomingOrders = orders.filter(
    (o) =>
      o.expected_delivery_date &&
      new Date(o.expected_delivery_date) >= new Date()
  );

  {
    /** Loading State */
  }
  if (loading) {
    return (
      <div className="flex justify-center items-center h-[70vh]">
        <div className="w-8 h-8 border-4 border-gray-300 border-t-blue-500 rounded-full animate-spin"></div>
      </div>
    );
  }

  return (
    <div className="p-6 space-y-8">
         <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between bg-white dark:bg-gray-800 p-6 rounded-2xl shadow-md mb-6">
      {/* Title + Icon */}
      <div className="flex items-center gap-3">
        <ShoppingCart className="w-10 h-10 text-green-600" />
        <div>
          <h1 className="text-3xl font-bold text-gray-800 dark:text-gray-100">
            Purchase Dashboard
          </h1>
          <p className="text-sm text-gray-500 dark:text-gray-300 mt-1">
            Overview of your purchase requisitions and orders
          </p>
        </div>
      </div>

      {/* Optional Action Button
      <button
        onClick={() => router.push("/purchase/purchaserequisition")}
        className="mt-4 sm:mt-0 flex items-center gap-1 px-4 py-2 bg-green-600 hover:bg-green-700 text-white text-sm font-medium rounded-lg shadow transition"
      >
        <PlusCircle className="w-4 h-4" />
        Add Purchase
      </button> */}
    </div>
      {/* KPI Cards for purchase requisitions */}
      {/* Section Header */}
      <h2 className="text-xl font-bold text-gray-800 dark:text-gray-100 mb-4 flex items-center gap-2">
        <Calendar className="w-5 h-5 text-blue-800" />
        Monthly Purchase Requisitions
        <div className="relative inline-block group">
          {/* Info Icon */}
          <span className="cursor-pointer text-blue-600">
            <FaInfoCircle className="w-4 h-4 text-gray-500 cursor-pointer" />
          </span>

          {/* Tooltip */}
          <div className="absolute bottom-full left-1/2 transform -translate-x-1/2 mb-2 w-100 bg-black text-white text-sm text-center rounded px-2 py-1 opacity-0 invisible group-hover:visible group-hover:opacity-100 transition-opacity duration-300">
            Shows the total number of purchase requisitions created this month
            along with their status.
            {/* Arrow */}
            <div className="absolute top-full left-1/2 transform -translate-x-1/2 w-2 h-2 bg-black rotate-45"></div>
          </div>
        </div>
      </h2>
      <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-6 mb-8">
        {/* Total Requisitions */}
        <div className="bg-blue-50 dark:bg-blue-900/30 rounded-2xl shadow-md p-5 border border-blue-200 dark:border-blue-700 hover:shadow-xl hover:scale-105 transition transform">
          <div className="flex items-center justify-between">
            <FileText className="w-6 h-6 text-blue-600 dark:text-blue-400" />
            <span className="text-xs font-medium text-blue-600 dark:text-blue-400">
              {getChange(thisMonthReqs.length, lastMonthReqs.length)}
            </span>
          </div>
          <h3 className="mt-2 text-sm font-medium text-blue-700 dark:text-blue-300">
            Total Requisitions (This Month)
          </h3>
          <p className="mt-2 text-3xl font-bold text-blue-800 dark:text-blue-200">
            <CountUp end={thisMonthReqs.length} duration={1.5} />
          </p>
        </div>

        {/* Pending */}
        <div className="bg-yellow-50 dark:bg-yellow-900/30 rounded-2xl shadow-md p-5 border border-yellow-200 dark:border-yellow-700 hover:shadow-xl hover:scale-105 transition transform">
          <div className="flex items-center justify-between">
            <Clock className="w-6 h-6 text-yellow-600 dark:text-yellow-400" />
            <span className="text-xs font-medium text-yellow-600 dark:text-yellow-400">
              {getChange(
                thisMonthCounts.Pending,
                lastMonthReqs.filter((r) => r.Status === "Pending").length
              )}
            </span>
          </div>
          <h3 className="mt-2 text-sm font-medium text-yellow-700 dark:text-yellow-300">
            Pending
          </h3>
          <p className="mt-2 text-3xl font-bold text-yellow-700 dark:text-yellow-300">
            <CountUp end={thisMonthCounts.Pending} duration={1.5} />
          </p>
        </div>

        {/* Approved */}
        <div className="bg-green-50 dark:bg-green-900/30 rounded-2xl shadow-md p-5 border border-green-200 dark:border-green-700 hover:shadow-xl hover:scale-105 transition transform">
          <div className="flex items-center justify-between">
            <CheckCircle className="w-6 h-6 text-green-600 dark:text-green-400" />
            <span className="text-xs font-medium text-green-600 dark:text-green-400">
              {getChange(
                thisMonthCounts.Approved,
                lastMonthReqs.filter((r) => r.Status === "Approved").length
              )}
            </span>
          </div>
          <h3 className="mt-2 text-sm font-medium text-green-700 dark:text-green-300">
            Approved
          </h3>
          <p className="mt-2 text-3xl font-bold text-green-700 dark:text-green-300">
            <CountUp end={thisMonthCounts.Approved} duration={1.5} />
          </p>
        </div>

        {/* Rejected */}
        <div className="bg-red-50 dark:bg-red-900/30 rounded-2xl shadow-md p-5 border border-red-200 dark:border-red-700 hover:shadow-xl hover:scale-105 transition transform">
          <div className="flex items-center justify-between">
            <XCircle className="w-6 h-6 text-red-600 dark:text-red-400" />
            <span className="text-xs font-medium text-red-600 dark:text-red-400">
              {getChange(
                thisMonthCounts.Rejected,
                lastMonthReqs.filter((r) => r.Status === "Rejected").length
              )}
            </span>
          </div>
          <h3 className="mt-2 text-sm font-medium text-red-700 dark:text-red-300">
            Rejected
          </h3>
          <p className="mt-2 text-3xl font-bold text-red-700 dark:text-red-300">
            <CountUp end={thisMonthCounts.Rejected} duration={1.5} />
          </p>
        </div>
      </div>
      {/* Dashboard Grid */}
      <div className="grid grid-cols-1  gap-6">
        {/* Requisitions Card */}
        <div className="bg-white dark:bg-gray-900 rounded-2xl shadow-lg p-5 border border-gray-200 dark:border-gray-700">
          <div className="flex items-center justify-between mb-4">
            {/* Header + Tooltip */}
            <h2 className="text-lg font-semibold text-gray-800 dark:text-gray-100 flex items-center gap-2">
              <History className="w-5 h-5 text-purple-600" />
              Recent Purchase Requisitions
              <div className="relative inline-block group">
                {/* Info Icon */}
                <span className="cursor-pointer text-gray-500">
                  <FaInfoCircle className="w-4 h-4" />
                </span>

                {/* Tooltip */}
                <div className="absolute bottom-full left-1/2 transform -translate-x-1/2 mb-2 w-64 bg-black text-white text-sm text-center rounded px-2 py-1 opacity-0 invisible group-hover:visible group-hover:opacity-100 transition-opacity duration-300">
                  Shows the list of the most recent purchase requisitions along
                  with their current status.
                  {/* Arrow */}
                  <div className="absolute top-full left-1/2 transform -translate-x-1/2 w-2 h-2 bg-black rotate-45"></div>
                </div>
              </div>
            </h2>

            {/* Add Button */}
             <button
      onClick={() => router.push("/purchase/purchaserequisition")}
      className="flex items-center gap-1 px-4 py-2 bg-green-600 hover:bg-green-700 text-white text-sm font-medium rounded-lg shadow transition"
    >
      <PlusCircle className="w-4 h-4" />
      Add Purchase
    </button>
          </div>
          <div className="overflow-x-auto">
            <table className="min-w-full text-sm">
              <thead className="bg-gray-100 dark:bg-gray-700 text-gray-700 dark:text-gray-200">
                <tr>
                  <th className="px-3 py-2 text-left">Product</th>
                  <th className="px-3 py-2 text-left">Dept</th>
                  <th className="px-3 py-2 text-left">Requester</th>
                  <th className="px-3 py-2 text-left">Qty</th>
                  <th className="px-3 py-2 text-left">Cost</th>
                  <th className="px-3 py-2 text-left">Priority</th>
                  <th className="px-3 py-2 text-left">Status</th>
                  <th className="px-3 py-2 text-left">Req Date</th>
                </tr>
              </thead>
              <tbody className="divide-y divide-gray-200 dark:divide-gray-700">
                {requisitions.length > 0 ? (
                  requisitions.slice(0, 5).map((req) => (
                    <tr
                      key={req.purchase_requisition_id}
                      className="hover:bg-gray-50 dark:hover:bg-gray-800 transition"
                    >
                      <td className="px-3 py-2">{req.product_name ?? "-"}</td>
                      <td className="px-3 py-2">{req.department ?? "-"}</td>
                      <td className="px-3 py-2">
                        {req.requested_by_name ?? "-"}
                      </td>
                      <td className="px-3 py-2">{req.quantity ?? 0}</td>
                      <td className="px-3 py-2">
                        ₹{req.estimated_cost ?? "-"}
                      </td>

                      {/* Priority Tag */}
                      <td className="px-3 py-2">
                        <span
                          className={`px-2 py-1 rounded-full text-xs font-medium 
                    ${
                      req.priority === "High"
                        ? "bg-red-100 text-red-700 dark:bg-red-900/40 dark:text-red-300"
                        : req.priority === "Medium"
                        ? "bg-yellow-100 text-yellow-700 dark:bg-yellow-900/40 dark:text-yellow-300"
                        : "bg-green-100 text-green-700 dark:bg-green-900/40 dark:text-green-300"
                    }`}
                        >
                          {req.priority ?? "-"}
                        </span>
                      </td>

                      {/* Status Tag */}
                      <td className="px-3 py-2">
                        <span
                          className={`px-2 py-1 rounded-full text-xs font-medium 
                    ${
                      req.Status === "Approved"
                        ? "bg-green-100 text-green-700 dark:bg-green-900/40 dark:text-green-300"
                        : req.Status === "Pending"
                        ? "bg-yellow-100 text-yellow-700 dark:bg-yellow-900/40 dark:text-yellow-300"
                        : "bg-red-100 text-red-700 dark:bg-red-900/40 dark:text-red-300"
                    }`}
                        >
                          {req.Status ?? "-"}
                        </span>
                      </td>

                      <td className="px-3 py-2">
                        {req.purchase_requisition_date
                          ? new Date(
                              req.purchase_requisition_date
                            ).toLocaleDateString()
                          : "-"}
                      </td>
                    </tr>
                  ))
                ) : (
                  <tr>
                    <td
                      colSpan={8}
                      className="px-4 py-3 text-center text-gray-500 dark:text-gray-400"
                    >
                      No requisitions found.
                    </td>
                  </tr>
                )}
              </tbody>
            </table>
          </div>
        </div>

        {/* Orders Card */}
      </div>
      {/* KPI Cards for purchase orders */}
      {/* Section Header */}
      <h2 className="text-xl font-bold text-gray-800 dark:text-gray-100 mb-4 flex items-center gap-2">
        <Calendar className="w-5 h-5 text-blue-800" />
        Monthly Purchase Orders
        <div className="relative inline-block group">
          {/* Info Icon */}
          <span className="cursor-pointer text-blue-600">
            <FaInfoCircle className="w-4 h-4 text-gray-500 cursor-pointer" />
          </span>

          {/* Tooltip */}
          <div className="absolute bottom-full left-1/2 transform -translate-x-1/2 mb-2 w-100 bg-black text-white text-sm text-center rounded px-2 py-1 opacity-0 invisible group-hover:visible group-hover:opacity-100 transition-opacity duration-300">
            Shows the total number of purchase orders created this month along
            with their status.
            {/* Arrow */}
            <div className="absolute top-full left-1/2 transform -translate-x-1/2 w-2 h-2 bg-black rotate-45"></div>
          </div>
        </div>
      </h2>
      <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 xl:grid-cols-6 gap-6 mb-8">
        {/* Draft */}
        <div className="bg-gray-50 dark:bg-gray-900/40 rounded-2xl shadow-md p-5 border border-gray-200 dark:border-gray-700 hover:shadow-xl hover:scale-105 transition transform">
          <div className="flex items-center justify-between">
            <FileText className="w-6 h-6 text-gray-600 dark:text-gray-400" />
            <span className="text-xs font-medium text-gray-600 dark:text-gray-400">
              {getChange(
                thisMonthOrderCounts.Draft,
                lastMonthOrders.filter((o) => o.Status === "Draft").length
              )}
            </span>
          </div>
          <h3 className="mt-2 text-sm font-medium text-gray-700 dark:text-gray-300">
            Draft
          </h3>
          <p className="mt-2 text-3xl font-bold text-gray-800 dark:text-gray-100">
            <CountUp end={thisMonthOrderCounts.Draft} duration={1.5} />
          </p>
        </div>

        {/* Sent */}
        <div className="bg-blue-50 dark:bg-blue-900/30 rounded-2xl shadow-md p-5 border border-blue-200 dark:border-blue-700 hover:shadow-xl hover:scale-105 transition transform">
          <div className="flex items-center justify-between">
            <Clock className="w-6 h-6 text-blue-600 dark:text-blue-400" />
            <span className="text-xs font-medium text-blue-600 dark:text-blue-400">
              {getChange(
                thisMonthOrderCounts.Sent,
                lastMonthOrders.filter((o) => o.Status === "Sent").length
              )}
            </span>
          </div>
          <h3 className="mt-2 text-sm font-medium text-blue-700 dark:text-blue-300">
            Sent
          </h3>
          <p className="mt-2 text-3xl font-bold text-blue-800 dark:text-blue-200">
            <CountUp end={thisMonthOrderCounts.Sent} duration={1.5} />
          </p>
        </div>

        {/* Confirmed */}
        <div className="bg-green-50 dark:bg-green-900/30 rounded-2xl shadow-md p-5 border border-green-200 dark:border-green-700 hover:shadow-xl hover:scale-105 transition transform">
          <div className="flex items-center justify-between">
            <CheckCircle className="w-6 h-6 text-green-600 dark:text-green-400" />
            <span className="text-xs font-medium text-green-600 dark:text-green-400">
              {getChange(
                thisMonthOrderCounts.Confirmed,
                lastMonthOrders.filter((o) => o.Status === "Confirmed").length
              )}
            </span>
          </div>
          <h3 className="mt-2 text-sm font-medium text-green-700 dark:text-green-300">
            Confirmed
          </h3>
          <p className="mt-2 text-3xl font-bold text-green-700 dark:text-green-300">
            <CountUp end={thisMonthOrderCounts.Confirmed} duration={1.5} />
          </p>
        </div>

        {/* Delivered */}
        <div className="bg-purple-50 dark:bg-purple-900/30 rounded-2xl shadow-md p-5 border border-purple-200 dark:border-purple-700 hover:shadow-xl hover:scale-105 transition transform">
          <div className="flex items-center justify-between">
            <CheckCircle className="w-6 h-6 text-purple-600 dark:text-purple-400" />
            <span className="text-xs font-medium text-purple-600 dark:text-purple-400">
              {getChange(
                thisMonthOrderCounts.Delivered,
                lastMonthOrders.filter((o) => o.Status === "Delivered").length
              )}
            </span>
          </div>
          <h3 className="mt-2 text-sm font-medium text-purple-700 dark:text-purple-300">
            Delivered
          </h3>
          <p className="mt-2 text-3xl font-bold text-purple-700 dark:text-purple-300">
            <CountUp end={thisMonthOrderCounts.Delivered} duration={1.5} />
          </p>
        </div>

        {/* Closed */}
        <div className="bg-indigo-50 dark:bg-indigo-900/30 rounded-2xl shadow-md p-5 border border-indigo-200 dark:border-indigo-700 hover:shadow-xl hover:scale-105 transition transform">
          <div className="flex items-center justify-between">
            <CheckCircle className="w-6 h-6 text-indigo-600 dark:text-indigo-400" />
            <span className="text-xs font-medium text-indigo-600 dark:text-indigo-400">
              {getChange(
                thisMonthOrderCounts.Closed,
                lastMonthOrders.filter((o) => o.Status === "Closed").length
              )}
            </span>
          </div>
          <h3 className="mt-2 text-sm font-medium text-indigo-700 dark:text-indigo-300">
            Closed
          </h3>
          <p className="mt-2 text-3xl font-bold text-indigo-700 dark:text-indigo-300">
            <CountUp end={thisMonthOrderCounts.Closed} duration={1.5} />
          </p>
        </div>

        {/* Cancelled */}
        <div className="bg-red-50 dark:bg-red-900/30 rounded-2xl shadow-md p-5 border border-red-200 dark:border-red-700 hover:shadow-xl hover:scale-105 transition transform">
          <div className="flex items-center justify-between">
            <XCircle className="w-6 h-6 text-red-600 dark:text-red-400" />
            <span className="text-xs font-medium text-red-600 dark:text-red-400">
              {getChange(
                thisMonthOrderCounts.Cancelled,
                lastMonthOrders.filter((o) => o.Status === "Cancelled").length
              )}
            </span>
          </div>
          <h3 className="mt-2 text-sm font-medium text-red-700 dark:text-red-300">
            Cancelled
          </h3>
          <p className="mt-2 text-3xl font-bold text-red-700 dark:text-red-300">
            <CountUp end={thisMonthOrderCounts.Cancelled} duration={1.5} />
          </p>
        </div>
      </div>
      {/* purchase order card */}
      {/* Placement of the recent purchase order is modified */}
      <div>
        <div className="bg-white dark:bg-gray-900 rounded-2xl shadow-lg p-5 border border-gray-200 dark:border-gray-700">
          <div className="flex items-center justify-between mb-4">
      {/* Header + Tooltip */}
      <h2 className="text-lg font-semibold text-gray-800 dark:text-gray-100 flex items-center gap-2">
        <Clock className="w-5 h-5 text-purple-600" />
        Recent Purchase Orders
        <div className="relative inline-block group">
          {/* Info Icon */}
          <span className="text-gray-500 cursor-pointer">
            <FaInfoCircle className="w-4 h-4" />
          </span>

          {/* Tooltip */}
          <div className="absolute bottom-full left-1/2 transform -translate-x-1/2 mb-2 w-64 bg-black text-white text-xs text-center rounded px-2 py-1 opacity-0 invisible group-hover:visible group-hover:opacity-100 transition-opacity duration-300 z-10">
            Displays the list of the most recent purchase orders along with their current status.
            {/* Arrow */}
            <div className="absolute top-full left-1/2 transform -translate-x-1/2 w-2 h-2 bg-black rotate-45"></div>
          </div>
        </div>
      </h2>

      {/* Add Button */}
      <button
        onClick={() => router.push("/purchase/purchaseorder")}
        className="flex items-center gap-1 px-4 py-2 bg-green-600 hover:bg-green-700 text-white text-sm font-medium rounded-lg shadow transition"
      >
        <PlusCircle className="w-4 h-4" />
        Add Purchase
      </button>
    </div>

          <div className="overflow-x-auto">
            <table className="min-w-full text-sm">
              <thead className="bg-gray-100 dark:bg-gray-700 text-gray-700 dark:text-gray-200">
                <tr>
                  <th className="px-3 py-2 text-left">Supplier</th>
                  <th className="px-3 py-2 text-left">PO #</th>
                  <th className="px-3 py-2 text-left">Status</th>
                  <th className="px-3 py-2 text-left">Approval</th>
                  <th className="px-3 py-2 text-left">Total</th>
                  <th className="px-3 py-2 text-left">PO Date</th>
                  <th className="px-3 py-2 text-left">Delivery Date</th>
                </tr>
              </thead>
              <tbody className="divide-y divide-gray-200 dark:divide-gray-700">
                {orders.length > 0 ? (
                  orders.slice(0, 5).map((order, i) => (
                    <tr
                      key={i}
                      className="hover:bg-gray-50 dark:hover:bg-gray-800 transition"
                    >
                      <td className="px-3 py-2">
                        {order.supplier_name ?? "-"}
                      </td>
                      <td className="px-3 py-2">
                        {order.purchase_requisition_id ?? "-"}
                      </td>

                      {/* Status Tag */}
                      <td className="px-3 py-2">
                        <span
                          className={`px-2 py-1 rounded-full text-xs font-medium 
                    ${
                      order.Status === "Confirmed"
                        ? "bg-green-100 text-green-700 dark:bg-green-900/40 dark:text-green-300"
                        : order.Status === "Sent"
                        ? "bg-blue-100 text-blue-700 dark:bg-blue-900/40 dark:text-blue-300"
                        : order.Status === "Draft"
                        ? "bg-gray-100 text-gray-700 dark:bg-gray-800/40 dark:text-gray-300"
                        : "bg-red-100 text-red-700 dark:bg-red-900/40 dark:text-red-300"
                    }`}
                        >
                          {order.Status ?? "-"}
                        </span>
                      </td>

                      {/* Approval Tag */}
                      <td className="px-3 py-2">
                        <span
                          className={`px-2 py-1 rounded-full text-xs font-medium 
                    ${
                      order.Approve_status === "Approved"
                        ? "bg-green-100 text-green-700 dark:bg-green-900/40 dark:text-green-300"
                        : order.Approve_status === "Pending"
                        ? "bg-yellow-100 text-yellow-700 dark:bg-yellow-900/40 dark:text-yellow-300"
                        : "bg-red-100 text-red-700 dark:bg-red-900/40 dark:text-red-300"
                    }`}
                        >
                          {order.Approve_status ?? "-"}
                        </span>
                      </td>

                      <td className="px-3 py-2">
                        ₹{order.total_amount ?? "-"}
                      </td>
                      <td className="px-3 py-2">
                        {order.po_date
                          ? new Date(order.po_date).toLocaleDateString()
                          : "-"}
                      </td>
                      <td className="px-3 py-2">
                        {order.expected_delivery_date
                          ? new Date(
                              order.expected_delivery_date
                            ).toLocaleDateString()
                          : "-"}
                      </td>
                    </tr>
                  ))
                ) : (
                  <tr>
                    <td
                      colSpan={7}
                      className="px-4 py-3 text-center text-gray-500 dark:text-gray-400"
                    >
                      No orders found.
                    </td>
                  </tr>
                )}
              </tbody>
            </table>
          </div>
        </div>
      </div>
      {/* Purchase Order Charts + Upcoming Orders Section */}
      <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
        {/* Purchase Order Charts Section */}
        <div className="bg-white dark:bg-gray-800 rounded-xl shadow p-4 sm:p-6">
          {/* Header */}
          <div className="flex items-center justify-between mb-4">
            <h2 className="text-lg font-semibold text-gray-800 dark:text-gray-100 mb-4 flex items-center gap-2">
              <BarChart2 className="w-5 h-5 text-blue-600" />
              Purchase Orders by Status
              <div className="relative inline-block group">
                {/* Info Icon */}
                <span className="cursor-pointer text-blue-600">
                  <FaInfoCircle className="w-4 h-4 text-gray-500 cursor-pointer" />
                </span>

                {/* Tooltip */}
                <div className="absolute bottom-full left-1/2 transform -translate-x-1/2 mb-2 w-100 bg-black text-white text-sm text-center rounded px-2 py-1 opacity-0 invisible group-hover:visible group-hover:opacity-100 transition-opacity duration-300">
                  Displays a breakdown of purchase orders by their current
                  status (e.g., pending, approved, completed).
                  {/* Arrow */}
                  <div className="absolute top-full left-1/2 transform -translate-x-1/2 w-2 h-2 bg-black rotate-45"></div>
                </div>
              </div>
            </h2>
            <select
              value={timeFilter}
              onChange={(e) => setTimeFilter(e.target.value)}
              className="text-sm border rounded-md px-2 py-1 bg-white dark:bg-gray-700 dark:text-gray-200 focus:ring-2 focus:ring-blue-500"
            >
              <option>Last 30 Days</option>
              <option>Last 15 Days</option>
              <option>Last 7 Days</option>
            </select>
          </div>

          {/* Chart */}
          <div className="flex justify-center">
            <Chart
              options={chartOptions}
              series={statusCounts}
              type="pie"
              height={300}
            />
          </div>
        </div>

        {/* Upcoming Purchase Orders */}
        <div className="bg-white dark:bg-gray-900 rounded-2xl shadow-lg p-5 border border-gray-200 dark:border-gray-700">
          <h2 className="text-lg font-semibold text-gray-800 dark:text-gray-100 mb-4 flex items-center gap-2">
            <Package className="w-5 h-5 text-purple-600" />
            Upcoming Purchase Orders
            <div className="relative inline-block group">
              {/* Info Icon */}
              <span className="cursor-pointer text-blue-600">
                <FaInfoCircle className="w-4 h-4 text-gray-500 cursor-pointer" />
              </span>

              {/* Tooltip */}
              <div className="absolute bottom-full left-1/2 transform -translate-x-1/2 mb-2 w-100 bg-black text-white text-sm text-center rounded px-2 py-1 opacity-0 invisible group-hover:visible group-hover:opacity-100 transition-opacity duration-300">
                Displays purchase orders scheduled to be created or delivered in
                the near future.
                {/* Arrow */}
                <div className="absolute top-full left-1/2 transform -translate-x-1/2 w-2 h-2 bg-black rotate-45"></div>
              </div>
            </div>
          </h2>
          <div className="overflow-x-auto">
            <table className="min-w-full text-sm">
              <thead className="bg-gray-100 dark:bg-gray-700 text-gray-700 dark:text-gray-200">
                <tr>
                  <th className="px-3 py-2 text-left">Supplier</th>
                  <th className="px-3 py-2 text-left">PO #</th>
                  <th className="px-3 py-2 text-left">Status</th>
                  <th className="px-3 py-2 text-left">Delivery Date</th>
                </tr>
              </thead>
              <tbody className="divide-y divide-gray-200 dark:divide-gray-700">
                {upcomingOrders.length > 0 ? (
                  upcomingOrders.slice(0, 5).map((order, i) => (
                    <tr
                      key={i}
                      className="hover:bg-gray-50 dark:hover:bg-gray-800 transition"
                    >
                      <td className="px-3 py-2">
                        {order.supplier_name ?? "-"}
                      </td>
                      <td className="px-3 py-2">
                        {order.purchase_order_id ?? "-"}
                      </td>
                      <td className="px-3 py-2">
                        <span
                          className={`px-2 py-1 rounded-full text-xs font-medium 
                    ${
                      order.Status === "Confirmed"
                        ? "bg-green-100 text-green-700 dark:bg-green-900/40 dark:text-green-300"
                        : order.Status === "Sent"
                        ? "bg-blue-100 text-blue-700 dark:bg-blue-900/40 dark:text-blue-300"
                        : order.Status === "Draft"
                        ? "bg-gray-100 text-gray-700 dark:bg-gray-800/40 dark:text-gray-300"
                        : "bg-red-100 text-red-700 dark:bg-red-900/40 dark:text-red-300"
                    }`}
                        >
                          {order.Status ?? "-"}
                        </span>
                      </td>
                      <td className="px-3 py-2">
                        {order.expected_delivery_date
                          ? new Date(
                              order.expected_delivery_date
                            ).toLocaleDateString()
                          : "-"}
                      </td>
                    </tr>
                  ))
                ) : (
                  <tr>
                    <td
                      colSpan={4}
                      className="px-4 py-3 text-center text-gray-500 dark:text-gray-400"
                    >
                      No upcoming orders.
                    </td>
                  </tr>
                )}
              </tbody>
            </table>
          </div>
        </div>
      </div>
      {/* Department-wise Requisition Chart + Requisition Status Chart Section */}
      <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
        {/* Purchase Requisitions by Department */}
        <div className="bg-white dark:bg-gray-800 rounded-xl shadow p-4 sm:p-6">
          <div className="flex items-center justify-between mb-4">
            <h2 className="text-lg font-semibold text-gray-800 dark:text-gray-100 mb-4 flex items-center gap-2">
              <LayoutGrid className="w-5 h-5 text-green-600" />
              Purchase Requisitions by Department
              <div className="relative inline-block group">
                {/* Icon */}
                <span className="text-gray-500">
                  <FaInfoCircle className="w-4 h-4" />
                </span>

                {/* Tooltip */}
                <div className="absolute bottom-full left-1/2 transform -translate-x-1/2 mb-2 hidden w-40 bg-black text-white text-xs text-center rounded px-2 py-1 group-hover:block z-10">
                  This chart shows the number of purchase requisitions per
                  department.
                </div>
              </div>
            </h2>
          </div>
          <div className="flex justify-center">
            <ReactApexChart
              options={deptChart}
              series={deptSeries}
              type="bar"
              height={275}
            />
          </div>
        </div>

        {/* Purchase Requisition Charts Section */}
        <div className="bg-white dark:bg-gray-800 rounded-xl shadow p-4 sm:p-6">
          {/* Header */}
          <div className="flex items-center justify-between mb-4">
            <h2 className="text-lg font-semibold text-gray-800 dark:text-gray-100 mb-4 flex items-center gap-2">
              <FileCheck className="w-5 h-5 text-yellow-600" />
              Purchase Requisition Status
              <div className="relative inline-block group">
                {/* Info Icon */}
                <span className="text-gray-500 cursor-pointer">
                  <FaInfoCircle className="w-4 h-4" />
                </span>

                {/* Tooltip */}
                <div className="absolute bottom-full left-1/2 transform -translate-x-1/2 mb-2 hidden w-100 bg-black text-white text-xs text-center rounded px-2 py-1 group-hover:block z-10">
                  This chart shows the status of purchase requisitions
                  (Approved, Pending, Rejected).
                </div>
              </div>
            </h2>
          </div>

          {/* Chart */}
          <div className="flex justify-center">
            <ReactApexChart
              options={requisitionChart}
              series={requisitionSeries}
              type="donut"
              height={240}
            />
          </div>

          {/* Custom Legend */}
          <div className="mt-4 space-y-2">
            <div className="flex items-center justify-between text-sm">
              <span className="flex items-center gap-2">
                <span className="w-3 h-3 rounded-full bg-yellow-500"></span>
                Pending
              </span>
              <span className="font-medium text-gray-700 dark:text-gray-300">
                {counts.Pending}
              </span>
            </div>
            <div className="flex items-center justify-between text-sm">
              <span className="flex items-center gap-2">
                <span className="w-3 h-3 rounded-full bg-blue-500"></span>
                Approved
              </span>
              <span className="font-medium text-gray-700 dark:text-gray-300">
                {counts.Approved}
              </span>
            </div>
            <div className="flex items-center justify-between text-sm">
              <span className="flex items-center gap-2">
                <span className="w-3 h-3 rounded-full bg-red-500"></span>
                Rejected
              </span>
              <span className="font-medium text-gray-700 dark:text-gray-300">
                {counts.Rejected}
              </span>
            </div>
          </div>
        </div>
      </div>
    </div>
  );
}
