"use client";

import React, { useCallback, useEffect, useState } from "react";
import Button from "antd/es/button";
import "antd/es/button/style";
import Card from "antd/es/card";
import "antd/es/card/style";
import DatePicker from "antd/es/date-picker";
import "antd/es/date-picker/style";
import Form from "antd/es/form";
import "antd/es/form/style";
import Input from "antd/es/input";
import "antd/es/input/style";
import Modal from "antd/es/modal";
import "antd/es/modal/style";
import Select from "antd/es/select";
import "antd/es/select/style";
import Space from "antd/es/space";
import "antd/es/space/style";
import Tooltip from "antd/es/tooltip";
import "antd/es/tooltip/style";
import Tag from "antd/es/tag";
import "antd/es/tag/style";
import Breadcrumb from "antd/es/breadcrumb";
import "antd/es/breadcrumb/style";
import Switch from "antd/es/switch";
import "antd/es/switch/style";
import Pagination from "antd/es/pagination";
import "antd/es/pagination/style";
import Upload from "antd/es/upload";
import "antd/es/upload/style";
import type { RcFile } from "antd/es/upload/interface";
import {
  PlusOutlined,
  EditOutlined,
  DeleteOutlined,
  EyeOutlined,
  HomeOutlined,
  TableOutlined,
  AppstoreOutlined,
  SearchOutlined,
  CheckCircleOutlined,
  UserOutlined,
  FileOutlined,
} from "@ant-design/icons";
import {
  PieChart,
  Pie,
  Cell,
  Tooltip as RechartsTooltip,
  Legend,
  ResponsiveContainer,
} from "recharts";
import { RadialBarChart, RadialBar } from "recharts";
import type { ColumnsType } from "antd/es/table";
import DataTable from "../components/common/DataTable";
import { useIsMobile } from "@/data/useIsMobile";
import { toast } from "sonner";
import { motion } from "framer-motion";
import dayjs from "@/data/utils/dayjs";
import {
  ManageUser,
  createManageUser,
  getAllManageUsers,
  updateManageUser,
  deleteManageUser,
} from "../api/manageusers/manageusers";
import GlassCard from "../components/common/GlassCard";
import { Edit2, PlusSquare } from "lucide-react";
import type { PieLabelRenderProps } from "recharts";
import { UsersIcon } from "lucide-react";

const { TextArea } = Input;
const roleOptions = ["admin", "user", "Hr", "auditor", "superadmin"] as const;
const genderOptions = ["MALE", "FEMALE", "NOT_CHOOSE"] as const;
const empStatusOptions = [
  "Active",
  "Inactive",
  "BlackListed",
  "Terminated",
] as const;

export default function ManageUsersPage() {
  const [form] = Form.useForm();
  const [users, setUsers] = useState<ManageUser[]>([]);
  const [isModalOpen, setIsModalOpen] = useState(false);
  const [editingUser, setEditingUser] = useState<ManageUser | null>(null);
  const [viewingUser, setViewingUser] = useState<ManageUser | null>(null);
  const [searchTerm, setSearchTerm] = useState("");
  const [isGridView, setIsGridView] = useState(false);
  const [selectedRows, setSelectedRows] = useState<string[]>([]);
  const [advancedFilters, setAdvancedFilters] = useState({
    role: undefined,
    emp_status: undefined,
  });
  const [dateRange, setDateRange] = useState<[dayjs.Dayjs, dayjs.Dayjs] | null>(
    null
  );
  const [currentPage, setCurrentPage] = useState(1);
  const [pageSize, setPageSize] = useState(10);
  const [loading, setLoading] = useState(false);
  const [photoFile, setPhotoFile] = useState<File | null>(null);
  const isMobile = useIsMobile();

  useEffect(() => {
    const fetchUsers = async () => {
      setLoading(true);
      try {
        const userData = await getAllManageUsers();
        setUsers(userData.filter((u) => !u.is_delete));
      } catch (error) {
        toast.error("Failed to fetch users");
      } finally {
        setLoading(false);
      }
    };
    fetchUsers();
  }, []);

  const openAddModal = () => {
    form.resetFields();
    setEditingUser(null);
    setPhotoFile(null);
    setIsModalOpen(true);
  };

  const openEditModal = (user: ManageUser) => {
    form.setFieldsValue({
      ...user,
      DOB: user.DOB ? dayjs(user.DOB) : undefined,
      join_date: user.join_date ? dayjs(user.join_date) : undefined,
    });
    setEditingUser(user);
    setPhotoFile(null);
    setIsModalOpen(true);
  };

  const handleDelete = (userId: string) => {
    Modal.confirm({
      title: "Are you sure you want to delete this user?",
      onOk: async () => {
        try {
          await deleteManageUser(userId);
          toast.success("User deleted successfully");
          setUsers((prev) => prev.filter((u) => u.id !== userId));
          setSelectedRows((prev) => prev.filter((id) => id !== userId));
        } catch (error) {
          toast.error("Failed to delete user");
        }
      },
    });
  };

  const handleBulkDelete = () => {
    if (selectedRows.length === 0) return;
    Modal.confirm({
      title: `Are you sure you want to delete ${selectedRows.length} users?`,
      onOk: async () => {
        try {
          for (const userId of selectedRows) {
            await deleteManageUser(userId);
          }
          toast.success("Users deleted successfully");
          setSelectedRows([]);
          setUsers((prev) => prev.filter((u) => !selectedRows.includes(u.id!)));
        } catch (error) {
          toast.error("Failed to delete users");
        }
      },
    });
  };

  const handleFormSubmit = async () => {
    try {
      const values = await form.validateFields();
      const data: Partial<ManageUser> = {
        ...values,
        DOB: values.DOB ? dayjs(values.DOB).format("YYYY-MM-DD") : "",
        join_date: values.join_date
          ? dayjs(values.join_date).format("YYYY-MM-DD")
          : "",
        photo: photoFile,
      };

      if (editingUser) {
        await updateManageUser(editingUser.id!, data);
        toast.success("User updated successfully");
      } else {
        await createManageUser(data as ManageUser);
        toast.success("User added successfully");
      }
      setIsModalOpen(false);
      form.resetFields();
      setPhotoFile(null);
      const userData = await getAllManageUsers();
      setUsers(userData.filter((u) => !u.is_delete));
    } catch (error) {
      toast.error("Failed to save user");
    }
  };

  const renderTag = (value: string) => {
    const colorMap: Record<string, string> = {
      Active: "green",
      Inactive: "red",
      BlackListed: "volcano",
      Terminated: "gray",
    };
    return <Tag color={colorMap[value] || "default"}>{value}</Tag>;
  };

  const filteredUsers = users.filter((user) => {
    const matchesSearch =
      user.email.toLowerCase().includes(searchTerm.toLowerCase()) ||
      user.First_name.toLowerCase().includes(searchTerm.toLowerCase()) ||
      user.Last_name.toLowerCase().includes(searchTerm.toLowerCase()) ||
      user.phone_number.includes(searchTerm.toLowerCase());
    const matchesDate =
      !dateRange ||
      (user.created_at &&
        (!dateRange[0] ||
          dayjs(user.created_at).isSame(dateRange[0], "day") ||
          dayjs(user.created_at).isAfter(dateRange[0])) &&
        (!dateRange[1] ||
          dayjs(user.created_at).isSame(dateRange[1], "day") ||
          dayjs(user.created_at).isBefore(dateRange[1])));
    const matchesAdvanced =
      (!advancedFilters.role || user.role === advancedFilters.role) &&
      (!advancedFilters.emp_status ||
        user.emp_status === advancedFilters.emp_status);
    return matchesSearch && matchesDate && matchesAdvanced;
  });
  const paginatedUsers = filteredUsers.slice(
    (currentPage - 1) * pageSize,
    currentPage * pageSize
  );

  const columns: ColumnsType<ManageUser> = [
    {
      title: (
        <input
          type="checkbox"
          onChange={(e) => {
            const checked = e.target.checked;
            setSelectedRows(checked ? filteredUsers.map((u) => u.id!) : []);
          }}
          className="w-4 h-4"
        />
      ),
      key: "selection",
      render: (_, record) => (
        <input
          type="checkbox"
          checked={selectedRows.includes(record.id!)}
          onChange={(e) => {
            const rowId = record.id!;
            setSelectedRows((prev) =>
              e.target.checked
                ? [...prev, rowId]
                : prev.filter((id) => id !== rowId)
            );
          }}
          className="w-4 h-4"
        />
      ),
      width: 44,
    },
    {
      title: "Name",
      key: "name",
      sorter: (a, b) =>
        `${a.First_name} ${a.Last_name}`.localeCompare(
          `${b.First_name} ${b.Last_name}`
        ),
      render: (_, record) => (
        <span className="text-green-600 font-medium">{`${record.First_name} ${record.Last_name}`}</span>
      ),
    },
    {
      title: "Email",
      dataIndex: "email",
      sorter: (a, b) => a.email.localeCompare(b.email),
    },
    {
      title: "Phone Number",
      dataIndex: "phone_number",
      sorter: (a, b) => a.phone_number.localeCompare(b.phone_number),
    },
    {
      title: "Gender",
      dataIndex: "Gender",
      filters: genderOptions.map((g) => ({ text: g, value: g })),
      onFilter: (value, record) => record.Gender === value,
    },
    {
      title: "DOB",
      dataIndex: "DOB",
      sorter: (a, b) =>
        dayjs(a.DOB || "1900-01-01").unix() -
        dayjs(b.DOB || "1900-01-01").unix(),
      render: (text) => dayjs(text).format("YYYY-MM-DD"),
    },
    {
      title: "Role",
      dataIndex: "role",
      filters: roleOptions.map((r) => ({ text: r, value: r })),
      onFilter: (value, record) => record.role === value,
      sorter: (a, b) => a.role.localeCompare(b.role),
    },
    {
      title: "Department",
      dataIndex: "department",
      sorter: (a, b) => (a.department || "").localeCompare(b.department || ""),
    },
    {
      title: "Manager",
      dataIndex: "manager",
      sorter: (a, b) => (a.manager || "").localeCompare(b.manager || ""),
    },
    {
      title: "Join Date",
      dataIndex: "join_date",
      sorter: (a, b) =>
        dayjs(a.join_date || "1900-01-01").unix() -
        dayjs(b.join_date || "1900-01-01").unix(),
      render: (text) => (text ? dayjs(text).format("YYYY-MM-DD") : "-"),
    },
    {
      title: "Location",
      dataIndex: "location",
      sorter: (a, b) => (a.location || "").localeCompare(b.location || ""),
    },
    {
      title: "Employee Status",
      dataIndex: "emp_status",
      filters: empStatusOptions.map((s) => ({ text: s, value: s })),
      onFilter: (value, record) => record.emp_status === value,
      render: renderTag,
    },
    {
      title: "Country",
      dataIndex: "country",
      sorter: (a, b) => (a.country || "").localeCompare(b.country || ""),
    },
    {
      title: "Qualification",
      dataIndex: "qualification",
      sorter: (a, b) =>
        (a.qualification || "").localeCompare(b.qualification || ""),
    },
    {
      title: "Experience (Years)",
      dataIndex: "experience_years",
      sorter: (a, b) =>
        (a.experience_years || "").localeCompare(b.experience_years || ""),
    },
    {
      title: "Certifications",
      dataIndex: "certifications",
    },
    {
      title: "Actions",
      key: "actions",
      fixed: "right",
      render: (_, record) => (
        <Space size="small">
          <Tooltip title="View">
            <Button
              icon={<EyeOutlined />}
              onClick={() => setViewingUser(record)}
              size="small"
              className="text-blue-600"
            />
          </Tooltip>
          <Tooltip title="Edit">
            <Button
              icon={<EditOutlined />}
              onClick={() => openEditModal(record)}
              size="small"
              type="primary"
              className="bg-blue-600 hover:bg-blue-700"
            />
          </Tooltip>
          <Tooltip title="Delete">
            <Button
              icon={<DeleteOutlined />}
              onClick={() => handleDelete(record.id!)}
              danger
              size="small"
              className="text-red-600"
            />
          </Tooltip>
        </Space>
      ),
    },
  ];

  const UserCard = ({ user }: { user: ManageUser }) => (
    <GlassCard gradient blurStrength="lg" hoverLift color="primary">
      <div className="border rounded-xl p-4 shadow-sm hover:shadow-md bg-white dark:bg-gray-900 transition-all w-full">
        {/* Header */}
        <div className="flex flex-col sm:flex-row justify-between items-start sm:items-center mb-2 gap-2">
          <h3 className="text-base sm:text-lg font-semibold text-gray-800 dark:text-white truncate">
            {`${user.First_name} ${user.Last_name}`}
          </h3>
          <Tag color={user.emp_status === "Active" ? "green" : "red"}>
            {user.emp_status || "N/A"}
          </Tag>
        </div>

        {/* User Info */}
        <div className="grid grid-cols-1 gap-y-2 text-sm">
          <p className="text-gray-600 dark:text-gray-300 truncate">
            <strong>Email:</strong> {user.email || "-"}
          </p>
          <p className="text-gray-600 dark:text-gray-300 truncate">
            <strong>Phone:</strong> {user.phone_number || "-"}
          </p>
          <p className="text-gray-600 dark:text-gray-300 truncate">
            <strong>Gender:</strong> {user.Gender || "-"}
          </p>
          <p className="text-gray-600 dark:text-gray-300 truncate">
            <strong>DOB:</strong>{" "}
            {user.DOB ? dayjs(user.DOB).format("YYYY-MM-DD") : "-"}
          </p>
          <p className="text-gray-600 dark:text-gray-300 truncate">
            <strong>Role:</strong> {user.role || "-"}
          </p>
          <p className="text-gray-600 dark:text-gray-300 truncate">
            <strong>Employee Role:</strong> {user.employee_role || "-"}
          </p>
          <p className="text-gray-600 dark:text-gray-300 truncate">
            <strong>Department:</strong> {user.department || "-"}
          </p>
          <p className="text-gray-600 dark:text-gray-300 truncate">
            <strong>Manager:</strong> {user.manager || "-"}
          </p>
          <p className="text-gray-600 dark:text-gray-300 truncate">
            <strong>Join Date:</strong>{" "}
            {user.join_date ? dayjs(user.join_date).format("YYYY-MM-DD") : "-"}
          </p>
          <p className="text-gray-600 dark:text-gray-300 truncate">
            <strong>Location:</strong> {user.location || "-"}
          </p>
          <p className="text-gray-600 dark:text-gray-300 truncate">
            <strong>Address:</strong> {user.address || "-"}
          </p>
          <p className="text-gray-600 dark:text-gray-300 truncate">
            <strong>Country:</strong> {user.country || "-"}
          </p>
          <p className="text-gray-600 dark:text-gray-300 truncate">
            <strong>Qualification:</strong> {user.qualification || "-"}
          </p>
          <p className="text-gray-600 dark:text-gray-300 truncate">
            <strong>Experience:</strong> {user.experience_years || "-"} years
          </p>
          <p className="text-gray-600 dark:text-gray-300 truncate">
            <strong>Certifications:</strong> {user.certifications || "-"}
          </p>
          {user.photo && (
            <p className="text-gray-600 dark:text-gray-300 truncate">
              <strong>Photo:</strong>{" "}
              {typeof user.photo === "string" ? user.photo : user.photo.name}
            </p>
          )}
        </div>

        {/* Actions */}
        <div className="mt-2 flex gap-2">
          <Tooltip title="View">
            <Button
              icon={<EyeOutlined />}
              onClick={() => setViewingUser(user)}
              size="small"
              className="text-blue-600"
            />
          </Tooltip>
          <Tooltip title="Edit">
            <Button
              icon={<EditOutlined />}
              onClick={() => openEditModal(user)}
              size="small"
              type="primary"
              className="bg-blue-600 hover:bg-blue-700"
            />
          </Tooltip>
          <Tooltip title="Delete">
            <Button
              icon={<DeleteOutlined />}
              onClick={() => handleDelete(user.id!)}
              size="small"
              danger
              className="text-red-600"
            />
          </Tooltip>
        </div>

        {/* Audit Info */}
        <p className="text-gray-500 dark:text-gray-400 text-xs mt-2">
          <strong>Created:</strong>{" "}
          {user.created_at
            ? dayjs(user.created_at).format("YYYY-MM-DD")
            : "N/A"}
        </p>
        <p className="text-gray-500 dark:text-gray-400 text-xs">
          <strong>Updated:</strong>{" "}
          {user.updated_at
            ? dayjs(user.updated_at).format("YYYY-MM-DD")
            : "N/A"}
        </p>
        <p className="text-gray-500 dark:text-gray-400 text-xs">
          <strong>Last Activity:</strong>{" "}
          {user.last_activity
            ? dayjs(user.last_activity).format("YYYY-MM-DD")
            : "N/A"}
        </p>
      </div>
    </GlassCard>
  );

  UserCard.displayName = "UserCard";

  // Stats
  const totalUsers = users.length;
  const activeUsers = users.filter((u) => u.emp_status === "Active").length;
  const roleData = roleOptions.map((role) => ({
    name: role,
    value: users.filter((u) => u.role === role).length,
  }));
  const COLORS = ["#0088FE", "#00C49F", "#FFBB28", "#FF8042", "#845EC2"];

  const recentActivity = users
    .filter((u) => u.last_activity)
    .sort(
      (a, b) => dayjs(b.last_activity).unix() - dayjs(a.last_activity).unix()
    )
    .slice(0, 5);

  return (
    <div className="px-2 sm:px-2 md:px-2 lg:px-2 md:py-2 lg:py-2 py-2 sm:py-2">
      <h2 className="flex items-center gap-2 text-xl sm:text-2xl md:text-3xl font-bold text-gray-800 dark:text-white mb-2">
        <UsersIcon className="w-7 h-7 text-green-600 dark:text-green-400 flex-shrink-0" />
        <span>Manage Users</span>
        <span className="text-gray-500 text-base sm:text-lg font-medium">
          ({users.length})
        </span>
      </h2>
      <p className="text-black-600 dark:text-white-300 text-sm sm:text-base mb-4">
        Manage user accounts and their details.
      </p>
      <Breadcrumb
        className="mb-4 sm:mb-6 mt-2 text-sm text-gray-500 dark:text-gray-400"
        separator=">"
        items={[
          {
            title: (
              <Tooltip title="Dashboard">
                <HomeOutlined />
              </Tooltip>
            ),
            href: "/dashboard",
          },
          {
            title: "Manage Users",
          },
        ]}
      />

      <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-4 mb-6 mt-3">
        <Card className="shadow-sm">
          <div className="flex items-center justify-between mb-2">
            <div>
              <p className="text-sm text-gray-600 dark:text-gray-400">
                Total Users
              </p>
              <h3 className="text-2xl font-bold text-gray-900 dark:text-white">
                {totalUsers}
              </h3>
            </div>
            <div className="text-3xl text-blue-600">
              <UserOutlined />
            </div>
          </div>

          {/* Breakdown */}
          <div className="mt-2 text-xs text-gray-500 dark:text-gray-400">
            <p>
              <span className="font-medium text-green-600">{activeUsers}</span>{" "}
              Active ·{" "}
              <span className="font-medium text-red-600">
                {totalUsers - activeUsers}
              </span>{" "}
              Inactive
            </p>
          </div>

          {/* Progress bar */}
          <div className="mt-3 w-full bg-gray-200 dark:bg-gray-700 rounded-full h-2">
            <div
              className="bg-green-500 h-2 rounded-full"
              style={{
                width: `${((activeUsers / (totalUsers || 1)) * 100).toFixed(
                  0
                )}%`,
              }}
            ></div>
          </div>

          {/* Trend vs last 7 days */}
          <div className="mt-3 flex items-center text-xs">
            <span className="text-green-600 font-medium flex items-center gap-1">
              ▲{" "}
              {
                users.filter((u) =>
                  dayjs(u.created_at).isAfter(dayjs().subtract(7, "day"))
                ).length
              }
            </span>
            <span className="ml-1 text-gray-500 dark:text-gray-400">
              new this week
            </span>
          </div>
          <ResponsiveContainer width="100%" height={120}>
            <PieChart>
              <Pie
                data={[
                  { name: "Active", value: activeUsers },
                  { name: "Inactive", value: totalUsers - activeUsers },
                ]}
                cx="50%"
                cy="50%"
                innerRadius={30}
                outerRadius={60}
                dataKey="value"
                labelLine={false}
                label={(props: PieLabelRenderProps) => {
                  const {
                    cx,
                    cy,
                    midAngle,
                    innerRadius,
                    outerRadius,
                    percent,
                  } = props;
                  if (
                    cx === undefined ||
                    cy === undefined ||
                    innerRadius == null ||
                    outerRadius == null
                  )
                    return null;

                  // // ✅ Normalize radius to numbers
                  // const resolveRadius = (
                  //   r: string | number | ((d: unknown) => number)
                  // ): number => {
                  //   if (typeof r === "number") return r;
                  //   if (typeof r === "string") return parseFloat(r); // handles "70%"
                  //   if (typeof r === "function") return r({}); // fallback
                  //   return 0;
                  // };

                  // const inner = resolveRadius(innerRadius);
                  // const outer = resolveRadius(outerRadius);

                  // const safeCx =
                  //   typeof cx === "number" ? cx : parseFloat(String(cx));
                  // const safeCy =
                  //   typeof cy === "number" ? cy : parseFloat(String(cy));

                  // const safeMidAngle = midAngle ?? 0;
                  // const radius = inner + (outer - inner) / 2;

                  // const x =
                  //   safeCx + radius * Math.cos(-safeMidAngle * (Math.PI / 180));
                  // const y =
                  //   safeCy + radius * Math.sin(-safeMidAngle * (Math.PI / 180));

                  const RADIUS_INNER = Number(innerRadius);
                  const RADIUS_OUTER = Number(outerRadius);
                  const safeCx = Number(cx);
                  const safeCy = Number(cy);
                  const safeMidAngle = Number(midAngle);

                  const radius =
                    RADIUS_INNER + (RADIUS_OUTER - RADIUS_INNER) / 2;
                  const x =
                    safeCx + radius * Math.cos(-safeMidAngle * (Math.PI / 180));
                  const y =
                    safeCy + radius * Math.sin(-safeMidAngle * (Math.PI / 180));

                  return (
                    <text
                      x={x}
                      y={y}
                      fill="#000"
                      textAnchor="middle"
                      dominantBaseline="central"
                      fontSize={10}
                      fontWeight="bold"
                    >
                      {`${Math.round((percent as number) * 100)}%`}
                    </text>
                  );
                }}
              >
                <Cell fill="#22c55e" />
                <Cell fill="#ef4444" />
              </Pie>
            </PieChart>
          </ResponsiveContainer>

          <div className="mt-2 text-xs">
            <span className="text-blue-600 font-medium">+12%</span> vs last
            month
          </div>
        </Card>

        <Card className="shadow-sm">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm text-Black-800 dark:text-Black-300">
                Active Users
              </p>
              <h3 className="text-2xl font-bold text-Black-800 dark:text-Black">
                {activeUsers}
              </h3>
            </div>
            <div className="text-3xl text-blue-600">
              <CheckCircleOutlined />
            </div>
          </div>
          <div>
            <ResponsiveContainer width="100%" height={220}>
              <RadialBarChart
                cx="50%"
                cy="50%"
                innerRadius="70%"
                outerRadius="100%"
                barSize={20}
                data={[
                  {
                    name: "Active",
                    value: (activeUsers / totalUsers) * 100,
                    fill: "#7eb7ecff",
                  },
                ]}
              >
                <RadialBar background dataKey="value" cornerRadius={10} />
                <text
                  x="50%"
                  y="50%"
                  textAnchor="middle"
                  dominantBaseline="middle"
                  className="text-xl font-bold fill-gray-800 dark:fill-gray-100"
                >
                  {((activeUsers / totalUsers) * 100).toFixed(0)}%
                </text>
              </RadialBarChart>
            </ResponsiveContainer>
            <p className="text-sm text-gray-500">
              {activeUsers} out of {totalUsers} users
            </p>
            <p className="text-xs text-black-500 mt-1">
              ▲ {((activeUsers / totalUsers) * 100).toFixed(0)}% Active
            </p>
          </div>
        </Card>
        <Card className="shadow-sm !bg-white">
          <p className="text-sm text-gray-800 mb-3 font-medium">
            Roles Distribution
          </p>
          <div style={{ width: "100%", height: 260 }}>
            <ResponsiveContainer>
              <PieChart>
                <Pie
                  data={roleData}
                  cx="50%"
                  cy="45%"
                  outerRadius={90}
                  innerRadius={50} // donut look
                  dataKey="value"
                  isAnimationActive={true}
                  paddingAngle={4}
                  labelLine={false}
                  label={({
                    cx,
                    cy,
                    midAngle,
                    innerRadius,
                    outerRadius,
                    percent,
                  }: PieLabelRenderProps) => {
                    if (
                      cx == null ||
                      cy == null ||
                      midAngle == null ||
                      innerRadius == null ||
                      outerRadius == null ||
                      percent == null
                    )
                      return null;

                    const inner = Number(innerRadius);
                    const outer = Number(outerRadius);
                    const radius = inner + (outer - inner) / 2;
                    const angle = Number(midAngle);

                    const x =
                      Number(cx) + radius * Math.cos(-angle * (Math.PI / 180));
                    const y =
                      Number(cy) + radius * Math.sin(-angle * (Math.PI / 180));
                    return (
                      <text
                        x={x}
                        y={y}
                        fill="#000" // ⬅️ black labels
                        textAnchor="middle"
                        dominantBaseline="central"
                        fontSize={12}
                        fontWeight="bold"
                      >
                        {`${Math.round((percent as number) * 100)}%`}
                      </text>
                    );
                  }}
                >
                  {roleData.map((_, index) => (
                    <Cell
                      key={`cell-${index}`}
                      fill={COLORS[index % COLORS.length]}
                      strokeWidth={2}
                      stroke="#fff"
                    />
                  ))}
                </Pie>

                <RechartsTooltip
                  contentStyle={{
                    backgroundColor: "#1f2937", // dark gray tooltip
                    borderRadius: "0.5rem",
                    border: "none",
                    color: "#fff",
                    padding: "8px 12px",
                    maxWidth: "200px",
                    whiteSpace: "normal",
                  }}
                  itemStyle={{ color: "#fff" }}
                  formatter={(value: number, name: string) => [
                    `${value} users`,
                    name,
                  ]}
                />

                <Legend
                  layout="horizontal"
                  align="center"
                  verticalAlign="bottom"
                  iconType="circle"
                  wrapperStyle={{
                    fontSize: "0.85rem",
                    marginTop: "8px",
                    color: "#374151", // gray-700 legend
                  }}
                />
              </PieChart>
            </ResponsiveContainer>
          </div>
        </Card>

        <Card className="shadow-sm">
          <p className="text-sm text-black-800 dark:text-black-300 font-medium mb-2">
            Recent Activity
          </p>

          {/* Mini Analytics */}
          <div className="grid grid-cols-3 gap-2 mb-4 text-center">
            <div className="p-2 bg-gray-100 dark:bg-gray-800 rounded-lg">
              <p className="text-xs text-gray-500 dark:text-gray-400">
                Last 7 Days
              </p>
              <p className="text-lg font-semibold text-black-800 dark:text-white">
                {
                  users.filter((u) =>
                    dayjs(u.last_activity).isAfter(dayjs().subtract(7, "day"))
                  ).length
                }
              </p>
            </div>
            <div className="p-2 bg-gray-100 dark:bg-gray-800 rounded-lg">
              <p className="text-xs text-gray-500 dark:text-gray-400">
                Most Active
              </p>
              <p className="text-sm font-semibold text-Black-600 dark:text-white truncate">
                {recentActivity[0] ? `${recentActivity[0].First_name}` : "-"}
              </p>
            </div>
            <div className="p-2 bg-gray-100 dark:bg-gray-800 rounded-lg">
              <p className="text-xs text-gray-500 dark:text-gray-400">
                Avg/User
              </p>
              <p className="text-lg font-semibold text-black-800 dark:text-white">
                {(
                  users.filter((u) => u.last_activity).length /
                  (users.length || 1)
                ).toFixed(1)}
              </p>
            </div>
          </div>

          {/* Feed */}
          <div className="space-y-3 max-h-48 overflow-y-auto pr-1 scrollbar-thin">
            {recentActivity.length > 0 ? (
              recentActivity.map((user) => (
                <div
                  key={user.id}
                  className="flex items-center gap-3 p-2 rounded-md hover:bg-black-50 dark:hover:bg-black-800 transition"
                >
                  {/* Avatar */}
                  <div className="w-8 h-8 flex items-center justify-center rounded-full bg-blue-600 text-white font-semibold">
                    {user.First_name?.[0]}
                  </div>

                  {/* Details */}
                  <div className="flex-1">
                    <p className="text-sm font-medium text-black-800 dark:text-black truncate">
                      {user.First_name} {user.Last_name}
                    </p>
                    <p className="text-xs text-black-500 dark:text-black-400">
                      Last active {dayjs(user.last_activity).fromNow()}
                    </p>
                  </div>
                </div>
              ))
            ) : (
              <div className="text-sm text-gray-500 dark:text-gray-400">
                No recent activity
              </div>
            )}
          </div>
        </Card>
      </div>
      <div className="mt-4">
        <Card title="User List" className="p-0 mb-6 shadow-sm">
          <div className="p-4 flex flex-col gap-4">
            <div className="flex flex-col sm:flex-row justify-between items-stretch sm:items-center gap-4">
              <Input
                prefix={<SearchOutlined />}
                allowClear
                placeholder="Search by Name, Email, or Phone"
                value={searchTerm}
                onChange={(e) => setSearchTerm(e.target.value)}
                className="w-full sm:w-64 border-gray-300 dark:border-gray-600 rounded-md shadow-sm"
              />
              <Button
                type="primary"
                icon={<PlusOutlined />}
                onClick={openAddModal}
                className="bg-blue-600 hover:bg-blue-700 w-full sm:w-auto"
              >
                Add User
              </Button>
            </div>
            <div className="flex flex-col gap-4 lg:flex-row lg:justify-between lg:items-center">
              <div className="grid grid-cols-1 sm:grid-cols-2 gap-4 w-full lg:w-auto">
                <Select
                  placeholder="Role"
                  className="w-full max-w-[160px]"
                  onChange={(value) =>
                    setAdvancedFilters((prev) => ({ ...prev, role: value }))
                  }
                  allowClear
                  options={roleOptions.map((v) => ({ label: v, value: v }))}
                />
                <Select
                  placeholder="Status"
                  className="w-full max-w-[160px]"
                  onChange={(value) =>
                    setAdvancedFilters((prev) => ({
                      ...prev,
                      emp_status: value,
                    }))
                  }
                  allowClear
                  options={empStatusOptions.map((v) => ({
                    label: v,
                    value: v,
                  }))}
                />
              </div>
              <div className="flex flex-col sm:flex-row flex-wrap gap-4 justify-start lg:justify-end w-full lg:w-auto">
                <DatePicker.RangePicker
                  value={dateRange}
                  onChange={(dates) =>
                    setDateRange(dates as [dayjs.Dayjs, dayjs.Dayjs] | null)
                  }
                  className="w-full sm:w-auto border-gray-300 dark:border-gray-600"
                />
                <Button
                  onClick={handleBulkDelete}
                  disabled={selectedRows.length === 0}
                  danger
                  className="w-full sm:w-auto"
                >
                  Bulk Delete
                </Button>
                <div className="hidden sm:block">
                  <Switch
                    checkedChildren={<AppstoreOutlined />}
                    unCheckedChildren={<TableOutlined />}
                    checked={isGridView}
                    onChange={setIsGridView}
                    className="w-full sm:w-auto"
                  />
                </div>
              </div>
            </div>

            {(advancedFilters.role || advancedFilters.emp_status) && (
              <div className="flex flex-wrap gap-2 mt-2">
                {advancedFilters.role && (
                  <Tag
                    closable
                    onClose={() =>
                      setAdvancedFilters((prev) => ({
                        ...prev,
                        role: undefined,
                      }))
                    }
                  >
                    Role: {advancedFilters.role}
                  </Tag>
                )}
                {advancedFilters.emp_status && (
                  <Tag
                    closable
                    onClose={() =>
                      setAdvancedFilters((prev) => ({
                        ...prev,
                        emp_status: undefined,
                      }))
                    }
                  >
                    Employee Status: {advancedFilters.emp_status}
                  </Tag>
                )}
              </div>
            )}
          </div>

          {isGridView || isMobile ? (
            <div className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 lg:grid-cols-3 gap-4 p-4">
              {paginatedUsers.map((user) => (
                <UserCard key={user.id} user={user} />
              ))}
            </div>
          ) : (
            <div className="overflow-auto">
              <DataTable<ManageUser>
                columns={columns}
                dataSource={paginatedUsers}
                rowKey={(record) => record.id!}
                scrollX="max-content"
                className="mt-4"
                pagination={false}
                loading={loading}
              />
            </div>
          )}
          <div className="flex flex-col sm:flex-row justify-between items-center px-4 py-2 text-sm text-gray-600 dark:text-gray-300">
            <div className="mb-2 sm:mb-0">
              Showing <strong>{(currentPage - 1) * pageSize + 1}</strong> to{" "}
              <strong>
                {Math.min(currentPage * pageSize, filteredUsers.length)}
              </strong>{" "}
              of <strong>{filteredUsers.length}</strong> users
            </div>
            <Pagination
              current={currentPage}
              pageSize={pageSize}
              total={filteredUsers.length}
              onChange={(page, size) => {
                setCurrentPage(page);
                setPageSize(size);
              }}
              showSizeChanger
              pageSizeOptions={["10", "30", "50", "100"]}
              size="small"
            />
          </div>
        </Card>
      </div>

      <motion.div
        initial={{ opacity: 0, scale: 0.95 }}
        animate={{ opacity: 1, scale: 1 }}
        exit={{ opacity: 0, scale: 0.95 }}
      >
        <Modal
          title={
            <div className="flex items-center gap-2 border-b pb-2 border-gray-300 dark:border-gray-600">
              {editingUser ? (
                <>
                  <Edit2 className="w-5 h-5 text-blue-600" />
                  <span className="font-semibold text-gray-800 dark:text-gray-200">
                    Edit User
                  </span>
                </>
              ) : (
                <>
                  <PlusSquare className="w-5 h-5 text-green-600" />
                  <span className="font-semibold text-gray-800 dark:text-gray-200">
                    Add User
                  </span>
                </>
              )}
            </div>
          }
          open={isModalOpen}
          onOk={handleFormSubmit}
          onCancel={() => {
            setIsModalOpen(false);
            form.resetFields();
            setPhotoFile(null);
          }}
          destroyOnHidden
          width="95%" // responsive width for all screens
          className="!top-5 font-semibold"
          styles={{
            body: { maxHeight: "80vh", overflowY: "auto" }, // ✅ fixed line
          }}
        >
          <Form form={form} layout="vertical" className="w-full px-4 space-y-8">
            {/* Personal Information */}
            <div>
              <h3 className="text-lg font-semibold text-gray-800 dark:text-gray-200 mb-4">
                Personal Information
              </h3>
              <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-4">
                <Form.Item
                  name="First_name"
                  label="First Name"
                  rules={[
                    { required: true, message: "First Name is required" },
                  ]}
                >
                  <Input />
                </Form.Item>
                <Form.Item
                  name="Last_name"
                  label="Last Name"
                  rules={[{ required: true, message: "Last Name is required" }]}
                >
                  <Input />
                </Form.Item>
                <Form.Item
                  name="email"
                  label="Email"
                  rules={[
                    { required: true, message: "Email is required" },
                    { type: "email", message: "Please enter a valid email" },
                  ]}
                >
                  <Input />
                </Form.Item>
                <Form.Item
                  name="phone_number"
                  label="Phone Number"
                  rules={[
                    { required: true, message: "Phone Number is required" },
                  ]}
                >
                  <Input />
                </Form.Item>
                <Form.Item
                  name="password"
                  label="Password"
                  rules={[
                    { required: !editingUser, message: "Password is required" },
                  ]}
                >
                  <Input.Password />
                </Form.Item>
                <Form.Item
                  name="Gender"
                  label="Gender"
                  rules={[{ required: true, message: "Gender is required" }]}
                >
                  <Select
                    options={genderOptions.map((v) => ({ label: v, value: v }))}
                  />
                </Form.Item>
                <Form.Item
                  name="DOB"
                  label="Date of Birth"
                  rules={[
                    { required: true, message: "Date of Birth is required" },
                  ]}
                >
                  <DatePicker className="w-full" />
                </Form.Item>
              </div>
              <hr className="my-6 border-gray-300 dark:border-gray-600" />
            </div>

            {/* Job & Role Information */}
            <div>
              <h3 className="text-lg font-semibold text-gray-800 dark:text-gray-200 mb-4">
                Job Information
              </h3>
              <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-4">
                <Form.Item
                  name="role"
                  label="Role"
                  rules={[{ required: true, message: "Role is required" }]}
                >
                  <Select
                    options={roleOptions.map((v) => ({ label: v, value: v }))}
                  />
                </Form.Item>
                <Form.Item
                  name="emp_status"
                  label="Employee Status"
                  rules={[
                    { required: true, message: "Employee Status is required" },
                  ]}
                >
                  <Select
                    options={empStatusOptions.map((v) => ({
                      label: v,
                      value: v,
                    }))}
                  />
                </Form.Item>
                <Form.Item name="department" label="Department">
                  <Input />
                </Form.Item>
                <Form.Item name="manager" label="Manager">
                  <Input />
                </Form.Item>
                <Form.Item name="join_date" label="Join Date">
                  <DatePicker className="w-full" />
                </Form.Item>
                <Form.Item name="location" label="Location">
                  <Input />
                </Form.Item>
              </div>
            </div>

            {/* Address & Personal Details */}
            <div>
              <h3 className="text-lg font-semibold text-gray-800 dark:text-gray-200 mb-4">
                Address & Details
              </h3>
              <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-4">
                <div className="col-span-full lg:col-span-2">
                  <Form.Item name="address" label="Address">
                    <TextArea rows={2} />
                  </Form.Item>
                </div>
                <Form.Item name="country" label="Country">
                  <Input />
                </Form.Item>
                <Form.Item name="qualification" label="Qualification">
                  <Input />
                </Form.Item>
                <Form.Item name="experience_years" label="Experience (Years)">
                  <Input />
                </Form.Item>
                <div className="col-span-full">
                  <Form.Item name="certifications" label="Certifications">
                    <TextArea rows={2} />
                  </Form.Item>
                </div>
              </div>
            </div>

            {/* Photo Upload */}
            <div>
              <h3 className="text-lg font-semibold text-gray-800 dark:text-gray-200 mb-4">
                Profile Photo
              </h3>
              <div className="grid grid-cols-1 lg:grid-cols-2 gap-4">
                <Form.Item label="Photo">
                  <Upload
                    fileList={
                      photoFile
                        ? [
                            {
                              uid: photoFile.name,
                              name: photoFile.name,
                              status: "done",
                              url: URL.createObjectURL(photoFile),
                              originFileObj: photoFile as RcFile,
                            },
                          ]
                        : []
                    }
                    onRemove={() => setPhotoFile(null)}
                    beforeUpload={(file) => {
                      setPhotoFile(file);
                      return false;
                    }}
                    maxCount={1}
                    accept="image/*"
                  >
                    <Button icon={<FileOutlined />}>Upload Photo</Button>
                  </Upload>
                  {photoFile && (
                    <p className="mt-2 text-sm text-gray-600 dark:text-gray-300">
                      Uploaded: {photoFile.name}
                    </p>
                  )}
                  {!photoFile &&
                    editingUser?.photo &&
                    typeof editingUser.photo === "string" && (
                      <p className="mt-2 text-sm text-gray-600 dark:text-gray-300">
                        Existing: {editingUser.photo}
                      </p>
                    )}
                </Form.Item>
              </div>
            </div>
          </Form>
        </Modal>
      </motion.div>

      <Modal
        title="User Details"
        open={!!viewingUser}
        onCancel={() => setViewingUser(null)}
        footer={null}
        className="w-full max-w-[90vw] sm:max-w-[500px] top-4"
      >
        {viewingUser && (
          <div className="space-y-2 text-sm sm:text-base">
            <div className="flex items-center gap-3">
              <div className="w-8 h-8 flex items-center justify-center rounded-full bg-blue-600 text-white font-semibold">
                {viewingUser.First_name?.[0]}
              </div>
              <p>
                <strong>Name:</strong>{" "}
                {`${viewingUser.First_name} ${viewingUser.Last_name}`}
              </p>
            </div>
            <p>
              <strong>Email:</strong> {viewingUser.email}
            </p>
            <p>
              <strong>Phone Number:</strong> {viewingUser.phone_number}
            </p>
            <p>
              <strong>Gender:</strong> {viewingUser.Gender}
            </p>
            <p>
              <strong>Date of Birth:</strong>{" "}
              {dayjs(viewingUser.DOB).format("YYYY-MM-DD")}
            </p>
            <p>
              <strong>Role:</strong> {viewingUser.role}
            </p>
            <p>
              <strong>Employee Status:</strong> {viewingUser.emp_status || "-"}
            </p>
            <p>
              <strong>Department:</strong> {viewingUser.department || "-"}
            </p>
            <p>
              <strong>Manager:</strong> {viewingUser.manager || "-"}
            </p>
            <p>
              <strong>Join Date:</strong>{" "}
              {viewingUser.join_date
                ? dayjs(viewingUser.join_date).format("YYYY-MM-DD")
                : "-"}
            </p>
            <p>
              <strong>Location:</strong> {viewingUser.location || "-"}
            </p>
            <p>
              <strong>Address:</strong> {viewingUser.address || "-"}
            </p>
            <p>
              <strong>Country:</strong> {viewingUser.country || "-"}
            </p>
            <p>
              <strong>Qualification:</strong> {viewingUser.qualification || "-"}
            </p>
            <p>
              <strong>Experience (Years):</strong>{" "}
              {viewingUser.experience_years || "-"}
            </p>
            <p>
              <strong>Certifications:</strong>{" "}
              {viewingUser.certifications || "-"}
            </p>
            <p>
              <strong>Created:</strong>{" "}
              {viewingUser.created_at
                ? dayjs(viewingUser.created_at).format("YYYY-MM-DD")
                : "-"}
            </p>
            <p>
              <strong>Updated:</strong>{" "}
              {viewingUser.updated_at
                ? dayjs(viewingUser.updated_at).format("YYYY-MM-DD")
                : "-"}
            </p>
            <p>
              <strong>Last Activity:</strong>{" "}
              {viewingUser.last_activity
                ? dayjs(viewingUser.last_activity).format("YYYY-MM-DD HH:mm:ss")
                : "-"}
            </p>
            {viewingUser.photo && typeof viewingUser.photo === "string" && (
              <p>
                <strong>Photo:</strong> {viewingUser.photo}
              </p>
            )}
          </div>
        )}
      </Modal>
    </div>
  );
}
