'use client';

import { useState } from 'react';
import { ToggleRight, ToggleLeft } from 'lucide-react';
import toast from 'react-hot-toast';

export default function SettingsPage() {
  const [settings, setSettings] = useState({
    notifications: true,
    twoFactorAuth: false,
    autoUpdate: true,
    language: 'English',
    timezone: 'IST (India Standard Time)',
    privacy: true,
    emailReports: false,
    dateFormat: 'DD/MM/YYYY',
  });

  const [saving, setSaving] = useState(false);

  const handleToggle = (key: keyof typeof settings) => {
    setSettings(prev => ({ ...prev, [key]: !prev[key] }));
  };

  const handleChange = (key: keyof typeof settings, value: string) => {
    setSettings(prev => ({ ...prev, [key]: value }));
  };

  const handleSave = async () => {
    try {
      setSaving(true);
      await new Promise(resolve => setTimeout(resolve, 1000)); // Simulate API
      toast.success('Settings updated successfully!');
    } catch (e) {
      toast.error('Failed to update settings.');
    } finally {
      setSaving(false);
    }
  };

  return (
    <main className="min-h-screen bg-gray-100 dark:bg-gray-900 transition-all duration-300">
      <div className="w-full px-6 py-10 space-y-10">
        {/* Header */}
        <header className="border-b pb-6 border-gray-200 dark:border-gray-700">
          <h1 className="text-4xl font-bold text-gray-800 dark:text-white">Settings</h1>
          <p className="text-sm text-gray-600 dark:text-gray-400 mt-1">Manage preferences for your QMS environment</p>
        </header>

        {/* System Settings */}
        <section className="bg-white dark:bg-gray-800 rounded-xl shadow-sm p-6 space-y-6">
          <h2 className="text-xl font-semibold text-gray-800 dark:text-white mb-2">System Preferences</h2>

          <SettingToggle
            title="Auto Update"
            description="Automatically receive QMS feature updates"
            enabled={settings.autoUpdate}
            onToggle={() => handleToggle('autoUpdate')}
          />

          <SettingToggle
            title="Email Notifications"
            description="Receive email alerts for system events"
            enabled={settings.notifications}
            onToggle={() => handleToggle('notifications')}
          />

          <SettingToggle
            title="Email Monthly Reports"
            description="Receive detailed monthly reports via email"
            enabled={settings.emailReports}
            onToggle={() => handleToggle('emailReports')}
          />

          <SettingSelect
            title="Preferred Language"
            options={['English', 'Hindi', 'Tamil', 'Telugu']}
            value={settings.language}
            onChange={value => handleChange('language', value)}
          />

          <SettingSelect
            title="Date Format"
            options={['DD/MM/YYYY', 'MM/DD/YYYY', 'YYYY-MM-DD']}
            value={settings.dateFormat}
            onChange={value => handleChange('dateFormat', value)}
          />

          <SettingSelect
            title="Time Zone"
            options={[
              'IST (India Standard Time)',
              'PST (Pacific Standard Time)',
              'EST (Eastern Standard Time)',
              'UTC',
            ]}
            value={settings.timezone}
            onChange={value => handleChange('timezone', value)}
          />
        </section>

        {/* Security Section */}
        <section className="bg-white dark:bg-gray-800 rounded-xl shadow-sm p-6 space-y-6">
          <h2 className="text-xl font-semibold text-gray-800 dark:text-white mb-2">Security</h2>

          <SettingToggle
            title="Two-Factor Authentication"
            description="Add an extra layer of security to your account"
            enabled={settings.twoFactorAuth}
            onToggle={() => handleToggle('twoFactorAuth')}
          />

          <SettingToggle
            title="Profile Privacy"
            description="Hide personal information from other users"
            enabled={settings.privacy}
            onToggle={() => handleToggle('privacy')}
          />
        </section>

        {/* Save Button */}
        <div className="flex justify-end">
          <button
            onClick={handleSave}
            className={`px-6 py-2 text-white rounded-lg font-medium transition ${
              saving ? 'bg-gray-400 cursor-not-allowed' : 'bg-green-600 hover:bg-green-700'
            }`}
            disabled={saving}
          >
            {saving ? 'Saving...' : 'Save Changes'}
          </button>
        </div>
      </div>
    </main>
  );
}

// Reusable toggle row
function SettingToggle({
  title,
  description,
  enabled,
  onToggle,
}: {
  title: string;
  description: string;
  enabled: boolean;
  onToggle: () => void;
}) {
  return (
    <div className="flex justify-between items-center">
      <div>
        <h3 className="text-md font-medium text-gray-800 dark:text-white">{title}</h3>
        <p className="text-sm text-gray-500  dark:text-gray-400">{description}</p>
      </div>
      <button onClick={onToggle}>
        {enabled ? (
          <ToggleRight className="w-8 h-8 text-green-500" />
        ) : (
          <ToggleLeft className="w-8 h-8 text-gray-400" />
        )}
      </button>
    </div>
  );
}

// Reusable select row
function SettingSelect({
  title,
  options,
  value,
  onChange,
}: {
  title: string;
  options: string[];
  value: string;
  onChange: (value: string) => void;
}) {
  return (
    <div>
      <label className="block text-md font-medium text-gray-800 dark:text-white mb-1">{title}</label>
      <select
        value={value}
        onChange={e => onChange(e.target.value)}
        className="w-full px-3 py-2 border rounded-lg bg-white dark:bg-gray-700 border-gray-300 dark:border-gray-600 text-gray-700 dark:text-white focus:outline-none focus:ring-2 focus:ring-green-500"
      >
        {options.map(opt => (
          <option key={opt}>{opt}</option>
        ))}
      </select>
    </div>
  );
}
