"use client";

import React, { useEffect, useState } from "react";
import { useSearchParams, useRouter } from "next/navigation";
import {
  Button,
  Card,
  Table,
  Tag,
  Breadcrumb,
  Spin,
  Alert,
  Input,
  Select,
  Modal,
  Form,
  Pagination,
  Tooltip,
  DatePicker,
} from "antd";

import {
  ArrowLeft,
  Users,
  Home,
  Edit,
  Trash2,
  Search,
  FileText,
  User,
} from "lucide-react";
import { LiaUserEditSolid } from "react-icons/lia";

import dayjs from "@/data/utils/dayjs";
import { toast } from "sonner";
import dynamic from "next/dynamic";

import {
  getTrainingSessionByCode,
  createAttendance,
  getAllAttendance,
  updateAttendance,
  deleteAttendance,
} from "@/app/api/trainingsession/trainingsession";

import type {
  AttendanceRecord,
  TrainingSessionRecord,
} from "@/app/api/trainingsession/trainingsession";
import { EditOutlined } from "@ant-design/icons";

/* Dynamic components */
const WidgetCard = dynamic(
  () => import("../../../components/common/WidgetCard"),
  {
    ssr: false,
  }
);
const ConfirmDialog = dynamic(
  () => import("../../../components/common/ConfirmDialog"),
  {
    ssr: false,
  }
);

const { TextArea } = Input;

/* ---------- Local types ---------- */
interface BackendEmployee {
  employee_id: string;
  employee_name: string;
  department?: string | null;
  profile_photo?: string | null;
}

interface EmployeeForAttendance {
  employee_id: string;
  employee_name: string;
  department?: string;
  profile_photo?: string | null;
  attended?: "Present" | "Absent" | "Leave" | "Training cancelled" | "Late";
  remarks?: string;
}

/* ---------- Component ---------- */
export default function RecordAttendancePage() {
  const searchParams = useSearchParams();
  const router = useRouter();
  const sessionQuery = searchParams.get("session");

  // session + data
  const [session, setSession] = useState<TrainingSessionRecord | null>(null);
  const [employees, setEmployees] = useState<EmployeeForAttendance[]>([]);
  const [existingAttendance, setExistingAttendance] = useState<
    AttendanceRecord[]
  >([]);
  const [loading, setLoading] = useState(true);

  // saving states
  const [saving, setSaving] = useState(false);

  // UI
  const [searchText, setSearchText] = useState("");
  const [statusFilter, setStatusFilter] = useState<
    "All" | "Present" | "Absent" | "On Leave" | "Training cancelled" | "Late"
  >("All");

  // Session fields (editable)
  const [sessionCodeInput, setSessionCodeInput] = useState<string>(
    sessionQuery || ""
  );
  const [sessionDateInput, setSessionDateInput] = useState<string>("");

  // edit modal for existing attendance
  const [editModal, setEditModal] = useState(false);
  const [editForm] = Form.useForm();
  const [selectedAttendance, setSelectedAttendance] =
    useState<AttendanceRecord | null>(null);

  // New single-attendance modal
  const [attendanceModal, setAttendanceModal] = useState(false);
  const [attendanceForm] = Form.useForm();

  // confirm dialog
  const [confirmDialog, setConfirmDialog] = useState({
    open: false,
    title: "",
    message: "",
    onConfirm: () => {},
  });

  // pagination
  const [currentPage, setCurrentPage] = useState(1);
  const [pageSize, setPageSize] = useState(12);

  // existing pagination
  const [existingPage, setExistingPage] = useState(1);
  const [existingPageSize, setExistingPageSize] = useState(10);

  /* ------------------ Fetch session + attendance + employees ------------------ */
  useEffect(() => {
    if (!sessionQuery) {
      toast.error("No session selected");
      router.push("/trainingsession");
      return;
    }

    const load = async () => {
      try {
        setLoading(true);

        const s = await getTrainingSessionByCode(sessionQuery);
        setSession(s);

        setSessionCodeInput(s.session_code || sessionQuery);
        setSessionDateInput(
          s.session_date ? dayjs(s.session_date).format("YYYY-MM-DD") : ""
        );

        await loadExistingAttendance(sessionQuery);
        await reloadEmployees(s);
      } catch (err) {
        toast.error("Failed to load session");
        router.push("/trainingsession");
      } finally {
        setLoading(false);
      }
    };

    load();
  }, [sessionQuery]);

  /* ------------------ Fetch employees function ------------------ */
  const fetchEmployeesFromBackend = async (code: string) => {
    try {
      const token = localStorage.getItem("authToken") || "";
      const url = `/api/trainingsession/employees?session=${encodeURIComponent(
        code
      )}`;

      const res = await fetch(url, {
        method: "GET",
        headers: { Accept: "application/json", Authorization: token },
      });

      if (!res.ok) return null;

      const body = await res.json();

      return Array.isArray(body)
        ? body
        : Array.isArray(body?.data)
        ? body.data
        : Array.isArray(body?.employees)
        ? body.employees
        : null;
    } catch {
      return null;
    }
  };

  /* ------------------ Reload employees ------------------ */
  const reloadEmployees = async (sessionData?: TrainingSessionRecord) => {
    const s = sessionData || session;
    if (!s) return;

    const fetched = await fetchEmployeesFromBackend(s.session_code);

    if (fetched && fetched.length > 0) {
      setEmployees(
        fetched.map((f: BackendEmployee) => ({
          employee_id: f.employee_id,
          employee_name: f.employee_name || f.employee_id,
          department: f.department || undefined,
          profile_photo: f.profile_photo || null,
          attended: undefined,
          remarks: "",
        }))
      );
    } else if (s.employee_count && s.employee_count > 0) {
      const mock = Array.from({ length: s.employee_count }, (_, i) => ({
        employee_id: `EMP${1000 + i}`,
        employee_name: `Employee ${i + 1}`,
        department: ["Production", "Quality", "HR"][i % 3],
        attended: undefined,
        remarks: "",
      }));
      setEmployees(mock);
    } else {
      setEmployees([]);
    }
  };

  /* ------------------ Attendance list ------------------ */
  const loadExistingAttendance = async (code: string) => {
    try {
      const all = await getAllAttendance();
      setExistingAttendance(all.filter((a) => a.session_code === code));
    } catch {
      toast.error("Unable to load attendance records");
    }
  };

  /* ------------------ Filters ------------------ */
  const filteredEmployees = employees.filter((e) => {
    const matchesText =
      !searchText ||
      e.employee_id.toLowerCase().includes(searchText.toLowerCase());
    const matchesStatus =
      statusFilter === "All" ? true : e.attended === statusFilter;
    return matchesText && matchesStatus;
  });

  const filteredExisting = existingAttendance.filter((a) => {
    const matchesText =
      !searchText ||
      (a.employee_name || "")
        .toLowerCase()
        .includes(searchText.toLowerCase()) ||
      (a.attendance_code || "")
        .toLowerCase()
        .includes(searchText.toLowerCase());
    const matchesStatus =
      statusFilter === "All" ? true : a.attended === statusFilter;
    return matchesText && matchesStatus;
  });

  /* ------------------ Change handlers ------------------ */
  function handleChangeEmployeeField(
    id: string,
    field: "attended" | "remarks",
    value: string | undefined
  ) {
    setEmployees((prev) =>
      prev.map((e) => (e.employee_id === id ? { ...e, [field]: value } : e))
    );
  }

  /* ------------------ Save all selected rows ------------------ */
  const handleSaveAll = async () => {
    const rowsToSave = employees.filter((e) => e.attended);

    if (rowsToSave.length === 0) {
      toast.error("Select attendance for at least one row.");
      return;
    }

    const sc = (sessionCodeInput || "").trim();
    const sd = (sessionDateInput || "").trim();

    if (!sc || !sd) {
      toast.error("Session Code and Session Date are required.");
      return;
    }

    setSaving(true);

    try {
      await Promise.all(
        rowsToSave.map((r) =>
          createAttendance({
            session_code: sc,
            session_date: sd,
            attended: r.attended!,
            remarks: r.remarks || "",
            // employee_name: r.employee_name,
          })
        )
      );

      toast.success("Selected attendance rows saved!");
      await loadExistingAttendance(sc);
      await reloadEmployees();
    } catch (err) {
      toast.error("Failed to save attendance. Try again.");
    } finally {
      setSaving(false);
    }
  };

  /* ------------------ Save single attendance from modal ------------------ */
  const handleSaveSingleAttendance = async () => {
    try {
      const values = await attendanceForm.validateFields();

      await createAttendance({
        session_code: values.session_code,
        session_date:
          values.session_date?.format("YYYY-MM-DD") || sessionDateInput,
        attended: values.attended,
        remarks: values.remarks || "",
        // employee_name: values.employee_name || "",
      });

      toast.success("Attendance created");
      setAttendanceModal(false);
      attendanceForm.resetFields();

      await loadExistingAttendance(values.session_code);
      await reloadEmployees();
    } catch (err) {
      toast.error("Failed to create attendance");
    }
  };

  /* ------------------ Edit / Delete existing ------------------ */
  const openEditModal = (record: AttendanceRecord) => {
    setSelectedAttendance(record);
    editForm.setFieldsValue({
      score: record.score,
      certification_status: record.certification_status,
      remarks: record.remarks,
    });
    setEditModal(true);
  };

  const handleUpdateAttendance = async () => {
    if (!selectedAttendance) return;
    try {
      const values = await editForm.validateFields();
      await updateAttendance(selectedAttendance.attendance_code, {
        score: values.score,
        certification_status: values.certification_status,
        remarks: values.remarks,
      });
      toast.success("Updated successfully!");
      setEditModal(false);
      if (sessionQuery) await loadExistingAttendance(sessionQuery);
    } catch {
      toast.error("Update failed");
    }
  };

  const handleDelete = (attendance_code: string) => {
    setConfirmDialog({
      open: true,
      title: "Delete Attendance",
      message: "Are you sure you want to delete this attendance record?",
      onConfirm: async () => {
        try {
          await deleteAttendance(attendance_code);
          toast.success("Deleted");
          setConfirmDialog((d) => ({ ...d, open: false }));
          if (sessionQuery) await loadExistingAttendance(sessionQuery);
        } catch {
          toast.error("Delete failed");
          setConfirmDialog((d) => ({ ...d, open: false }));
        }
      },
    });
  };

  /* ------------------ Columns ------------------ */
  const employeeColumns = [
    {
      title: "Employee ID",
      dataIndex: "employee_id",
      key: "employee_id",
      render: (text: string) => <strong>{text}</strong>,
    },
    {
      title: "Session Code",
      key: "session_code",
      render: () => (
        <Input
          value={sessionCodeInput}
          onChange={(e) => setSessionCodeInput(e.target.value)}
          placeholder="Session Code (applies to all rows)"
          style={{ minWidth: 160 }}
        />
      ),
    },
    {
      title: "Session Date",
      key: "session_date",
      render: () => (
        <Input
          value={sessionDateInput}
          onChange={(e) => setSessionDateInput(e.target.value)}
          placeholder="YYYY-MM-DD (applies to all rows)"
          style={{ minWidth: 140 }}
        />
      ),
      width: 160,
    },
    {
      title: "Attended",
      dataIndex: "attended",
      key: "attended",
      render: (_: any, r: EmployeeForAttendance) => (
        <Select
          value={r.attended}
          onChange={(v) =>
            handleChangeEmployeeField(r.employee_id, "attended", v)
          }
          placeholder="Select"
          style={{ minWidth: 140 }}
        >
          <Select.Option value="Present">Present</Select.Option>
          <Select.Option value="Absent">Absent</Select.Option>
          <Select.Option value="On Leave">On Leave</Select.Option>
          <Select.Option value="Late">Late</Select.Option>
          <Select.Option value="Training Cancelled">
            Training Cancelled
          </Select.Option>
        </Select>
      ),
      width: 220,
    },
    {
      title: "Remarks",
      dataIndex: "remarks",
      key: "remarks",
      render: (_: any, r: EmployeeForAttendance) => (
        <TextArea
          rows={1}
          value={r.remarks}
          onChange={(e) =>
            handleChangeEmployeeField(r.employee_id, "remarks", e.target.value)
          }
        />
      ),
    },
  ];

  const existingColumns = [
    {
      title: "Attendance Code",
      dataIndex: "attendance_code",
      key: "attendance_code",
    },
    { title: "Employee", dataIndex: "employee_name", key: "employee_name" },
    {
      title: "Attended",
      dataIndex: "attended",
      key: "attended",
      render: (v: string) => (
        <Tag
          color={
            v === "Present"
              ? "green"
              : v === "Absent"
              ? "red"
              : v === "Leave"
              ? "blue"
              : v === "Training cancelled"
              ? "gray"
              : v === "Late"
              ? "orange"
              : "default"
          }
        >
          {v}
        </Tag>
      ),
    },
    { title: "Score", dataIndex: "score", key: "score" },
    {
      title: "Certification",
      dataIndex: "certification_status",
      key: "certification_status",
    },
    { title: "Remarks", dataIndex: "remarks", key: "remarks" },
    {
      title: "Actions",
      key: "actions",
      render: (_: any, r: AttendanceRecord) => (
        <div className="flex gap-2">
          <Tooltip title="Edit">
            <Button
              icon={<EditOutlined />}
              size="small"
              type="primary"
              onClick={() => openEditModal(r)}
            />
          </Tooltip>
          <Tooltip title="Delete">
            <Button
              icon={<Trash2 className="w-4 h-4" />}
              size="small"
              danger
              onClick={() => handleDelete(r.attendance_code)}
            />
          </Tooltip>
        </div>
      ),
    },
  ];

  /* ------------------ Loading state ------------------ */
  if (loading)
    return (
      <div className="flex justify-center py-32">
        <Spin size="large" />
      </div>
    );
  if (!session)
    return <Alert message="Session not found" type="error" showIcon />;

  /* ------------------ Render ------------------ */
  return (
    <div className="p-6 max-w-7xl mx-auto">
      {/* HEADER */}
      <div className="flex items-center justify-between mb-6">
        <div className="flex items-center gap-3">
          <Users className="w-9 h-9 text-blue-600" />
          <div>
            <h1 className="text-3xl font-bold">Record Attendance</h1>
          </div>
        </div>
        <div className="flex items-center gap-2">
          <Tooltip title="Back">
            <Button icon={<ArrowLeft />} onClick={() => router.back()}>
              Back
            </Button>
          </Tooltip>
        </div>
      </div>

      {/* BREADCRUMB */}
      <Breadcrumb
        className="mb-6"
        items={[
          { title: <Home className="w-4 h-4" />, href: "/hrm" },
          { title: "Training Management", href: "/trainingsession" },
          { title: "Record Attendance" },
        ]}
      />

      {/* SESSION INFO + SUMMARY */}
      <div className="grid grid-cols-1 md:grid-cols-4 gap-4 mb-6 mt-3">
        <Card>
          <div className="text-sm">
            <p>
              <strong>Training Code:</strong> {session.training_code}
            </p>
            <p>
              <strong>Date:</strong>{" "}
              {dayjs(session.session_date).format("DD MMM YYYY")}
            </p>
            <p>
              <strong>Duration:</strong> {session.duration_hours} hrs
            </p>
            <p>
              <strong>Trainer:</strong> {session.trainer_name}
            </p>
          </div>
        </Card>
        {/* <WidgetCard
          title="Total Employees"
          value={employees.length}
          icon={<User />}
          color="blue"
          percentage={0}
          description="Employees for this session"
        /> */}
        <WidgetCard
          title="Present"
          value={
            existingAttendance.filter((x) => x.attended === "Present").length
          }
          icon={<FileText />}
          color="green"
          percentage={0}
          description="Saved present"
        />
        <WidgetCard
          title="Absent / Leave"
          value={
            existingAttendance.filter(
              (x) => x.attended === "Absent" || x.attended === "Leave"
            ).length
          }
          icon={<FileText />}
          color="red"
          percentage={0}
          description="Saved absent/leave"
        />
      </div>

      {/* EXISTING ATTENDANCE */}
      <Card title={`Existing Attendance (${existingAttendance.length})`}>
        <div className="flex flex-wrap gap-3 items-center mb-5">
          <Input
            placeholder="Search by ID or existing attendance code..."
            value={searchText}
            onChange={(e) => setSearchText(e.target.value)}
            style={{ width: 320 }}
            prefix={<Search className="w-4 h-4" />}
          />
          <Select
            value={statusFilter}
            onChange={(v) => setStatusFilter(v)}
            style={{ width: 180 }}
            options={[
              { label: "All", value: "All" },
              { label: "Present", value: "Present" },
              { label: "Absent", value: "Absent" },
              { label: "On Leave", value: "On Leave" },
              { label: "Late", value: "Late" },
              { label: "Training Cancelled", value: "Training Cancelled" },
            ]}
          />
          <div className="flex-1" />
          <Button
            type="primary"
            onClick={() => {
              attendanceForm.resetFields();
              attendanceForm.setFieldsValue({
                session_code: sessionCodeInput,
                session_date: sessionDateInput
                  ? dayjs(sessionDateInput)
                  : dayjs(session?.session_date),
                attended: undefined,
                remarks: "",
              });
              setAttendanceModal(true);
            }}
          >
            + New Attendance
          </Button>
        </div>

        <Table
          columns={existingColumns}
          dataSource={filteredExisting.slice(
            (existingPage - 1) * existingPageSize,
            existingPage * existingPageSize
          )}
          rowKey="attendance_code"
          pagination={{
            current: existingPage,
            pageSize: existingPageSize,
            total: filteredExisting.length,
            onChange: (p, s) => {
              setExistingPage(p);
              if (s) setExistingPageSize(s);
            },
            showSizeChanger: true,
            pageSizeOptions: ["10", "25", "50"],
          }}
        />
      </Card>

      {/* EDIT MODAL - FULLY FIXED */}
      <Modal
        title="Edit Attendance"
        open={editModal}
        onCancel={() => setEditModal(false)}
        onOk={handleUpdateAttendance}
        okText="Update"
        destroyOnHidden
        styles={{ body: { maxHeight: "70vh", overflowY: "auto" } }}
      >
        <Form form={editForm} layout="vertical">
          <Form.Item name="score" label="Score">
            <Input type="number" placeholder="Enter score" />
          </Form.Item>
          <Form.Item name="certification_status" label="Certification Status">
            <Select placeholder="Select">
              <Select.Option value="Certified">Certified</Select.Option>
              <Select.Option value="Not Certified">Not Certified</Select.Option>
            </Select>
          </Form.Item>
          <Form.Item name="remarks" label="Remarks">
            <TextArea rows={3} />
          </Form.Item>
        </Form>
      </Modal>

      {/* NEW SINGLE ATTENDANCE MODAL - FULLY FIXED */}
      <Modal
        title={
          <div className="flex items-center gap-3">
            <Users className="w-6 h-6 text-blue-600" />
            New Attendance
          </div>
        }
        open={attendanceModal}
        onOk={handleSaveSingleAttendance}
        onCancel={() => {
          setAttendanceModal(false);
          attendanceForm.resetFields();
        }}
        width="70vw"
        className="!top-10"
        destroyOnHidden
        styles={{ body: { maxHeight: "70vh", overflowY: "auto" } }}
      >
        <Form form={attendanceForm} layout="vertical" className="mt-4">
          <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
            <Form.Item
              name="session_code"
              label="Session Code"
              rules={[{ required: true, message: "Session Code is required" }]}
            >
              <Input
                disabled
                value={sessionCodeInput}
                className="bg-gray-100 font-semibold"
              />
            </Form.Item>

            <Form.Item
              name="session_date"
              label="Session Date"
              rules={[{ required: true, message: "Session Date is required" }]}
            >
              <DatePicker
                disabled
                value={sessionDateInput ? dayjs(sessionDateInput) : null}
                className="w-full"
              />
            </Form.Item>

            <Form.Item
              name="attended"
              label="Attended "
              rules={[{ required: true, message: "Select attendance" }]}
            >
              <Select placeholder="Select attendance">
                <Select.Option value="Absent">Absent</Select.Option>
                <Select.Option value="Late">Late</Select.Option>
                <Select.Option value="On Leave">On Leave</Select.Option>
                <Select.Option value="Present">Present</Select.Option>
                <Select.Option value="Training Cancelled">
                  Training Cancelled
                </Select.Option>
              </Select>
            </Form.Item>

            <Form.Item name="remarks" label="Remarks">
              <TextArea rows={3} placeholder="Optional remark" />
            </Form.Item>
          </div>
        </Form>
      </Modal>

      {/* CONFIRM DIALOG */}
      {typeof ConfirmDialog !== "undefined" && (
        <ConfirmDialog
          open={confirmDialog.open}
          title={confirmDialog.title}
          message={confirmDialog.message}
          onClose={() => setConfirmDialog((d) => ({ ...d, open: false }))}
          onConfirm={confirmDialog.onConfirm}
        />
      )}
    </div>
  );
}
