"use client";

import React, { useEffect, useState } from "react";
import Button from "antd/es/button";
import "antd/es/button/style";
import Form from "antd/es/form";
import "antd/es/form/style";
import Input from "antd/es/input";
import "antd/es/input/style";
import Modal from "antd/es/modal";
import "antd/es/modal/style";
import Select from "antd/es/select";
import "antd/es/select/style";
import Space from "antd/es/space";
import "antd/es/space/style";
import Tooltip from "antd/es/tooltip";
import "antd/es/tooltip/style";
import Tag from "antd/es/tag";
import "antd/es/tag/style";
import Breadcrumb from "antd/es/breadcrumb";
import "antd/es/breadcrumb/style";
import Card from "antd/es/card";
import "antd/es/card/style";
import DatePicker from "antd/es/date-picker";
import "antd/es/date-picker/style";
import Switch from "antd/es/switch";
import "antd/es/switch/style";
import Pagination from "antd/es/pagination";
import "antd/es/pagination/style";
import Dropdown from "antd/es/dropdown";
import "antd/es/dropdown/style";
import {
  PlusOutlined,
  EditOutlined,
  DeleteOutlined,
  HomeOutlined,
  TableOutlined,
  AppstoreOutlined,
  CheckCircleTwoTone,
  EyeOutlined,
  SearchOutlined,
  CheckCircleOutlined,
  PauseCircleOutlined,
  CloseCircleOutlined,
  TeamOutlined,
  CalendarOutlined,
  ClockCircleOutlined,
  IdcardOutlined,
} from "@ant-design/icons";
import type { ColumnsType } from "antd/es/table";
import DataTable from "../../components/common/DataTable";
import { useIsMobile } from "@/data/useIsMobile";
import { toast } from "sonner";
import { motion } from "framer-motion";
import { useRouter } from "next/navigation";
import {
  createTrainingSession,
  createAttendance,
  deleteTrainingSession,
  getAllTrainingSessions,
  updateTrainingSession,
  updateTrainingSessionStatus,
} from "@/app/api/trainingsession/trainingsession";
import dayjs from "@/data/utils/dayjs";
import dynamic from "next/dynamic";
import GlassCard from "../../components/common/GlassCard";
import { FaFileExport, FaTrash, FaFileExcel, FaFilePdf } from "react-icons/fa";
import { BookOpen } from "lucide-react";
import type { TrainingSessionRecord } from "@/app/api/trainingsession/trainingsession";
import * as XLSX from "xlsx";
import { saveAs } from "file-saver";
import jsPDF from "jspdf";
import autoTable from "jspdf-autotable";
import {
  getAllManageUsers,
  ManageUser,
} from "@/app/api/manageusers/manageusers";
import { getAllTrainingNeeds } from "@/app/api/trainingneedidentification/trainingneeds";

const WidgetCard = dynamic(() => import("../../components/common/WidgetCard"), {
  ssr: false,
});
const ConfirmDialog = dynamic(
  () => import("../../components/common/ConfirmDialog"),
  { ssr: false }
);
const ColumnVisibilityToggle = dynamic(
  () => import("../../components/common/ManageColumn"),
  { ssr: false }
);

const { RangePicker } = DatePicker;
const statusOptions = ["Pending", "Approved", "Rejected"] as const;

export default function TrainingSessionsPage() {
  const [form] = Form.useForm();
  const [approvalForm] = Form.useForm();

  const [attendanceForm] = Form.useForm();
  const [isAttendanceModalOpen, setIsAttendanceModalOpen] = useState(false);
  const [attendanceSession, setAttendanceSession] =
    useState<TrainingSessionRecord | null>(null);

  const [sessions, setSessions] = useState<TrainingSessionRecord[]>([]);
  const [trainingNeedCodes, setTrainingNeedCodes] = useState<string[]>([]);
  const [managers, setManagers] = useState<{ id: string; name: string }[]>([]);
  const [editingApprovalSession, setEditingApprovalSession] =
    useState<TrainingSessionRecord | null>(null);
  const [forceRender, setForceRender] = useState(0);

  const [isModalOpen, setIsModalOpen] = useState(false);
  const [isApprovalModalOpen, setIsApprovalModalOpen] = useState(false);
  const [editingSession, setEditingSession] =
    useState<TrainingSessionRecord | null>(null);
  const [searchTerm, setSearchTerm] = useState("");
  const [confirmDialog, setConfirmDialog] = useState({
    open: false,
    onConfirm: () => {},
    title: "",
    message: "",
  });
  const [loading, setLoading] = useState(false);
  const [dateRange, setDateRange] = useState<[dayjs.Dayjs, dayjs.Dayjs] | null>(
    null
  );
  const [isGridView, setIsGridView] = useState(false);
  const [selectedRows, setSelectedRows] = useState<string[]>([]);
  const [advancedFilters, setAdvancedFilters] = useState<{
    status: (typeof statusOptions)[number] | undefined;
  }>({ status: undefined });
  const [currentPage, setCurrentPage] = useState(1);
  const [pageSize, setPageSize] = useState(10);
  const isMobile = useIsMobile();
  const router = useRouter();

  useEffect(() => {
    const fetchSessions = async () => {
      try {
        setLoading(true);
        const data = await getAllTrainingSessions();
        setSessions(data);
      } catch {
        toast.error("Failed to fetch training sessions");
      } finally {
        setLoading(false);
      }
    };
    fetchSessions();
  }, []);

  // ================= FETCH TRAINING NEED IDENTIFICATION CODES ================= //
  useEffect(() => {
    const fetchTrainingCodes = async () => {
      try {
        const needs = await getAllTrainingNeeds();

        // Extract unique training_code from training needs list
        const codes = needs
          .map((n: any) => n.training_code as string)
          .filter(Boolean);

        setTrainingNeedCodes(Array.from(new Set(codes)) as string[]);
      } catch (error) {
        toast.error("Failed to load training codes");
      }
    };

    fetchTrainingCodes();
  }, []);
  // =========================================================================== //

  useEffect(() => {
    const fetchUsers = async () => {
      try {
        const users: ManageUser[] = await getAllManageUsers();

        const formatted = users.map((u) => ({
          id: u.id || u.email, // backend ID
          name: `${u.First_name} ${u.Last_name || ""}`.trim(), // UI display text
        }));

        setManagers(formatted);
      } catch {
        toast.error("Failed to load trainers");
      }
    };

    fetchUsers();
  }, []);

  const openAddModal = () => {
    setEditingSession(null);
    form.resetFields();
    setIsModalOpen(true);
  };

  const openEditModal = (session: TrainingSessionRecord) => {
    setEditingSession(session);
    form.resetFields();

    const trainer = managers.find((m) => m.id === session.trainer_id);

    form.setFieldsValue({
      session_code: session.session_code,
      training_code: session.training_code,
      session_date: dayjs(session.session_date),
      duration_hours: session.duration_hours,
      trainer_id: undefined,
      venue: session.venue,
      remarks: session.remarks,
    });

    setIsModalOpen(true);
  };

  const openApprovalModal = (session: TrainingSessionRecord) => {
    setEditingApprovalSession(session);
    approvalForm.resetFields();

    const initialFields = {
      approvalStatus: session.status || "Pending",
      comments: session.comments || undefined,
      rejectionReason:
        session.status === "Rejected"
          ? session.rejection_reason || undefined
          : undefined,
      approvedBy: session.status === "Approved" ? "Current User" : undefined,
      approvedDate: session.status === "Approved" ? dayjs() : undefined,
    };

    approvalForm.setFieldsValue(initialFields);
    setForceRender((prev) => prev + 1);
    setIsApprovalModalOpen(true);
  };

  const openAttendanceModal = (session: TrainingSessionRecord) => {
    setAttendanceSession(session);
    attendanceForm.resetFields();

    attendanceForm.setFieldsValue({
      attended: undefined,
      remarks: "",
    });

    setIsAttendanceModalOpen(true);
  };

  const handleApprovalFormSubmit = async () => {
    if (!editingApprovalSession) return;

    try {
      const values = await approvalForm.validateFields();

      const payload = {
        status: values.approvalStatus,
        comments: values.comments || "",
        rejection_reason: values.rejectionReason || "",
      };

      // Use the correct API function!
      await updateTrainingSessionStatus(
        editingApprovalSession.session_code,
        payload
      );

      toast.success("Approval status updated successfully");

      const updated = await getAllTrainingSessions();
      setSessions(updated);

      setIsApprovalModalOpen(false);
      approvalForm.resetFields();
    } catch (err: any) {
      // Show backend validation error (e.g., rejection reason required)
      if (err?.message?.includes("Rejection reason")) {
        toast.error("Rejection reason is required when status is Rejected");
      } else {
        toast.error("Failed to update approval status");
      }
    }
  };

  const handleAttendanceSubmit = async () => {
    if (!attendanceSession) return;

    try {
      const values = await attendanceForm.validateFields();

      await createAttendance({
        session_code: attendanceSession.session_code,
        session_date: attendanceSession.session_date,
        attended: values.attended,
        remarks: values.remarks || "",
      });

      toast.success("Attendance recorded successfully");

      setIsAttendanceModalOpen(false);
      attendanceForm.resetFields();
    } catch (err) {
      toast.error("Failed to save attendance");
    }
  };

  const handleDelete = (session_code: string) => {
    setConfirmDialog({
      open: true,
      title: "Delete Session",
      message: "Are you sure you want to delete this session?",
      onConfirm: async () => {
        await deleteTrainingSession(session_code);
        const updated = await getAllTrainingSessions();
        setSessions(updated);
        toast.success("Deleted");
      },
    });
  };

  const handleBulkDelete = () => {
    if (selectedRows.length === 0) return;

    setConfirmDialog({
      open: true,
      title: "Delete Multiple Sessions",
      message: `Delete ${selectedRows.length} selected session(s)?`,
      onConfirm: async () => {
        try {
          await Promise.all(
            selectedRows.map((code) => deleteTrainingSession(code))
          );
          const updated = await getAllTrainingSessions();
          setSessions(updated);
          setSelectedRows([]);
          toast.success("Deleted successfully");
        } catch {
          toast.error("Bulk delete failed");
        }
      },
    });
  };

  const handleFormSubmit = async () => {
    try {
      const values = await form.validateFields();
      const payload = {
        training_code: values.training_code,
        session_date: values.session_date.format("YYYY-MM-DD"),
        duration_hours: values.duration_hours,
        trainer_id: values.trainer_id,
        venue: values.venue || "",
        remarks: values.remarks || "",
      };
      if (editingSession) {
        await updateTrainingSession(editingSession.session_code, payload);
        toast.success("Updated");
      } else {
        await createTrainingSession(payload);
        toast.success("Created");
      }
      const updated = await getAllTrainingSessions();
      setSessions(updated);
      setIsModalOpen(false);
      form.resetFields();
    } catch {
      toast.error("Save failed");
    }
  };

  const renderStatusTag = (status: string) => {
    const colorMap: Record<string, string> = {
      Pending: "orange",
      Approved: "green",
      Rejected: "red",
    };
    const iconMap: Record<string, React.ReactNode> = {
      Pending: <PauseCircleOutlined />,
      Approved: <CheckCircleOutlined twoToneColor="#52c41a" />,
      Rejected: <CloseCircleOutlined />,
    };
    return (
      <Tag icon={iconMap[status]} color={colorMap[status] || "default"}>
        {status}
      </Tag>
    );
  };

  const filteredSessions = sessions.filter((s) => {
    const matchesSearch =
      s.session_code.toLowerCase().includes(searchTerm.toLowerCase()) ||
      s.training_code.toLowerCase().includes(searchTerm.toLowerCase()) ||
      (s.topic ?? "").toLowerCase().includes(searchTerm.toLowerCase()) ||
      (s.trainer_name ?? "").toLowerCase().includes(searchTerm.toLowerCase());

    const matchesDate =
      !dateRange ||
      (dayjs(s.session_date).isSameOrAfter(dateRange[0].startOf("day")) &&
        dayjs(s.session_date).isSameOrBefore(dateRange[1].endOf("day")));

    const matchesStatus =
      !advancedFilters.status || s.status === advancedFilters.status;

    return matchesSearch && matchesDate && matchesStatus;
  });

  // ================= EXPORT FUNCTIONS ================= //
  const exportExcel = () => {
    const data = filteredSessions.map((s) => ({
      "Session Code": s.session_code,
      "Training Code": s.training_code,
      Topic: s.topic || "-",
      Trainer: s.trainer_name || "-",
      Date: dayjs(s.session_date).format("DD-MM-YYYY"),
      "Duration (hrs)": s.duration_hours,
      Attendees: s.employee_count,
      Venue: s.venue || "-",
      Status: s.status,
    }));

    const worksheet = XLSX.utils.json_to_sheet(data);
    const workbook = XLSX.utils.book_new();
    XLSX.utils.book_append_sheet(workbook, worksheet, "Training Sessions");

    const excelBuffer = XLSX.write(workbook, {
      bookType: "xlsx",
      type: "array",
    });

    saveAs(
      new Blob([excelBuffer], { type: "application/octet-stream" }),
      "Training_Sessions.xlsx"
    );
    toast.success("Excel exported successfully!");
  };

  const exportPDF = () => {
    const doc = new jsPDF("landscape");
    doc.setFontSize(14);
    doc.text("Training Sessions Report", 14, 10);

    const tableData = filteredSessions.map((s) => [
      s.session_code,
      s.training_code,
      s.topic || "-",
      s.trainer_name || "-",
      dayjs(s.session_date).format("DD-MM-YYYY"),
      s.duration_hours,
      s.employee_count,
      s.status,
    ]);

    autoTable(doc, {
      head: [
        [
          "Session Code",
          "Training Code",
          "Topic",
          "Trainer",
          "Date",
          "Duration",
          "Attendees",
          "Status",
        ],
      ],
      body: tableData,
      startY: 20,
      styles: { fontSize: 9 },
      headStyles: { fillColor: [22, 160, 133] },
    });

    doc.save("Training_Sessions.pdf");
    toast.success("PDF exported successfully!");
  };

  const handleExport = (type: "pdf" | "excel") => {
    if (type === "excel") exportExcel();
    if (type === "pdf") exportPDF();
  };
  // ==================================================== //

  const paginatedSessions = filteredSessions.slice(
    (currentPage - 1) * pageSize,
    currentPage * pageSize
  );

  // ================= COLUMN VISIBILITY CONFIG ================= //
  const allColumns = [
    { key: "selection", label: "Select" },
    { key: "session_code", label: "Session Code" },
    { key: "training_code", label: "Training Code" },
    { key: "topic", label: "Topic" },
    { key: "trainer_name", label: "Trainer" },
    { key: "session_date", label: "Date" },
    { key: "duration_hours", label: "Duration (hrs)" },
    { key: "employee_count", label: "Attendees" },
    { key: "venue", label: "Venue" },
    { key: "status", label: "Status" },
    { key: "actions", label: "Actions" },
  ];

  const columnKeys = allColumns.map((c) => c.key);
  const labelsMap = Object.fromEntries(allColumns.map((c) => [c.key, c.label]));

  const [visibleColumnKeys, setVisibleColumnKeys] =
    useState<string[]>(columnKeys);

  const handleVisibleColumnChange = (cols: string[]) => {
    setVisibleColumnKeys(cols);
  };
  // ============================================================ //

  const columns: ColumnsType<TrainingSessionRecord> = [
    {
      title: (
        <input
          type="checkbox"
          onChange={(e) => {
            setSelectedRows(
              e.target.checked
                ? filteredSessions.map((s) => s.session_code)
                : []
            );
          }}
          className="w-4 h-4"
        />
      ),
      key: "selection",
      render: (_, record) => (
        <input
          type="checkbox"
          checked={selectedRows.includes(record.session_code)}
          onChange={(e) => {
            setSelectedRows((prev) =>
              e.target.checked
                ? [...prev, record.session_code]
                : prev.filter((id) => id !== record.session_code)
            );
          }}
          className="w-4 h-4"
        />
      ),
      width: 44,
    },

    {
      title: "Session Code",
      dataIndex: "session_code",
      key: "session_code",
      render: (text) => <span className="font-medium">{text}</span>,
    },
    {
      title: "Training Code",
      dataIndex: "training_code",
      key: "training_code",
    },
    {
      title: "Topic",
      dataIndex: "topic",
      key: "topic",
      render: (text) => text || "-",
    },
    {
      title: "Trainer",
      dataIndex: "trainer_name",
      key: "trainer_name",
      render: (text) => text || "-",
    },
    {
      title: "Date",
      dataIndex: "session_date",
      key: "session_date",
      render: (date) => dayjs(date).format("DD-MM-YYYY"),
    },
    {
      title: "Duration (hrs)",
      dataIndex: "duration_hours",
      key: "duration_hours",
    },
    {
      title: "Attendees",
      dataIndex: "employee_count",
      key: "employee_count",
      render: (count) => <Tag color="cyan">{count}</Tag>,
    },
    {
      title: "Venue",
      dataIndex: "venue",
      key: "venue",
      render: (text) => text || "-",
    },
    {
      title: "Status",
      dataIndex: "status",
      key: "status",
      render: renderStatusTag,
    },
    {
      title: "Actions",
      key: "actions",
      fixed: "right" as const,
      render: (_, session) => (
        <Space size="small">
          <Tooltip title="Edit Session">
            <Button
              icon={<EditOutlined />}
              size="small"
              type="primary"
              onClick={() => openEditModal(session)}
            />
          </Tooltip>

          <Tooltip title="Edit Approval Status">
            <Button
              icon={<IdcardOutlined />}
              size="small"
              className="text-purple-600 border-purple-600 hover:bg-purple-50"
              onClick={() => openApprovalModal(session)}
            />
          </Tooltip>

          <Tooltip title="Record Attendance">
            <Button
              icon={<TeamOutlined />}
              size="small"
              className="text-blue-600 border-blue-600 hover:bg-blue-50"
              onClick={() =>
                router.push(
                  `/hrm/trainingsession/recordattendance?session=${session.session_code}`
                )
              }
            />
          </Tooltip>

          <Tooltip title="Delete">
            <Button
              icon={<DeleteOutlined />}
              size="small"
              danger
              onClick={() => handleDelete(session.session_code)}
            />
          </Tooltip>
        </Space>
      ),
    },
  ];
  const filteredColumns = columns.filter(
    (col) => typeof col.key === "string" && visibleColumnKeys.includes(col.key)
  );

  const SessionCard = ({ session }: { session: TrainingSessionRecord }) => (
    <GlassCard gradient blurStrength="lg" hoverLift color="success">
      <div className="border rounded-xl p-4 shadow-sm hover:shadow-md bg-white dark:bg-gray-900 transition-all space-y-3">
        <div className="flex justify-between items-start">
          <h3 className="text-lg font-semibold">
            {session.topic || "Untitled Session"}
          </h3>
          {renderStatusTag(session.status)}
        </div>
        <div className="space-y-1 text-sm">
          <p>
            <strong>Code:</strong> {session.session_code}
          </p>
          <p>
            <strong>Training:</strong> {session.training_code}
          </p>
          <p>
            <strong>Date:</strong>{" "}
            {dayjs(session.session_date).format("DD MMM YYYY")}
          </p>
          <p>
            <strong>Duration:</strong> {session.duration_hours} hrs
          </p>
          <p>
            <strong>Trainer:</strong> {session.trainer_name || "N/A"}
          </p>
          <p>
            <strong>Venue:</strong> {session.venue || "N/A"}
          </p>
          <p>
            <strong>Attendees:</strong>{" "}
            <Tag color="cyan">{session.employee_count}</Tag>
          </p>
        </div>
        <div className="flex gap-2 pt-3 border-t dark:border-gray-700">
          <Button
            icon={<EyeOutlined />}
            size="small"
            onClick={() =>
              router.push(
                `/training/sessiondetail?code=${session.session_code}`
              )
            }
          />
          <Button
            icon={<EditOutlined />}
            size="small"
            type="primary"
            onClick={() => openEditModal(session)}
          />
          <Button
            icon={<IdcardOutlined />}
            size="small"
            className="text-purple-600 border-purple-600"
            onClick={() => openApprovalModal(session)}
          />
          <Button
            icon={<DeleteOutlined />}
            size="small"
            danger
            onClick={() => handleDelete(session.session_code)}
          />
          <Tooltip title="Record Attendance">
            <Button
              icon={<TeamOutlined />}
              size="small"
              className="text-purple-600 border-purple-600 hover:bg-purple-50"
              onClick={() =>
                router.push(
                  `/hrm/trainingsession/recordattendance?session=${session.session_code}`
                )
              }
            />
          </Tooltip>
        </div>
      </div>
    </GlassCard>
  );

  const total = sessions.length;

  const pending = sessions.filter((s) => s.status === "Pending").length;
  const approved = sessions.filter((s) => s.status === "Approved").length;
  const rejected = sessions.filter((s) => s.status === "Rejected").length;

  return (
    <>
      <div className="px-2 sm:px-4 lg:px-6 py-4">
        <h2 className="flex items-center gap-2 text-2xl sm:text-3xl font-bold text-gray-800 dark:text-white mb-2">
          <BookOpen className="w-8 h-8 text-blue-600" />
          <span>Training Sessions</span>
          <span className="text-gray-500 text-lg font-medium">({total})</span>
        </h2>
        <p className="text-gray-600 dark:text-gray-300 mb-4">
          Manage and track all employee training sessions and attendance.
        </p>
        <Breadcrumb
          className="mb-4 sm:mb-6 mt-2 text-sm text-gray-500 dark:text-gray-400"
          separator=">"
          items={[
            {
              title: (
                <Tooltip title="Dashboard">
                  <HomeOutlined />
                </Tooltip>
              ),
              href: "/hrm",
            },
            {
              title: "Training Management",
            },
            {
              title: "Seesion",
            },
          ]}
        />

        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4 mt-3 mb-6">
          <WidgetCard
            title="Total Sessions"
            value={total}
            icon={<CalendarOutlined />}
            color="blue"
            percentage={12}
            description="All sessions"
          />

          <WidgetCard
            title="Pending"
            value={pending}
            icon={<PauseCircleOutlined />}
            color="orange"
            percentage={-3}
            description="Awaiting review"
          />

          <WidgetCard
            title="Approved"
            value={approved}
            icon={<CheckCircleTwoTone twoToneColor="#52c41a" />}
            color="green"
            percentage={8}
            description="Approved sessions"
          />

          <WidgetCard
            title="Rejected"
            value={rejected}
            icon={<CloseCircleOutlined />}
            color="red"
            percentage={-5}
            description="Rejected sessions"
          />
        </div>

        <Card title="Training Session List" className="shadow-sm">
          <div className="p-4 space-y-6">
            <div className="flex flex-col lg:flex-row justify-between gap-4 items-start lg:items-center">
              <Input
                prefix={<SearchOutlined />}
                placeholder="Search by code, topic, trainer..."
                value={searchTerm}
                onChange={(e) => setSearchTerm(e.target.value)}
                className="w-full lg:w-80"
              />
              <div className="flex gap-3 flex-wrap lg:flex-nowrap">
                <Dropdown
                  menu={{
                    items: [
                      {
                        key: "excel",
                        label: (
                          <span className="flex items-center gap-2">
                            <FaFileExcel className="text-green-600" /> Export
                            Excel
                          </span>
                        ),
                      },
                      {
                        key: "pdf",
                        label: (
                          <span className="flex items-center gap-2">
                            <FaFilePdf className="text-red-600" /> Export PDF
                          </span>
                        ),
                      },
                    ],
                    onClick: ({ key }) => handleExport(key as "excel" | "pdf"),
                  }}
                >
                  <Button icon={<FaFileExport />}>Export</Button>
                </Dropdown>

                <Button
                  type="primary"
                  icon={<PlusOutlined />}
                  onClick={openAddModal}
                >
                  New Session
                </Button>
              </div>
            </div>

            <div className="flex flex-col gap-4 lg:flex-row lg:justify-between lg:items-center">
              {/* LEFT SIDE FILTERS */}
              <div className="flex flex-wrap gap-3 w-full lg:w-auto">
                <Select
                  placeholder="Filter by Status"
                  allowClear
                  className="w-full sm:w-48"
                  options={statusOptions.map((s) => ({ label: s, value: s }))}
                  onChange={(v) =>
                    setAdvancedFilters((prev) => ({ ...prev, status: v }))
                  }
                />
              </div>

              {/* RIGHT SIDE CONTROLS */}
              <div className="flex flex-wrap gap-3 w-full lg:w-auto items-center justify-start lg:justify-end">
                <ColumnVisibilityToggle
                  columns={columnKeys}
                  labels={labelsMap}
                  storageKey="training_session_table_visibility"
                  onChange={handleVisibleColumnChange}
                />

                <RangePicker
                  onChange={(dates) => setDateRange(dates as any)}
                  className="w-full sm:w-auto min-w-[200px]"
                />

                <Button
                  danger
                  icon={<FaTrash />}
                  disabled={selectedRows.length === 0}
                  onClick={handleBulkDelete}
                  className="w-full sm:w-auto"
                >
                  Bulk Delete
                </Button>

                <Switch
                  checkedChildren={<AppstoreOutlined />}
                  unCheckedChildren={<TableOutlined />}
                  checked={isGridView}
                  onChange={setIsGridView}
                />
              </div>
            </div>

            {isGridView || isMobile ? (
              <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
                {paginatedSessions.map((session) => (
                  <SessionCard key={session.session_code} session={session} />
                ))}
              </div>
            ) : (
              <DataTable
                columns={filteredColumns}
                dataSource={paginatedSessions}
                rowKey="session_code"
                loading={loading}
                pagination={false}
              />
            )}

            <div className="flex flex-col sm:flex-row justify-between items-start sm:items-center py-2 gap-2 text-sm text-gray-600">
              <span>
                Showing {(currentPage - 1) * pageSize + 1} to{" "}
                {Math.min(currentPage * pageSize, filteredSessions.length)} of{" "}
                {filteredSessions.length}
              </span>
              <Pagination
                current={currentPage}
                pageSize={pageSize}
                total={filteredSessions.length}
                onChange={(p, s) => {
                  setCurrentPage(p);
                  setPageSize(s);
                }}
                showSizeChanger
                pageSizeOptions={["10", "25", "50", "100"]}
              />
            </div>
          </div>
        </Card>

        {/* Create/Edit Training Session Modal - WITH training_code ADDED */}
        <motion.div initial={{ opacity: 0 }} animate={{ opacity: 1 }}>
          <Modal
            title={
              <div className="flex items-center gap-3">
                <CalendarOutlined className="w-6 h-6 text-blue-600" />
                <span>
                  {editingSession ? "Edit" : "Create"} Training Session
                </span>
              </div>
            }
            open={isModalOpen}
            onOk={handleFormSubmit}
            onCancel={() => {
              setIsModalOpen(false);
              form.resetFields();
            }}
            width="95vw"
            className="!top-8"
            styles={{ body: { maxHeight: "80vh", overflowY: "auto" } }}
          >
            <Form form={form} layout="vertical" className="mt-4">
              <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
                {/* Session Code - Only show in Edit mode */}
                {editingSession && (
                  <Form.Item
                    name="session_code"
                    label="Session Code"
                    rules={[
                      { required: true, message: "Session Code is required" },
                    ]}
                  >
                    <Input disabled className="font-semibold bg-gray-50" />
                  </Form.Item>
                )}

                {/* TRAINING CODE — ONLY SHOW IN CREATE MODE */}
                {!editingSession && (
                  <Form.Item
                    name="training_code"
                    label="Training Code"
                    rules={[
                      { required: true, message: "Training Code is required" },
                    ]}
                  >
                    <Select
                      placeholder="Select Training Code"
                      showSearch
                      options={trainingNeedCodes.map((c) => ({
                        label: c,
                        value: c,
                      }))}
                      filterOption={(input, option) =>
                        (option?.label ?? "")
                          .toLowerCase()
                          .includes(input.toLowerCase())
                      }
                    />
                  </Form.Item>
                )}

                {/* Session Date */}
                <Form.Item
                  name="session_date"
                  label="Session Date "
                  rules={[
                    { required: true, message: "Session Date is required" },
                  ]}
                >
                  <DatePicker className="w-full" />
                </Form.Item>

                {/* Duration */}
                <Form.Item
                  name="duration_hours"
                  label="Duration (hours) "
                  rules={[{ required: true, message: "Duration is required" }]}
                >
                  <Input
                    type="number"
                    min="0.5"
                    step="0.5"
                    placeholder="e.g. 2.5"
                  />
                </Form.Item>

                {/* Trainer */}
                <Form.Item
                  name="trainer_id"
                  label="Trainer "
                  rules={[
                    { required: true, message: "Please select a trainer" },
                  ]}
                >
                  <Select
                    showSearch
                    placeholder="Select Trainer"
                    loading={managers.length === 0}
                    options={managers.map((trainer) => ({
                      label: trainer.name,
                      value: trainer.id,
                    }))}
                    filterOption={(input, option) =>
                      (option?.label ?? "")
                        .toLowerCase()
                        .includes(input.toLowerCase())
                    }
                  />
                </Form.Item>

                {/* Venue */}
                <Form.Item name="venue" label="Venue">
                  <Input placeholder="Conference Room A, Online, etc." />
                </Form.Item>

                {/* Remarks */}
                <Form.Item name="remarks" label="Remarks">
                  <Input.TextArea
                    rows={2}
                    placeholder="Any additional notes..."
                  />
                </Form.Item>
              </div>
            </Form>
          </Modal>
        </motion.div>

        {/* Edit Approval Status Modal - EXACTLY LIKE YOUR OTHER PAGE */}
        <motion.div
          key={forceRender}
          initial={{ opacity: 0, scale: 0.95 }}
          animate={{ opacity: 1, scale: 1 }}
          exit={{ opacity: 0, scale: 0.95 }}
        >
          <Modal
            title={
              <div className="flex items-center gap-2 border-b pb-2 border-gray-500 dark:border-gray-600">
                <IdcardOutlined className="w-5 h-5 text-purple-600" />
                <span>Edit Approval Status</span>
              </div>
            }
            open={isApprovalModalOpen}
            onOk={handleApprovalFormSubmit}
            onCancel={() => {
              setIsApprovalModalOpen(false);
              approvalForm.resetFields();
              setForceRender((prev) => prev + 1);
            }}
            destroyOnClose
            width="95vw"
            className="!top-5 font-semibold"
            styles={{ body: { maxHeight: "80vh", overflowY: "auto" } }}
          >
            <Form form={approvalForm} layout="vertical" className="w-full px-4">
              <div className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 gap-4">
                <div className="col-span-full">
                  <h3 className="text-lg font-semibold mb-2">
                    Approval Details
                  </h3>
                </div>

                <Form.Item label="Session Code" className="col-span-full">
                  <Input
                    value={editingApprovalSession?.session_code}
                    disabled
                    className="font-mono font-semibold bg-gray-50"
                  />
                </Form.Item>

                <Form.Item
                  name="approvalStatus"
                  label="Approval Status "
                  rules={[
                    { required: true, message: "Approval Status is required" },
                  ]}
                >
                  <Select
                    options={[
                      { label: "Pending", value: "Pending" },
                      { label: "Approved", value: "Approved" },
                      { label: "Rejected", value: "Rejected" },
                    ]}
                    onChange={(value) => {
                      const currentComments =
                        approvalForm.getFieldValue("comments") ||
                        editingApprovalSession?.comments;
                      const currentRejection =
                        approvalForm.getFieldValue("rejectionReason") ||
                        editingApprovalSession?.rejection_reason;

                      approvalForm.setFieldsValue({
                        approvalStatus: value,
                        comments: currentComments,
                        rejectionReason:
                          value === "Rejected" ? currentRejection : undefined,
                        approvedBy:
                          value === "Approved" ? "Current User" : undefined,
                        approvedDate:
                          value === "Approved" ? dayjs() : undefined,
                      });
                      setForceRender((prev) => prev + 1);
                    }}
                  />
                </Form.Item>

                {approvalForm.getFieldValue("approvalStatus") !==
                  "Rejected" && (
                  <Form.Item name="comments" label="Comments">
                    <Input.TextArea rows={3} placeholder="Add any remarks..." />
                  </Form.Item>
                )}

                {approvalForm.getFieldValue("approvalStatus") ===
                  "Rejected" && (
                  <Form.Item
                    name="rejectionReason"
                    label="Rejection Reason "
                    rules={[
                      {
                        required: true,
                        message: "Rejection reason is required when rejecting",
                      },
                    ]}
                  >
                    <Input.TextArea
                      rows={3}
                      placeholder="Please provide a reason for rejection"
                    />
                  </Form.Item>
                )}

                {approvalForm.getFieldValue("approvalStatus") ===
                  "Approved" && (
                  <>
                    <Form.Item name="approvedBy" label="Approved By">
                      <Input disabled value="Current User" />
                    </Form.Item>
                    <Form.Item name="approvedDate" label="Approved Date">
                      <DatePicker disabled className="w-full" />
                    </Form.Item>
                  </>
                )}
              </div>
            </Form>
          </Modal>
        </motion.div>

        {/* Attendance Modal */}
        <motion.div
          key="attendanceModal"
          initial={{ opacity: 0, scale: 0.95 }}
          animate={{ opacity: 1, scale: 1 }}
          exit={{ opacity: 0, scale: 0.95 }}
        >
          <Modal
            title={
              <div className="flex items-center gap-2 border-b pb-2 border-gray-500 dark:border-gray-600">
                <TeamOutlined className="w-5 h-5 text-blue-600" />
                <span>Record Attendance</span>
              </div>
            }
            open={isAttendanceModalOpen}
            onOk={handleAttendanceSubmit}
            onCancel={() => {
              setIsAttendanceModalOpen(false);
              attendanceForm.resetFields();
            }}
            destroyOnClose
            width="90vw"
            className="!top-6 font-semibold"
            styles={{ body: { maxHeight: "75vh", overflowY: "auto" } }}
          >
            <Form
              form={attendanceForm}
              layout="vertical"
              className="w-full px-4"
            >
              <div className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 gap-4">
                {/* ✅ FIXED → Added name + required validation */}
                <Form.Item
                  name="session_code"
                  label="Session Code "
                  rules={[
                    { required: true, message: "Session Code is required" },
                  ]}
                >
                  <Input
                    disabled
                    value={attendanceSession?.session_code}
                    className="font-semibold bg-gray-50"
                  />
                </Form.Item>

                {/* ✅ FIXED → Added name + required validation */}
                <Form.Item
                  name="session_date"
                  label="Session Date "
                  rules={[
                    { required: true, message: "Session Date is required" },
                  ]}
                >
                  <DatePicker
                    disabled
                    value={
                      attendanceSession
                        ? dayjs(attendanceSession.session_date)
                        : null
                    }
                    className="w-full"
                  />
                </Form.Item>

                {/* Attended */}
                <Form.Item
                  name="attended"
                  label="Attended "
                  rules={[
                    {
                      required: true,
                      message: "Please choose attendance status",
                    },
                  ]}
                >
                  <Select
                    placeholder="Select status"
                    options={[
                      { label: "Present", value: "Present" },
                      { label: "Absent", value: "Absent" },
                      { label: "On Leave", value: "Leave" },
                    ]}
                  />
                </Form.Item>

                <Form.Item name="remarks" label="Remarks">
                  <Input.TextArea rows={3} placeholder="Optional remarks" />
                </Form.Item>
              </div>
            </Form>
          </Modal>
        </motion.div>

        <ConfirmDialog
          open={confirmDialog.open}
          title={confirmDialog.title}
          message={confirmDialog.message}
          onClose={() => setConfirmDialog({ ...confirmDialog, open: false })}
          onConfirm={confirmDialog.onConfirm}
        />
      </div>
    </>
  );
}
