"use client";

import React, { useCallback, useEffect, useState } from "react";
import Button from "antd/es/button";
import "antd/es/button/style";
import Form from "antd/es/form";
import "antd/es/form/style";
import Input from "antd/es/input";
import "antd/es/input/style";
import Modal from "antd/es/modal";
import "antd/es/modal/style";
import Select from "antd/es/select";
import "antd/es/select/style";
import Space from "antd/es/space";
import "antd/es/space/style";
import Tooltip from "antd/es/tooltip";
import "antd/es/tooltip/style";
import Tag from "antd/es/tag";
import "antd/es/tag/style";
import Breadcrumb from "antd/es/breadcrumb";
import "antd/es/breadcrumb/style";
import Card from "antd/es/card";
import "antd/es/card/style";
import DatePicker from "antd/es/date-picker";
import "antd/es/date-picker/style";
import Switch from "antd/es/switch";
import "antd/es/switch/style";
import Pagination from "antd/es/pagination";
import "antd/es/pagination/style";
import Dropdown from "antd/es/dropdown";
import "antd/es/dropdown/style";
import {
  PlusOutlined,
  EditOutlined,
  DeleteOutlined,
  HomeOutlined,
  TableOutlined,
  AppstoreOutlined,
  CheckCircleTwoTone,
  StopOutlined,
  WarningOutlined,
  EyeOutlined,
  SearchOutlined,
  CheckCircleOutlined,
  PauseCircleOutlined,
  FireOutlined,
  MinusCircleOutlined,
  CloseCircleOutlined,
  FileTextOutlined,
  IdcardOutlined,
} from "@ant-design/icons";
import type { ColumnsType } from "antd/es/table";
import DataTable from "../../components/common/DataTable";
import { useIsMobile } from "@/data/useIsMobile";
import { toast } from "sonner";
import { motion } from "framer-motion";
import { useRouter } from "next/navigation";
import {
  createTrainingNeed,
  deleteTrainingNeed,
  getAllTrainingNeeds,
  TrainingNeed,
  TrainingNeedData,
  updateTrainingNeed,
  updateApprovalStatus,
} from "@/app/api/trainingneedidentification/trainingneeds";
import { getAllEmployees, Employee } from "@/app/api/employee/employee";
import dayjs from "@/data/utils/dayjs";
import dynamic from "next/dynamic";
import GlassCard from "../../components/common/GlassCard";
import { highlightMatch } from "../../components/common/HighLightMatch";
import { FaFileExcel, FaFileExport, FaFilePdf, FaTrash } from "react-icons/fa";
import { UserCheck, Users, BookOpen } from "lucide-react";
import { LiaUserEditSolid } from "react-icons/lia";

const WidgetCard = dynamic(() => import("../../components/common/WidgetCard"), {
  ssr: false,
  loading: () => <p className="text-sm text-gray-400">Loading widget...</p>,
});

const ColumnVisibilityToggle = dynamic(
  () => import("../../components/common/ManageColumn"),
  {
    ssr: false,
    loading: () => <p className="text-sm text-gray-400">Loading columns...</p>,
  }
);

const ConfirmDialog = dynamic(
  () => import("../../components/common/ConfirmDialog"),
  {
    ssr: false,
    loading: () => <p className="text-sm text-gray-400">Preparing dialog...</p>,
  }
);

const { RangePicker } = DatePicker;
const priorityLevelOptions = ["Low", "Medium", "High"] as const;
const approvalStatusOptions = ["Pending", "Approved", "Rejected"] as const;

const { Option } = Select;

export default function TrainingNeedIdentificationPage() {
  const [form] = Form.useForm();
  const [approvalForm] = Form.useForm();
  const [trainingNeeds, setTrainingNeeds] = useState<TrainingNeedData[]>([]);
  const [employees, setEmployees] = useState<Employee[]>([]);
  const [isModalOpen, setIsModalOpen] = useState(false);
  const [isApprovalModalOpen, setIsApprovalModalOpen] = useState(false);
  const [editingTrainingNeed, setEditingTrainingNeed] =
    useState<TrainingNeedData | null>(null);
  const [editingApprovalTrainingNeed, setEditingApprovalTrainingNeed] =
    useState<TrainingNeedData | null>(null);
  const [searchTerm, setSearchTerm] = useState("");
  const [confirmDialog, setConfirmDialog] = useState({
    open: false,
    onConfirm: () => {},
    title: "",
    message: "",
  });
  const [loading, setLoading] = useState(false);
  const [dateRange, setDateRange] = useState<[dayjs.Dayjs, dayjs.Dayjs] | null>(
    null
  );
  const [viewingTrainingNeed, setViewingTrainingNeed] =
    useState<TrainingNeedData | null>(null);
  const [isGridView, setIsGridView] = useState(false);
  const [selectedRows, setSelectedRows] = useState<string[]>([]);
  const [advancedFilters, setAdvancedFilters] = useState({
    priorityLevel: undefined,
    approvalStatus: undefined,
  });
  const [currentPage, setCurrentPage] = useState(1);
  const [pageSize, setPageSize] = useState(10);
  const [forceRender, setForceRender] = useState(0);
  const isMobile = useIsMobile();
  const router = useRouter();

  useEffect(() => {
    const fetchData = async () => {
      try {
        setLoading(true);
        const [trainingData, employeeData] = await Promise.all([
          getAllTrainingNeeds(),
          getAllEmployees(),
        ]);

        // Newest training need first (top-la varum)
        const sortedTrainingNeeds = trainingData.sort(
          (a: TrainingNeedData, b: TrainingNeedData) => {
            const dateA = a.created_at || "";
            const dateB = b.created_at || "";
            return dateB.localeCompare(dateA); // Descending order
          }
        );

        setTrainingNeeds(sortedTrainingNeeds);
        setEmployees(employeeData);
      } catch (error) {
        console.error("Failed to fetch data:", error);
        toast.error("Failed to fetch training needs or employees");
      } finally {
        setLoading(false);
      }
    };
    fetchData();
  }, []);

  const openAddModal = () => {
    setEditingTrainingNeed(null);
    form.resetFields();
    setIsModalOpen(true);
  };

  const openEditModal = (trainingNeed: TrainingNeedData) => {
    setEditingTrainingNeed(trainingNeed);
    form.setFieldsValue({
      employee: trainingNeed.employee_code,
      trainingTopic: trainingNeed.training_topic,
      trainingReason: trainingNeed.training_reason,
      priorityLevel: trainingNeed.priority_level,
      proposedDate: trainingNeed.proposed_date
        ? dayjs(trainingNeed.proposed_date)
        : null,
      comments: trainingNeed.comments,
    });
    setIsModalOpen(true);
  };

  const openApprovalEditModal = (trainingNeed: TrainingNeedData) => {
    setEditingApprovalTrainingNeed(trainingNeed);
    approvalForm.resetFields();
    const initialFields = {
      approvalStatus: trainingNeed.approval_status || "Pending",
      comments: trainingNeed.comments || undefined,
      rejectionReason:
        trainingNeed.approval_status === "Rejected"
          ? trainingNeed.rejection_reason || undefined
          : undefined,
      approvedBy:
        trainingNeed.approval_status === "Approved"
          ? trainingNeed.approved_by_name || "Shankar.S"
          : undefined,
      approvedDate:
        trainingNeed.approval_status === "Approved" &&
        trainingNeed.approved_date
          ? dayjs(trainingNeed.approved_date)
          : undefined,
    };
    approvalForm.setFieldsValue(initialFields);
    setForceRender((prev) => prev + 1);
    setIsApprovalModalOpen(true);
  };

  const handleDelete = (trainingCode: string) => {
    setConfirmDialog({
      open: true,
      title: "Delete Training Need",
      message: "Are you sure you want to delete this training need?",
      onConfirm: async () => {
        try {
          await deleteTrainingNeed(trainingCode);
          const updated = await getAllTrainingNeeds();
          setTrainingNeeds(updated);
          toast.success("Training need deleted successfully");
        } catch {
          toast.error("Failed to delete training need");
        }
      },
    });
  };

  const handleBulkDelete = () => {
    if (selectedRows.length === 0) return;
    setConfirmDialog({
      open: true,
      title: "Delete Multiple Training Needs",
      message: `Are you sure you want to delete ${selectedRows.length} training needs?`,
      onConfirm: async () => {
        try {
          await Promise.all(
            selectedRows.map((code) => deleteTrainingNeed(code))
          );
          const updated = await getAllTrainingNeeds();
          setTrainingNeeds(updated);
          setSelectedRows([]);
          toast.success("Training needs deleted successfully");
        } catch {
          toast.error("Failed to delete selected training needs");
        }
      },
    });
  };

  const handleFormSubmit = async () => {
    try {
      const values = await form.validateFields(); // AntD validation (required fields)

      const payload: TrainingNeed = {
        employee_code: values.employee,
        training_topic: values.trainingTopic,
        training_reason: values.trainingReason,
        priority_level: values.priorityLevel,
        proposed_date: values.proposedDate
          ? dayjs(values.proposedDate).format("YYYY-MM-DD")
          : "",
        comments: values.comments || null,
      };

      if (editingTrainingNeed) {
        await updateTrainingNeed(editingTrainingNeed.training_code, payload);
        toast.success("Training need updated successfully");
      } else {
        await createTrainingNeed(payload);
        toast.success("Training need added successfully");
      }

      // Refresh list
      const updatedTrainingNeeds = await getAllTrainingNeeds();
      // Newest first sort pannu
      const sortedTrainingNeeds = updatedTrainingNeeds.sort(
        (a: TrainingNeedData, b: TrainingNeedData) => {
          const dateA = a.created_at || "";
          const dateB = b.created_at || "";
          return dateB.localeCompare(dateA);
        }
      );

      setTrainingNeeds(sortedTrainingNeeds);
      setTrainingNeeds(updatedTrainingNeeds);

      // Close modal
      setIsModalOpen(false);
      form.resetFields();
      setEditingTrainingNeed(null);
    } catch (error: any) {
      console.error("Form submission error:", error);

      // Frontend validation error (AntD - missing required fields)
      if (error.errorFields && error.errorFields.length > 0) {
        const missingFields = error.errorFields.map((field: any) => {
          const fieldName = field.name[0];
          const labelMap: Record<string, string> = {
            employee: "Employee",
            trainingTopic: "Training Topic",
            trainingReason: "Training Reason",
            priorityLevel: "Priority Level",
            proposedDate: "Proposed Date",
          };
          return labelMap[fieldName] || fieldName;
        });

        const message =
          missingFields.length > 1
            ? `${missingFields.join(", ")} are required`
            : `${missingFields[0]} is required`;

        toast.error(message);
        return; // Stop here — don't go to backend
      }

      // Backend API error
      const backendError = error?.response?.data?.message || error.message;
      if (backendError) {
        toast.error(backendError);
      } else {
        toast.error("Failed to save training need. Please try again.");
      }
    }
  };

  const handleApprovalFormSubmit = async () => {
    if (!editingApprovalTrainingNeed) return;
    try {
      const values = await approvalForm.validateFields();
      const payload = {
        approval_status: values.approvalStatus,
        rejection_reason: values.rejectionReason || null,
        comments: values.comments || null,
      };

      await updateApprovalStatus(
        editingApprovalTrainingNeed.training_code,
        payload
      );
      toast.success("Approval status updated successfully");
      const updatedTrainingNeeds = await getAllTrainingNeeds();
      setTrainingNeeds(updatedTrainingNeeds);
      setIsApprovalModalOpen(false);
      approvalForm.resetFields();
    } catch (err) {
      toast.error("Failed to update approval status");
    }
  };

  const renderTag = (value: string) => {
    const colorMap: Record<string, string> = {
      Low: "green",
      Medium: "orange",
      High: "red",
      Pending: "gold",
      Approved: "green",
      Rejected: "red",
    };

    const iconMap: Record<string, React.ReactNode> = {
      Low: <CheckCircleOutlined />,
      Medium: <PauseCircleOutlined />,
      High: <FireOutlined />,
      Pending: <PauseCircleOutlined />,
      Approved: <CheckCircleOutlined />,
      Rejected: <CloseCircleOutlined />,
    };

    return (
      <Tag icon={iconMap[value]} color={colorMap[value] || "default"}>
        {value}
      </Tag>
    );
  };

  const filteredTrainingNeeds = trainingNeeds.filter((trainingNeed) => {
    const matchesSearch =
      trainingNeed.training_topic
        .toLowerCase()
        .includes(searchTerm.toLowerCase()) ||
      trainingNeed.training_code
        .toLowerCase()
        .includes(searchTerm.toLowerCase()) ||
      trainingNeed.employee_name
        ?.toLowerCase()
        .includes(searchTerm.toLowerCase());
    const matchesDate =
      !dateRange ||
      (trainingNeed.created_at &&
        (!dateRange[0] ||
          dayjs(trainingNeed.created_at).isSame(dateRange[0]) ||
          dayjs(trainingNeed.created_at).isAfter(dateRange[0])) &&
        (!dateRange[1] ||
          dayjs(trainingNeed.created_at).isSame(dateRange[1]) ||
          dayjs(trainingNeed.created_at).isBefore(dateRange[1])));
    const matchesAdvanced =
      (!advancedFilters.priorityLevel ||
        trainingNeed.priority_level === advancedFilters.priorityLevel) &&
      (!advancedFilters.approvalStatus ||
        trainingNeed.approval_status === advancedFilters.approvalStatus);
    return matchesSearch && matchesDate && matchesAdvanced;
  });

  const paginatedTrainingNeeds = filteredTrainingNeeds.slice(
    (currentPage - 1) * pageSize,
    currentPage * pageSize
  );

  const columns: ColumnsType<TrainingNeedData> = [
    {
      title: (
        <input
          type="checkbox"
          onChange={(e) => {
            const checked = e.target.checked;
            setSelectedRows(
              checked ? filteredTrainingNeeds.map((t) => t.training_code) : []
            );
          }}
          className="w-4 h-4"
        />
      ),
      key: "selection",
      render: (_, record) => (
        <input
          type="checkbox"
          checked={selectedRows.includes(record.training_code)}
          onChange={(e) => {
            setSelectedRows((prev) =>
              e.target.checked
                ? [...prev, record.training_code]
                : prev.filter((id) => id !== record.training_code)
            );
          }}
          className="w-4 h-4"
        />
      ),
      width: 44,
    },
    {
      title: "Training Code",
      dataIndex: "training_code",
      key: "training_code",
      sorter: (a, b) => a.training_code.localeCompare(b.training_code),
    },
    {
      title: "Employee Name",
      dataIndex: "employee_name",
      key: "employee_name",
      sorter: (a, b) =>
        (a.employee_name || "").localeCompare(b.employee_name || ""),
      render: (text) => highlightMatch(text || "-", searchTerm),
    },
    {
      title: "Training Topic",
      dataIndex: "training_topic",
      key: "training_topic",
      sorter: (a, b) => a.training_topic.localeCompare(b.training_topic),
      render: (text) => (
        <Tooltip title={text}>
          <span className="block truncate max-w-[100px] sm:max-w-[150px] md:max-w-[180px]">
            {highlightMatch(text, searchTerm)}
          </span>
        </Tooltip>
      ),
    },
    {
      title: "Training Reason",
      dataIndex: "training_reason",
      key: "training_reason",
      render: (text) => (
        <Tooltip title={text}>
          <span className="block truncate max-w-[100px] sm:max-w-[150px] md:max-w-[180px]">
            {text || "-"}
          </span>
        </Tooltip>
      ),
    },
    {
      title: "Priority Level",
      dataIndex: "priority_level",
      key: "priority_level",
      filters: priorityLevelOptions.map((p) => ({ text: p, value: p })),
      onFilter: (value, record) => record.priority_level === value,
      render: renderTag,
      sorter: (a, b) => a.priority_level.localeCompare(b.priority_level),
    },
    {
      title: "Proposed Date",
      dataIndex: "proposed_date",
      key: "proposed_date",
      sorter: (a, b) =>
        dayjs(a.proposed_date).unix() - dayjs(b.proposed_date).unix(),
      render: (value: string) =>
        value ? dayjs(value).format("DD-MM-YYYY") : "-",
    },
    {
      title: "Approval Status",
      dataIndex: "approval_status",
      key: "approval_status",
      filters: approvalStatusOptions.map((s) => ({ text: s, value: s })),
      onFilter: (value, record) => record.approval_status === value,
      render: renderTag,
      sorter: (a, b) =>
        (a.approval_status || "").localeCompare(b.approval_status || ""),
    },
    {
      title: "Comments",
      dataIndex: "comments",
      key: "comments",
      render: (text) => (
        <Tooltip title={text}>
          <span className="block truncate max-w-[100px] sm:max-w-[150px] md:max-w-[180px]">
            {text || "-"}
          </span>
        </Tooltip>
      ),
    },
    {
      title: "Identified By",
      dataIndex: "identified_by_name",
      key: "identified_by_name",
      render: (text: string) => text || "-",
    },
    {
      title: "Approved By",
      dataIndex: "approved_by_name",
      key: "approved_by_name",
      render: (text: string) => text || "-",
    },
    {
      title: "Approved Date",
      dataIndex: "approved_date",
      key: "approved_date",
      sorter: (a, b) =>
        dayjs(a.approved_date).unix() - dayjs(b.approved_date).unix(),
      render: (value: string) =>
        value ? dayjs(value).format("DD-MM-YYYY") : "-",
    },
    {
      title: "Rejection Reason",
      dataIndex: "rejection_reason",
      key: "rejection_reason",
      render: (text) => (
        <Tooltip title={text}>
          <span className="block truncate max-w-[100px] sm:max-w-[150px] md:max-w-[180px]">
            {text || "-"}
          </span>
        </Tooltip>
      ),
    },
    {
      title: "Created At",
      dataIndex: "created_at",
      key: "created_at",
      sorter: (a, b) => dayjs(a.created_at).unix() - dayjs(b.created_at).unix(),
      render: (value: string) =>
        value
          ? `${dayjs(value).format("DD-MM-YYYY hh:mm A")} (${dayjs(
              value
            ).fromNow()})`
          : "-",
    },
    {
      title: "Actions",
      key: "actions",
      fixed: "right",
      render: (_, record) => (
        <Space size="small">
          {/* <Tooltip title="View">
            <Button
              icon={<EyeOutlined />}
              onClick={() =>
                router.push(
                  `/trainingneeds/trainingdetail?code=${record.training_code}&topic=${encodeURIComponent(record.training_topic)}`
                )
              }
              size="small"
              className="text-blue-600"
            />
          </Tooltip> */}
          <Tooltip title="Edit">
            <Button
              icon={<EditOutlined />}
              onClick={() => openEditModal(record)}
              size="small"
              type="primary"
              className="bg-blue-600 hover:bg-blue-700"
            />
          </Tooltip>
          <Tooltip title="Edit Approval">
            <Button
              icon={<LiaUserEditSolid />}
              onClick={() => openApprovalEditModal(record)}
              size="small"
              className="text-purple-600"
            />
          </Tooltip>
          <Tooltip title="Delete">
            <Button
              icon={<DeleteOutlined />}
              onClick={() => handleDelete(record.training_code)}
              danger
              size="small"
              className="text-red-600"
            />
          </Tooltip>
        </Space>
      ),
    },
  ];

  const allColumns = [
    { key: "selection", label: "Select" },
    { key: "training_code", label: "Training Code" },
    { key: "employee_name", label: "Employee Name" },
    { key: "training_topic", label: "Training Topic" },
    { key: "training_reason", label: "Training Reason" },
    { key: "priority_level", label: "Priority Level" },
    { key: "proposed_date", label: "Proposed Date" },
    { key: "approval_status", label: "Approval Status" },
    { key: "comments", label: "Comments" },
    { key: "identified_by_name", label: "Identified By" },
    { key: "approved_by_name", label: "Approved By" },
    { key: "approved_date", label: "Approved Date" },
    { key: "rejection_reason", label: "Rejection Reason" },
    { key: "created_at", label: "Created At" },
    { key: "actions", label: "Actions" },
  ];

  const columnKeys = allColumns.map((c) => c.key);
  const labelsMap = Object.fromEntries(allColumns.map((c) => [c.key, c.label]));
  const [visibleColumnKeys, setVisibleColumnKeys] =
    useState<string[]>(columnKeys);

  const handleVisibleColumnChange = useCallback((cols: string[]) => {
    setVisibleColumnKeys(cols);
  }, []);

  const filteredColumns = columns.filter(
    (col) => typeof col.key === "string" && visibleColumnKeys.includes(col.key)
  );

  const TrainingNeedCard = ({
    trainingNeed,
  }: {
    trainingNeed: TrainingNeedData;
  }) => (
    <GlassCard gradient blurStrength="lg" hoverLift color="success">
      <div className="border rounded-xl p-4 shadow-sm hover:shadow-md bg-white dark:bg-gray-900 transition-all w-full space-y-2">
        <div className="flex flex-col sm:flex-row justify-between items-start sm:items-center mb-2">
          <h3 className="text-lg font-semibold text-gray-800 dark:text-white">
            {trainingNeed.training_topic || "N/A"}
          </h3>
          <Tag color="blue">{trainingNeed.employee_name || "N/A"}</Tag>
        </div>
        <p className="text-sm text-gray-600 dark:text-gray-300">
          <strong>Training Code:</strong> {trainingNeed.training_code || "N/A"}
        </p>
        <p className="text-sm text-gray-600 dark:text-gray-300">
          <strong>Employee Name:</strong> {trainingNeed.employee_name || "N/A"}
        </p>
        <p className="text-sm text-gray-600 dark:text-gray-300">
          <strong>Priority:</strong> {renderTag(trainingNeed.priority_level)}
        </p>
        <p className="text-sm text-gray-600 dark:text-gray-300">
          <strong>Approval Status:</strong>{" "}
          {renderTag(trainingNeed.approval_status || "Pending")}
        </p>
        <p className="text-sm text-gray-600 dark:text-gray-300">
          <strong>Training Reason:</strong>{" "}
          {trainingNeed.training_reason || "N/A"}
        </p>
        <p className="text-sm text-gray-600 dark:text-gray-300">
          <strong>Proposed Date:</strong>{" "}
          {trainingNeed.proposed_date
            ? dayjs(trainingNeed.proposed_date).format("YYYY-MM-DD")
            : "N/A"}
        </p>
        <p className="text-sm text-gray-600 dark:text-gray-300">
          <strong>Comments:</strong> {trainingNeed.comments || "N/A"}
        </p>
        <p className="text-sm text-gray-600 dark:text-gray-300">
          <strong>Identified By:</strong>{" "}
          {trainingNeed.identified_by_name || "N/A"}
        </p>
        <p className="text-sm text-gray-600 dark:text-gray-300">
          <strong>Approved By:</strong> {trainingNeed.approved_by_name || "N/A"}
        </p>
        <p className="text-sm text-gray-600 dark:text-gray-300">
          <strong>Approved Date:</strong>{" "}
          {trainingNeed.approved_date
            ? dayjs(trainingNeed.approved_date).format("YYYY-MM-DD")
            : "N/A"}
        </p>
        <p className="text-sm text-gray-600 dark:text-gray-300">
          <strong>Rejection Reason:</strong>{" "}
          {trainingNeed.rejection_reason || "N/A"}
        </p>
        <p className="text-xs text-gray-500 dark:text-gray-400 mt-2">
          <strong>Created:</strong>{" "}
          {trainingNeed.created_at
            ? dayjs(trainingNeed.created_at).format("YYYY-MM-DD")
            : "N/A"}
        </p>
        <div className="flex gap-2 pt-2 border-t dark:border-gray-700">
          <Tooltip title="View">
            <Button
              icon={<EyeOutlined />}
              onClick={() => setViewingTrainingNeed(trainingNeed)}
              size="small"
              className="text-blue-600"
            />
          </Tooltip>
          <Tooltip title="Edit">
            <Button
              icon={<EditOutlined />}
              onClick={() => openEditModal(trainingNeed)}
              size="small"
              type="primary"
              className="bg-blue-600 hover:bg-blue-700"
            />
          </Tooltip>
          <Tooltip title="Delete">
            <Button
              icon={<DeleteOutlined />}
              onClick={() => handleDelete(trainingNeed.training_code)}
              size="small"
              danger
              className="text-red-600"
            />
          </Tooltip>
          <Tooltip title="Edit Approval">
            <Button
              icon={<LiaUserEditSolid />}
              onClick={() => openApprovalEditModal(trainingNeed)}
              size="small"
              className="text-purple-600"
            />
          </Tooltip>
        </div>
      </div>
    </GlassCard>
  );

  const total = trainingNeeds.length;
  const approved = trainingNeeds.filter(
    (t) => t.approval_status === "Approved"
  ).length;
  const pending = trainingNeeds.filter(
    (t) => t.approval_status === "Pending"
  ).length;
  const highPriority = trainingNeeds.filter(
    (t) => t.priority_level === "High"
  ).length;

  const downloadFile = (data: Blob, fileName: string) => {
    const url = window.URL.createObjectURL(data);
    const link = document.createElement("a");
    link.href = url;
    link.setAttribute("download", fileName);
    document.body.appendChild(link);
    link.click();
    link.remove();
  };

  const handleExport = async (type: "pdf" | "excel") => {
    try {
      toast.info(`Export ${type} functionality not implemented yet.`);
    } catch (err) {
      console.error("Export failed:", err);
      toast.error("Export failed");
    }
  };

  return (
    <>
      <div className="px-2 sm:px-2 md:px-2 lg:px-2 md:py-2 lg:py-2 py-2 sm:py-2">
        <h2 className="flex items-center gap-2 text-xl sm:text-2xl md:text-3xl font-bold text-gray-800 dark:text-white mb-2">
          <BookOpen className="w-7 h-7 text-green-600 dark:text-green-400 flex-shrink-0" />
          <span>Training Need Identification</span>
          <span className="text-gray-500 text-base sm:text-lg font-medium">
            ({trainingNeeds.length})
          </span>
        </h2>
        <p className="text-gray-600 dark:text-gray-300 text-sm sm:text-base mb-4">
          Manage training needs identification for employees in the QMS system.
        </p>
        <Breadcrumb
          className="mb-4 sm:mb-6 mt-2 text-sm text-gray-500 dark:text-gray-400"
          separator=">"
          items={[
            {
              title: (
                <Tooltip title="Dashboard">
                  <HomeOutlined />
                </Tooltip>
              ),
              href: "/hrm",
            },
            {
              title: "Training Management",
            },
            {
              title: "Training Needs",
            },
          ]}
        />

        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4 mt-3 mb-6">
          <WidgetCard
            title="Total Training Needs"
            value={total}
            icon={<AppstoreOutlined />}
            color="green"
            percentage={12}
            description="All identified training needs"
          />
          <WidgetCard
            title="Approved"
            value={approved}
            icon={<CheckCircleTwoTone twoToneColor="#52c41a" />}
            color="blue"
            percentage={8}
            description="Approved training needs"
          />
          <WidgetCard
            title="Pending"
            value={pending}
            icon={<PauseCircleOutlined />}
            color="gold"
            percentage={-3}
            description="Pending approval training needs"
          />
          <WidgetCard
            title="High Priority"
            value={highPriority}
            icon={<WarningOutlined />}
            color="orange"
            percentage={5}
            description="High-priority training needs"
          />
        </div>

        <div className="mt-4">
          <Card title="Training Needs List" className="p-0 mb-6">
            <div className="p-4 flex flex-col gap-4">
              <div className="flex flex-col sm:flex-row justify-between items-stretch sm:items-center gap-4">
                <Input
                  prefix={<SearchOutlined />}
                  allowClear
                  placeholder="Search by training topic, code, or employee name"
                  value={searchTerm}
                  onChange={(e) => setSearchTerm(e.target.value)}
                  className="w-full sm:w-64 border-gray-300 dark:border-gray-600 rounded-md shadow-sm"
                />
                <Dropdown
                  className="w-full sm:w-auto border-gray-300 dark:border-gray-600"
                  menu={{
                    items: [
                      {
                        key: "pdf",
                        label: (
                          <span className="flex items-center gap-2">
                            <FaFilePdf className="text-red-500" /> Export PDF
                          </span>
                        ),
                      },
                      {
                        key: "excel",
                        label: (
                          <span className="flex items-center gap-2">
                            <FaFileExcel className="text-green-500" /> Export
                            Excel
                          </span>
                        ),
                      },
                    ],
                    onClick: ({ key }) => handleExport(key as "pdf" | "excel"),
                  }}
                >
                  <Button type="default" icon={<FaFileExport />}>
                    Export
                  </Button>
                </Dropdown>
                <Button
                  type="primary"
                  icon={<PlusOutlined />}
                  onClick={openAddModal}
                  className="bg-blue-600 hover:bg-blue-700 w-full sm:w-auto"
                >
                  Add Training Need
                </Button>
              </div>
              <div className="flex flex-col gap-4 lg:flex-row lg:justify-between lg:items-center">
                <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-4 w-full lg:w-auto">
                  <Select
                    placeholder="Priority Level"
                    className="w-full max-w-[160px]"
                    onChange={(value) =>
                      setAdvancedFilters((prev) => ({
                        ...prev,
                        priorityLevel: value,
                      }))
                    }
                    allowClear
                    options={priorityLevelOptions.map((v) => ({
                      label: v,
                      value: v,
                    }))}
                  />
                  <Select
                    placeholder="Approval Status"
                    className="w-full max-w-[160px]"
                    onChange={(value) =>
                      setAdvancedFilters((prev) => ({
                        ...prev,
                        approvalStatus: value,
                      }))
                    }
                    allowClear
                    options={approvalStatusOptions.map((v) => ({
                      label: v,
                      value: v,
                    }))}
                  />
                </div>
                <div className="flex flex-col sm:flex-row flex-wrap gap-4 justify-start lg:justify-end w-full lg:w-auto">
                  <ColumnVisibilityToggle
                    columns={columnKeys}
                    labels={labelsMap}
                    storageKey="training_need_table_visibility"
                    onChange={handleVisibleColumnChange}
                  />
                  <RangePicker
                    value={dateRange}
                    onChange={(d) => setDateRange(d as any)}
                    className="w-full sm:w-auto border-gray-300 dark:border-gray-600"
                  />
                  <Button
                    onClick={handleBulkDelete}
                    disabled={selectedRows.length === 0}
                    danger
                    icon={<FaTrash />}
                    className="w-full sm:w-auto"
                  >
                    Bulk Delete
                  </Button>
                  <div className="hidden sm:block">
                    <Switch
                      checkedChildren={<AppstoreOutlined />}
                      unCheckedChildren={<TableOutlined />}
                      checked={isGridView}
                      onChange={setIsGridView}
                      className="w-full sm:w-auto"
                    />
                  </div>
                </div>
              </div>

              {(advancedFilters.priorityLevel ||
                advancedFilters.approvalStatus) && (
                <div className="flex flex-wrap gap-2 mt-2">
                  {advancedFilters.priorityLevel && (
                    <Tag
                      closable
                      onClose={() =>
                        setAdvancedFilters((prev) => ({
                          ...prev,
                          priorityLevel: undefined,
                        }))
                      }
                    >
                      Priority Level: {advancedFilters.priorityLevel}
                    </Tag>
                  )}
                  {advancedFilters.approvalStatus && (
                    <Tag
                      closable
                      onClose={() =>
                        setAdvancedFilters((prev) => ({
                          ...prev,
                          approvalStatus: undefined,
                        }))
                      }
                    >
                      Approval Status: {advancedFilters.approvalStatus}
                    </Tag>
                  )}
                </div>
              )}
            </div>

            {isGridView || isMobile ? (
              <div className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 lg:grid-cols-3 gap-4 p-4">
                {paginatedTrainingNeeds.map((trainingNeed) => (
                  <TrainingNeedCard
                    key={trainingNeed.training_code}
                    trainingNeed={trainingNeed}
                  />
                ))}
              </div>
            ) : (
              <div className="overflow-auto">
                <DataTable<TrainingNeedData>
                  columns={filteredColumns}
                  dataSource={paginatedTrainingNeeds}
                  rowKey={(record) => record.training_code}
                  scrollX="max-content"
                  className="mt-2"
                  pagination={false}
                  loading={loading}
                />
              </div>
            )}
            <div className="flex flex-col sm:flex-row justify-between items-center px-4 py-2 text-sm text-gray-600 dark:text-gray-300">
              <div className="mb-2 sm:mb-0">
                Showing <strong>{(currentPage - 1) * pageSize + 1}</strong> to{" "}
                <strong>
                  {Math.min(
                    currentPage * pageSize,
                    filteredTrainingNeeds.length
                  )}
                </strong>{" "}
                of <strong>{filteredTrainingNeeds.length}</strong> training
                needs
              </div>
              <Pagination
                current={currentPage}
                pageSize={pageSize}
                total={filteredTrainingNeeds.length}
                onChange={(page, size) => {
                  setCurrentPage(page);
                  setPageSize(size);
                }}
                showSizeChanger
                pageSizeOptions={["10", "30", "50", "100"]}
                size="small"
              />
            </div>
          </Card>
        </div>

        <motion.div
          initial={{ opacity: 0, scale: 0.95 }}
          animate={{ opacity: 1, scale: 1 }}
          exit={{ opacity: 0, scale: 0.95 }}
        >
          <Modal
            title={
              <div className="flex items-center gap-2 border-b pb-2 border-gray-500 dark:border-gray-600">
                <div className="flex items-center gap-1">
                  <Users
                    className={`w-5 h-5 ${
                      !editingTrainingNeed ? "text-green-600" : "text-gray-400"
                    }`}
                  />
                  <UserCheck
                    className={`w-5 h-5 ${
                      editingTrainingNeed ? "text-blue-600" : "text-gray-400"
                    }`}
                  />
                </div>
                <span>
                  {editingTrainingNeed
                    ? "Edit Training Need"
                    : "Add Training Need"}
                </span>
              </div>
            }
            open={isModalOpen}
            onOk={handleFormSubmit}
            onCancel={() => {
              setIsModalOpen(false);
              form.resetFields();
            }}
            destroyOnHidden
            width="95vw"
            className="!top-5 font-semibold"
            styles={{
              body: { maxHeight: "80vh", overflowY: "auto" },
            }}
          >
            <Form form={form} layout="vertical" className="w-full px-4">
              <div className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-4">
                <div className="col-span-full">
                  <h3 className="text-lg font-semibold mb-2">
                    Training Need Details
                  </h3>
                </div>

                <Form.Item
                  name="employee"
                  label="Employee"
                  rules={[{ required: true, message: "Employee is required" }]}
                >
                  <Select
                    showSearch
                    placeholder="Select an employee"
                    optionFilterProp="children"
                    filterOption={(input, option) =>
                      (option?.label?.toString() ?? "")
                        .toLowerCase()
                        .includes(input.toLowerCase())
                    }
                    options={employees.map((employee) => ({
                      label: `${employee.first_name} ${
                        employee.last_name || ""
                      }`.trim(),
                      value: employee.employee_code,
                    }))}
                  />
                </Form.Item>

                <Form.Item
                  name="trainingTopic"
                  label="Training Topic"
                  rules={[
                    { required: true, message: "Training Topic is required" },
                  ]}
                >
                  <Input />
                </Form.Item>

                <Form.Item
                  name="trainingReason"
                  label="Training Reason"
                  rules={[
                    { required: true, message: "Training Reason is required" },
                  ]}
                >
                  <Input.TextArea rows={2} />
                </Form.Item>

                <Form.Item
                  name="priorityLevel"
                  label="Priority Level"
                  rules={[
                    { required: true, message: "Priority Level is required" },
                  ]}
                >
                  <Select
                    options={priorityLevelOptions.map((v) => ({
                      label: v,
                      value: v,
                    }))}
                  />
                </Form.Item>

                <Form.Item
                  name="proposedDate"
                  label="Proposed Date"
                  rules={[
                    { required: true, message: "Proposed Date is required" },
                  ]}
                >
                  <DatePicker format="YYYY-MM-DD" className="w-full" />
                </Form.Item>

                <Form.Item name="comments" label="Comments">
                  <Input.TextArea rows={2} />
                </Form.Item>
              </div>
            </Form>
          </Modal>
        </motion.div>

        <motion.div
          key={forceRender}
          initial={{ opacity: 0, scale: 0.95 }}
          animate={{ opacity: 1, scale: 1 }}
          exit={{ opacity: 0, scale: 0.95 }}
        >
          <Modal
            title={
              <div className="flex items-center gap-2 border-b pb-2 border-gray-500 dark:border-gray-600">
                <IdcardOutlined className="w-5 h-5 text-purple-600" />
                <span>Edit Approval Status</span>
              </div>
            }
            open={isApprovalModalOpen}
            onOk={handleApprovalFormSubmit}
            onCancel={() => {
              setIsApprovalModalOpen(false);
              approvalForm.resetFields();
              setForceRender((prev) => prev + 1);
            }}
            destroyOnHidden
            width="95vw"
            className="!top-5 font-semibold"
            styles={{
              body: { maxHeight: "80vh", overflowY: "auto" },
            }}
          >
            <Form form={approvalForm} layout="vertical" className="w-full px-4">
              <div className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 gap-4">
                <div className="col-span-full">
                  <h3 className="text-lg font-semibold mb-2">
                    Approval Details
                  </h3>
                </div>

                <Form.Item
                  name="approvalStatus"
                  label="Approval Status"
                  rules={[
                    { required: true, message: "Approval Status is required" },
                  ]}
                >
                  <Select
                    options={approvalStatusOptions.map((v) => ({
                      label: v,
                      value: v,
                    }))}
                    onChange={(value) => {
                      const currentComments =
                        approvalForm.getFieldValue("comments") ||
                        editingApprovalTrainingNeed?.comments ||
                        undefined;
                      const currentRejectionReason =
                        approvalForm.getFieldValue("rejectionReason") ||
                        editingApprovalTrainingNeed?.rejection_reason ||
                        undefined;
                      const fields = {
                        approvalStatus: value,
                        comments: currentComments,
                        rejectionReason:
                          value === "Rejected"
                            ? currentRejectionReason
                            : undefined,
                        approvedBy:
                          value === "Approved"
                            ? editingApprovalTrainingNeed?.approved_by_name ||
                              "Shankar.S"
                            : undefined,
                        approvedDate:
                          value === "Approved"
                            ? editingApprovalTrainingNeed?.approved_date
                              ? dayjs(editingApprovalTrainingNeed.approved_date)
                              : dayjs()
                            : undefined,
                      };
                      approvalForm.setFieldsValue(fields);
                      setForceRender((prev) => prev + 1);
                    }}
                  />
                </Form.Item>
                {/* Comments field - only show if status is not Rejected */}
                {approvalForm.getFieldValue("approvalStatus") !==
                  "Rejected" && (
                  <Form.Item name="comments" label="Comments">
                    <Input.TextArea rows={2} />
                  </Form.Item>
                )}
                {/* Rejection Reason - only show when Rejected */}
                {approvalForm.getFieldValue("approvalStatus") ===
                  "Rejected" && (
                  <Form.Item
                    name="rejectionReason"
                    label="Rejection Reason"
                    rules={[
                      {
                        required: true,
                        message: "Rejection Reason is required",
                      },
                    ]}
                  >
                    <Input.TextArea rows={2} />
                  </Form.Item>
                )}

                {approvalForm.getFieldValue("approvalStatus") ===
                  "Approved" && (
                  <>
                    <Form.Item
                      name="approvedBy"
                      label="Approved By"
                      rules={[
                        { required: true, message: "Approved By is required" },
                      ]}
                    >
                      <Input disabled />
                    </Form.Item>
                    <Form.Item
                      name="approvedDate"
                      label="Approved Date"
                      rules={[
                        {
                          required: true,
                          message: "Approved Date is required",
                        },
                      ]}
                    >
                      <DatePicker
                        format="YYYY-MM-DD"
                        className="w-full"
                        disabled
                      />
                    </Form.Item>
                  </>
                )}
              </div>
            </Form>
          </Modal>
        </motion.div>

        <Modal
          title="Training Need Details"
          open={!!viewingTrainingNeed}
          onCancel={() => setViewingTrainingNeed(null)}
          footer={null}
          className="w-full max-w-[90vw] sm:max-w-[500px] top-4"
        >
          {viewingTrainingNeed && (
            <div className="space-y-2 text-sm sm:text-base">
              <p>
                <strong>Training Code:</strong>{" "}
                {viewingTrainingNeed.training_code}
              </p>
              <p>
                <strong>Employee Name:</strong>{" "}
                <span className="text-green-600">
                  {viewingTrainingNeed.employee_name || "-"}
                </span>
              </p>
              <p>
                <strong>Training Topic:</strong>{" "}
                {viewingTrainingNeed.training_topic}
              </p>
              <p>
                <strong>Training Reason:</strong>{" "}
                {viewingTrainingNeed.training_reason || "-"}
              </p>
              <p>
                <strong>Priority Level:</strong>{" "}
                {renderTag(viewingTrainingNeed.priority_level)}
              </p>
              <p>
                <strong>Proposed Date:</strong>{" "}
                {viewingTrainingNeed.proposed_date
                  ? dayjs(viewingTrainingNeed.proposed_date).format(
                      "YYYY-MM-DD"
                    )
                  : "-"}
              </p>
              <p>
                <strong>Approval Status:</strong>{" "}
                {renderTag(viewingTrainingNeed.approval_status || "Pending")}
              </p>
              <p>
                <strong>Comments:</strong> {viewingTrainingNeed.comments || "-"}
              </p>
              <p>
                <strong>Identified By:</strong>{" "}
                {viewingTrainingNeed.identified_by_name || "-"}
              </p>
              <p>
                <strong>Approved By:</strong>{" "}
                {viewingTrainingNeed.approved_by_name || "-"}
              </p>
              <p>
                <strong>Approved Date:</strong>{" "}
                {viewingTrainingNeed.approved_date
                  ? dayjs(viewingTrainingNeed.approved_date).format(
                      "YYYY-MM-DD"
                    )
                  : "-"}
              </p>
              <p>
                <strong>Rejection Reason:</strong>{" "}
                {viewingTrainingNeed.rejection_reason || "-"}
              </p>
              <p>
                <strong>Created At:</strong>{" "}
                {viewingTrainingNeed.created_at
                  ? dayjs(viewingTrainingNeed.created_at).format("YYYY-MM-DD")
                  : "-"}
              </p>
            </div>
          )}
        </Modal>
        <ConfirmDialog
          open={confirmDialog.open}
          title={confirmDialog.title}
          message={confirmDialog.message}
          onClose={() => setConfirmDialog({ ...confirmDialog, open: false })}
          onConfirm={confirmDialog.onConfirm}
        />
      </div>
    </>
  );
}
