import React from "react";
import { motion, Variants } from "framer-motion";

interface GlassCardProps {
  children: React.ReactNode;
  className?: string;
  gradient?: boolean; // enables soft gradient overlay
  hoverLift?: boolean; // enables subtle lift on hover
  blurStrength?: "sm" | "md" | "lg"; // control blur level
  color?: "primary" | "success" | "danger" | "warning" | "none"; // preset theme colors
  ariaLabel?: string; // accessibility label
}

export default function GlassCard({
  children,
  className = "",
  gradient = false,
  hoverLift = true,
  blurStrength = "md",
  color = "none",
  ariaLabel,
}: GlassCardProps) {
  const blurMap = {
    sm: "backdrop-blur-sm",
    md: "backdrop-blur-md",
    lg: "backdrop-blur-lg",
  };

  const colorMap = {
    primary: "shadow-blue-400/40 hover:shadow-blue-500/60 border-blue-300/20",
    success: "shadow-green-400/40 hover:shadow-green-500/60 border-green-300/20",
    danger: "shadow-red-400/40 hover:shadow-red-500/60 border-red-300/20",
    warning: "shadow-yellow-400/40 hover:shadow-yellow-500/60 border-yellow-300/20",
    none: "",
  };

  const baseGradient = gradient
    ? "bg-gradient-to-br from-white/70 via-white/60 to-white/50 dark:from-gray-800/70 dark:via-gray-800/60 dark:to-gray-800/50"
    : "bg-white/70 dark:bg-gray-800/70";

  const cardVariants: Variants = {
    rest: { scale: 1, y: 0 },
    hover: hoverLift ? { scale: 1.03, y: -2 } : {},
    tap: { scale: 0.98 },
  };

  return (
    <motion.div
      role="region"
      aria-label={ariaLabel}
      tabIndex={0}
      variants={cardVariants}
      initial="rest"
      whileHover="hover"
      whileTap="tap"
      transition={{ type: "spring", stiffness: 250, damping: 15 }}
      className={`
        ${blurMap[blurStrength]}
        ${baseGradient}
        ${colorMap[color]}
        rounded-2xl p-5
        shadow-md hover:shadow-xl
        border border-white/20 dark:border-gray-700/20
        transition-all duration-300 ease-in-out
        focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-green-400 dark:focus:ring-green-500
        ${className}
      `}
    >
      {children}
    </motion.div>
  );
}
