// lib/api/supplier.ts
import axios from "axios";
import { API_BASE_PATH } from "@/config/apiConfig";
import { toast } from "sonner";

const API_BASE_URL = API_BASE_PATH;

// Interface for Supplier data
export interface Supplier {
  supplier_id?: string;
  company_name: string;
  contact_person?: string | null;
  Suplier_type: string;
  email?: string | null;
  phone_number?: string | null;
  country?: string | null;
  address?: string | null;
  company_profile?: string | null;
  ownership_details?: string | null;
  gst_tax_id?: string | null;
  risk_classification: string;
  status: string;
  approved_status: "Not_Choose" | "Approved" | "Disapproved";
  fax?: string | null;
  license_numbers?: string | null;
  critical_information?: string | null;
  number_of_employees?: number | null;
  nature_of_business?: string | null;
  bank_name?: string | null;
  bank_branch?: string | null;
  account_number?: string | null;
  ifsc_code?: string | null;
  swift_code?: string | null;
  created_at?: string;
  created_by_id?: string;
  created_by_name?: string;
  updated_by_id?: string;
  updated_by_name?: string;
}

// Get auth token from localStorage (your original)
const getAuthHeaders = () => {
  const token = localStorage.getItem("authToken") || "";
  return {
    headers: {
      Authorization: token,
      "Content-Type": "application/x-www-form-urlencoded",
      Accept: "application/json",
    },
  };
};

// Helper to build supplier form data (your original – untouched)
const buildSupplierForm = (data: Supplier): URLSearchParams => {
  return new URLSearchParams({
    companyName: data.company_name,
    companyProfile: data.company_profile || "",
    Suplier_type: data.Suplier_type,
    ownershipDetails: data.ownership_details || "",
    address: data.address || "",
    riskClassification: data.risk_classification,
    contactPerson: data.contact_person || "",
    country: data.country || "",
    Suplier_status: data.status,
    phone_number: data.phone_number || "",
    email: data.email || "",
    gstTaxId: data.gst_tax_id || "",
    fax: data.fax || "",
    license_numbers: data.license_numbers || "",
    critical_information: data.critical_information || "",
    number_of_employees: data.number_of_employees?.toString() || "",
    nature_of_business: data.nature_of_business || "",
    bank_name: data.bank_name || "",
    bank_branch: data.bank_branch || "",
    account_number: data.account_number || "",
    ifsc_code: data.ifsc_code || "",
    swift_code: data.swift_code || "",
    approved_status: data.approved_status,
  });
};

// Show exact backend error only
const showBackendError = (error: any) => {
  const msg =
    error?.response?.data?.detail ||
    error?.response?.data?.message ||
    error?.message ||
    "Something went wrong";
  toast.error(msg);
};

// CREATE SUPPLIER
export const createSupplier = async (data: Supplier) => {
  try {
    const response = await axios.post(
      `${API_BASE_URL}/create-supplier`,
      buildSupplierForm(data),
      getAuthHeaders()
    );
    return response.data;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// GET ALL SUPPLIERS
export const getAllSuppliers = async (): Promise<Supplier[]> => {
  try {
    const response = await axios.get(
      `${API_BASE_URL}/view-all-suppliers`,
      getAuthHeaders()
    );
    return response.data.Suppliers;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// UPDATE SUPPLIER
export const updateSupplier = async (supplierId: string, data: Supplier) => {
  try {
    const response = await axios.put(
      `${API_BASE_URL}/update-supplier/${supplierId}`,
      buildSupplierForm(data),
      getAuthHeaders()
    );
    return response.data;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// DELETE SUPPLIER
export const deleteSupplier = async (supplierId: string) => {
  try {
    const response = await axios.delete(
      `${API_BASE_URL}/delete-supplier/${supplierId}`,
      getAuthHeaders()
    );
    return response.data;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// GET ALL DISQUALIFIED SUPPLIERS
export const getAlldisqualifiedSuppliers = async (): Promise<Supplier[]> => {
  try {
    const response = await axios.get(
      `${API_BASE_URL}/view-all-disqualified-suppliers`,
      getAuthHeaders()
    );
    return response.data.Suppliers;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};
