// app/api/export-suppliers.ts
import axios from "axios";
import type { AxiosRequestConfig, ResponseType } from "axios";
import { API_BASE_PATH } from "@/config/apiConfig";
import { toast } from "sonner";

const API_BASE_URL = API_BASE_PATH;

// Auth Headers (your original logic – untouched)
const getAuthHeaders = (
  responseType: ResponseType = "json"
): AxiosRequestConfig => {
  const token = localStorage.getItem("authToken") || "";
  return {
    headers: {
      Authorization: token,
      Accept: "application/json",
    },
    responseType,
  };
};

// Show exact backend error message only
const showBackendError = (error: any) => {
  const msg =
    error?.response?.data?.detail ||
    error?.response?.data?.message ||
    error?.message ||
    "Export failed";
  toast.error(msg);
};

// Export Suppliers - All (PDF)
export const exportSuppliersPDF = async () => {
  try {
    const response = await axios.get(
      `${API_BASE_URL}/export-suppliers-pdf`,
      getAuthHeaders("blob")
    );

    // Trigger download
    const url = window.URL.createObjectURL(new Blob([response.data]));
    const link = document.createElement("a");
    link.href = url;
    link.setAttribute("download", "All_Suppliers.pdf");
    document.body.appendChild(link);
    link.click();
    link.remove();

    return response.data;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// Export Suppliers - All (Excel)
export const exportSuppliersExcel = async () => {
  try {
    const response = await axios.get(
      `${API_BASE_URL}/export-suppliers-xl`,
      getAuthHeaders("blob")
    );

    const url = window.URL.createObjectURL(new Blob([response.data]));
    const link = document.createElement("a");
    link.href = url;
    link.setAttribute("download", "All_Suppliers.xlsx");
    document.body.appendChild(link);
    link.click();
    link.remove();

    return response.data;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// Export Suppliers - Selected (PDF)
export const exportSelectedSuppliersPDF = async (supplierIds: string[]) => {
  try {
    const ids = supplierIds.join(",");
    const response = await axios.get(
      `${API_BASE_URL}/export-suppliers-pdf/${ids}`,
      getAuthHeaders("blob")
    );

    const url = window.URL.createObjectURL(new Blob([response.data]));
    const link = document.createElement("a");
    link.href = url;
    link.setAttribute(
      "download",
      `Selected_Suppliers_${supplierIds.length}.pdf`
    );
    document.body.appendChild(link);
    link.click();
    link.remove();

    return response.data;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// Export Suppliers - Selected (Excel)
export const exportSelectedSuppliersExcel = async (supplierIds: string[]) => {
  try {
    const ids = supplierIds.join(",");
    const response = await axios.get(
      `${API_BASE_URL}/export-suppliers-xl/${ids}`,
      getAuthHeaders("blob")
    );

    const url = window.URL.createObjectURL(new Blob([response.data]));
    const link = document.createElement("a");
    link.href = url;
    link.setAttribute(
      "download",
      `Selected_Suppliers_${supplierIds.length}.xlsx`
    );
    document.body.appendChild(link);
    link.click();
    link.remove();

    return response.data;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};
