// lib/api/supplier-agreement.ts
import axios from "axios";
import { API_BASE_PATH } from "@/config/apiConfig";
import { toast } from "sonner";

const API_BASE_URL = API_BASE_PATH;

// Supplier Agreement Interface (exactly as you wrote – NOT touched)
export interface SupplierAgreement {
  supplier_id: string;
  aggreement_id?: string;
  aggreement_type: string;
  agreement_date: string;
  expiry_date: string;
  renewal_date: string;
  termination_date: string;
  signature_date: string;
  signature: string;
  notice_period: string;
  Status: string;

  // Files
  agreement_file?: File | string | null;
  renewal_file?: File | string | null;
  termination_file?: File | string | null;

  // System fields
  created_at?: string;
  updated_at?: string;
  created_by_id?: string;
  created_by_name?: string;
  updated_by_id?: string | null;
  updated_by_name?: string | null;
}

// Auth headers helper (your original)
const getAuthHeaders = () => {
  const token = localStorage.getItem("authToken") || "";
  return {
    headers: {
      Authorization: token,
      Accept: "application/json",
    },
  };
};

// Show exact backend error only
const showBackendError = (error: any) => {
  const msg =
    error?.response?.data?.detail ||
    error?.response?.data?.message ||
    error?.message ||
    "Something went wrong";
  toast.error(msg);
};

// CREATE SUPPLIER AGREEMENT
export const createSupplierAgreement = async (data: SupplierAgreement) => {
  try {
    const form = new FormData();

    form.append("supplier_id", data.supplier_id);
    form.append("aggreement_type", data.aggreement_type);
    form.append("agreement_date", data.agreement_date);
    form.append("expiry_date", data.expiry_date);
    form.append("renewal_date", data.renewal_date);
    form.append("termination_date", data.termination_date);
    form.append("signature_date", data.signature_date);
    form.append("signature", data.signature);
    form.append("notice_period", data.notice_period);
    form.append("Status", data.Status);

    if (data.agreement_file && data.agreement_file instanceof File) {
      form.append("agreement_file", data.agreement_file);
    }
    if (data.renewal_file && data.renewal_file instanceof File) {
      form.append("renewal_file", data.renewal_file);
    }
    if (data.termination_file && data.termination_file instanceof File) {
      form.append("termination_file", data.termination_file);
    }

    const response = await axios.post(
      `${API_BASE_URL}/create-supplier-agreement`,
      form,
      {
        headers: {
          ...getAuthHeaders().headers,
          "Content-Type": "multipart/form-data",
        },
      }
    );

    return response.data;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// GET ALL SUPPLIER AGREEMENTS
export const getAllSupplierAgreements = async (): Promise<
  SupplierAgreement[]
> => {
  try {
    const response = await axios.get(
      `${API_BASE_URL}/get-supplier-agreement`,
      getAuthHeaders()
    );
    return response.data?.Agreements ?? [];
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// GET SUPPLIER AGREEMENT BY SUPPLIER ID
export const getSupplierAgreementById = async (
  supplierId: string
): Promise<SupplierAgreement[]> => {
  try {
    const response = await axios.get(
      `${API_BASE_URL}/get-supplier-agreement/${supplierId}`,
      getAuthHeaders()
    );
    return response.data?.Agreements ?? [];
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// UPDATE SUPPLIER AGREEMENT
export const updateSupplierAgreement = async (
  supplierId: string,
  data: Partial<SupplierAgreement>
) => {
  try {
    const form = new FormData();

    Object.entries(data).forEach(([key, value]) => {
      if (value !== undefined && value !== null) {
        if (
          (key === "agreement_file" ||
            key === "renewal_file" ||
            key === "termination_file") &&
          value instanceof File
        ) {
          form.append(key, value);
        } else {
          form.append(key, String(value));
        }
      }
    });

    const response = await axios.put(
      `${API_BASE_URL}/update-supplier-agreement/${supplierId}`,
      form,
      {
        headers: {
          ...getAuthHeaders().headers,
          "Content-Type": "multipart/form-data",
        },
      }
    );

    return response.data;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// DELETE SUPPLIER AGREEMENT
export const deleteSupplierAgreement = async (supplierId: string) => {
  try {
    const response = await axios.delete(
      `${API_BASE_URL}/delete-supplier-agreement/${supplierId}`,
      getAuthHeaders()
    );
    return response.data;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// DOWNLOAD AGREEMENT DOCUMENT
export const downloadAgreementDocument = async (supplierId: string) => {
  try {
    const response = await axios.get(
      `${API_BASE_URL}/download-supplier-aggreemnt-document/${supplierId}`,
      { ...getAuthHeaders(), responseType: "blob" }
    );
    const url = window.URL.createObjectURL(new Blob([response.data]));
    const link = document.createElement("a");
    link.href = url;
    link.setAttribute("download", `SupplierAgreement_${supplierId}.pdf`);
    document.body.appendChild(link);
    link.click();
    link.remove();
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// DOWNLOAD RENEWAL DOCUMENT
export const downloadRenewalDocument = async (supplierId: string) => {
  try {
    const response = await axios.get(
      `${API_BASE_URL}/download-supplier-renewal-document/${supplierId}`,
      { ...getAuthHeaders(), responseType: "blob" }
    );
    const url = window.URL.createObjectURL(new Blob([response.data]));
    const link = document.createElement("a");
    link.href = url;
    link.setAttribute("download", `SupplierRenewal_${supplierId}.pdf`);
    document.body.appendChild(link);
    link.click();
    link.remove();
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// DOWNLOAD TERMINATION DOCUMENT
export const downloadTerminationDocument = async (supplierId: string) => {
  try {
    const response = await axios.get(
      `${API_BASE_URL}/download-supplier-termination-document/${supplierId}`,
      { ...getAuthHeaders(), responseType: "blob" }
    );
    const url = window.URL.createObjectURL(new Blob([response.data]));
    const link = document.createElement("a");
    link.href = url;
    link.setAttribute("download", `SupplierTermination_${supplierId}.pdf`);
    document.body.appendChild(link);
    link.click();
    link.remove();
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};
