// lib/api/product.ts
import axios from "axios";
import { API_BASE_PATH } from "@/config/apiConfig";
import { toast } from "sonner";

const API_BASE_URL = API_BASE_PATH;

// ==============================
// Product Interface (exactly as you wrote – NOT touched)
// ==============================
export interface Product {
  product_id?: string;
  product_name: string;
  supplier_name?: string;
  product_code: string;
  category: string;
  classification: string;
  description: string;
  risk_level: string;
  specfication: string;
  status: string;
  manufacturing_date: string;
  expiry_date: string;
  supplier_id: string;
  batch_no: string;
  reorder_level: string;
  current_stock: string;
  price: string;
  product_file?: File | string | null;
  created_at?: string;
  updated_at?: string;
  created_by_name?: string;
  created_by?: string;
  updated_by?: string;
  updated_by_name?: string;
}

// Token headers (your original)
const getAuthHeaders = () => {
  const token = localStorage.getItem("authToken") || "";
  return {
    headers: {
      Authorization: token,
      "Content-Type": "application/x-www-form-urlencoded",
      Accept: "application/json",
    },
  };
};

// Helper – show exact backend error message only
const showBackendError = (error: any) => {
  const msg =
    error?.response?.data?.detail ||
    error?.response?.data?.message ||
    error?.message ||
    "Something went wrong";
  toast.error(msg);
};

// CREATE PRODUCT
export const createProduct = async (data: Product) => {
  try {
    const form = new FormData();
    form.append("user_status", data.status);
    form.append("classification", data.classification);
    form.append("supplier_id", data.supplier_id);
    form.append("manufacturing_date", data.manufacturing_date);
    form.append("specfication", data.specfication);
    form.append("risk_level", data.risk_level);
    form.append("expiry_date", data.expiry_date);
    form.append("product_name", data.product_name);
    form.append("product_code", data.product_code);
    form.append("category", data.category);
    form.append("description", data.description);
    form.append("batch_no", data.batch_no);
    form.append("reorder_level", data.reorder_level);
    form.append("current_stock", data.current_stock);
    form.append("price", data.price);

    if (data.product_file && data.product_file instanceof File) {
      form.append("product_file", data.product_file);
    }

    const response = await axios.post(`${API_BASE_URL}/create-product`, form, {
      headers: {
        ...getAuthHeaders().headers,
        "Content-Type": "multipart/form-data",
      },
    });

    return response.data;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// GET ALL PRODUCTS
export const getAllProducts = async (): Promise<Product[]> => {
  try {
    const response = await axios.get(
      `${API_BASE_URL}/view-product-detail`,
      getAuthHeaders()
    );
    return response.data.Products;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// UPDATE PRODUCT
export const updateProduct = async (productId: string, data: Product) => {
  try {
    const form = new FormData();
    form.append("user_status", data.status);
    form.append("classification", data.classification);
    form.append("manufacturing_date", data.manufacturing_date);
    form.append("specfication", data.specfication);
    form.append("risk_level", data.risk_level);
    form.append("expiry_date", data.expiry_date);
    form.append("product_name", data.product_name);
    form.append("product_code", data.product_code);
    form.append("category", data.category);
    form.append("description", data.description);
    form.append("supplier_id", data.supplier_id);
    form.append("batch_no", data.batch_no);
    form.append("reorder_level", data.reorder_level);
    form.append("current_stock", data.current_stock);
    form.append("price", data.price);

    if (data.product_file && data.product_file instanceof File) {
      form.append("product_file", data.product_file);
    }

    const response = await axios.put(
      `${API_BASE_URL}/update-product-detail/${productId}`,
      form,
      {
        headers: {
          ...getAuthHeaders().headers,
          "Content-Type": "multipart/form-data",
        },
      }
    );

    return response.data;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// DELETE PRODUCT
export const deleteProduct = async (productId: string) => {
  try {
    const response = await axios.delete(
      `${API_BASE_URL}/delete-product/${productId}`,
      {
        headers: {
          Authorization: localStorage.getItem("authToken") || "",
          Accept: "application/json",
        },
      }
    );

    return response.data;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// GET PRODUCT BY ID
export const getProductById = async (productId: string): Promise<Product> => {
  try {
    const response = await axios.get(
      `${API_BASE_URL}/view-product-detail/${productId}/`,
      getAuthHeaders()
    );
    return response.data.Detail;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// DOWNLOAD PRODUCT DOCUMENT
export const downloadProductDocument = async (productId: string) => {
  try {
    const response = await axios.get(
      `${API_BASE_URL}/download-product-document/${productId}`,
      {
        ...getAuthHeaders(),
        responseType: "blob",
      }
    );

    const url = window.URL.createObjectURL(new Blob([response.data]));
    const link = document.createElement("a");
    link.href = url;
    link.setAttribute("download", `Product_${productId}.xlsx`);
    document.body.appendChild(link);
    link.click();
    link.remove();

    return true;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};
