// app/api/employee/EmployeeSkillMatrix.ts
import axios from "axios";
import { API_BASE_PATH } from "@/config/apiConfig";
import { toast } from "sonner"; // <-- Only this line added

// Interface for Employee Skill (matches backend response)
export interface EmployeeSkill {
  employee_id?: number;
  employee_code: string;
  employee_name?: string;
  designation?: string | null;
  department?: string | null;
  skill_code?: string;
  skill_name: string;
  skill_category: string;
  proficiency_level: string;
  last_assessed_date?: string | null;
  verified_by: number | null;
  verified_by_name?: string | null;
  remarks?: string | null;
  created_at?: string;
  updated_at?: string;
}

// Helper: Show exact backend error message in toast
const showApiError = (error: any) => {
  const msg =
    error?.response?.data?.detail ||
    error?.response?.data?.message ||
    error?.response?.data?.error ||
    error?.message ||
    "Something went wrong";
  toast.error(msg);
};

// Helper: Get Auth Headers
const getAuthHeaders = () => {
  const token = localStorage.getItem("authToken") || "";
  const authHeader = token.startsWith("Bearer ") ? token : `Bearer ${token}`;
  return {
    headers: {
      Authorization: authHeader,
      Accept: "application/json",
    },
  };
};

// Create Employee Skill
export const createEmployeeSkill = async (data: EmployeeSkill) => {
  if (
    !data.employee_code ||
    !data.skill_name ||
    !data.skill_category ||
    !data.proficiency_level
  ) {
    throw new Error(
      "Missing required fields: employee_code, skill_name, skill_category, proficiency_level"
    );
  }

  const params = new URLSearchParams();
  Object.entries(data).forEach(([key, value]) => {
    if (value !== undefined && value !== null) {
      params.append(key, String(value));
    }
  });

  try {
    const response = await axios.post(
      `${API_BASE_PATH}/create_employee_skill`,
      params,
      {
        ...getAuthHeaders(),
        headers: {
          ...getAuthHeaders().headers,
          "Content-Type": "application/x-www-form-urlencoded",
        },
      }
    );
    return response.data;
  } catch (error: any) {
    showApiError(error);
    throw error; // so caller can also handle if needed
  }
};

// Get All Employee Skills
export const getAllEmployeeSkills = async (): Promise<EmployeeSkill[]> => {
  try {
    const response = await axios.get(
      `${API_BASE_PATH}/get_all_employee_skills`,
      getAuthHeaders()
    );
    return Array.isArray(response.data?.data) ? response.data.data : [];
  } catch (error: any) {
    showApiError(error);
    return [];
  }
};

// Get Single Employee Skill by Skill Code
export const getEmployeeSkillByCode = async (
  skill_code: string
): Promise<EmployeeSkill> => {
  if (!skill_code) throw new Error("Skill code is required");

  try {
    const response = await axios.get(
      `${API_BASE_PATH}/get_employee_skill/${skill_code}`,
      getAuthHeaders()
    );

    if (response.data?.status === "success" && response.data.data) {
      return response.data.data;
    }
    throw new Error("No skill data returned from API");
  } catch (error: any) {
    showApiError(error);
    throw error;
  }
};

// Update Employee Skill
export const updateEmployeeSkill = async (
  skill_code: string,
  data: Partial<EmployeeSkill>
) => {
  if (!skill_code) throw new Error("Skill code is required to update skill");

  const params = new URLSearchParams();
  Object.entries(data).forEach(([key, value]) => {
    if (value !== undefined && value !== null) {
      params.append(key, String(value));
    }
  });

  try {
    const response = await axios.put(
      `${API_BASE_PATH}/update_employee_skill/${skill_code}`,
      params,
      {
        ...getAuthHeaders(),
        headers: {
          ...getAuthHeaders().headers,
          "Content-Type": "application/x-www-form-urlencoded",
        },
      }
    );
    return response.data;
  } catch (error: any) {
    showApiError(error);
    throw error;
  }
};

// Delete Employee Skill
export const deleteEmployeeSkill = async (skill_code: string) => {
  if (!skill_code) throw new Error("Skill code is required to delete skill");

  try {
    const response = await axios.delete(
      `${API_BASE_PATH}/delete_employee_skill/${skill_code}`,
      getAuthHeaders()
    );

    if (response.data?.status === "success") {
      return {
        status: "success",
        message:
          response.data.message || `Skill '${skill_code}' deleted successfully`,
      };
    }

    throw new Error("Failed to delete employee skill");
  } catch (error: any) {
    showApiError(error);
    throw error;
  }
};
